/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.calendar.search;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.apache.cocoon.components.source.impl.SitemapSource;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.cms.tag.Tag;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.DateUtils;
import org.ametys.core.util.IgnoreRootHandler;
import org.ametys.core.util.LambdaUtils;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.tag.TagAwareAmetysObject;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.type.DataContext;
import org.ametys.web.WebConstants;
import org.ametys.web.content.GetSiteAction;
import org.ametys.web.frontoffice.search.metamodel.ReturnableSaxer;
import org.ametys.web.frontoffice.search.metamodel.impl.AbstractContentBasedReturnable;
import org.ametys.web.frontoffice.search.metamodel.impl.ContentSaxer;
import org.ametys.web.frontoffice.search.requesttime.SearchComponentArguments;
import org.ametys.web.repository.content.WebContent;
import org.ametys.web.repository.site.Site;

/**
 * {@link ReturnableSaxer} for {@link CalendarContentReturnable}
 */
public class CalendarContentSaxer extends ContentSaxer
{
    /**
     * Constructor
     * @param contentReturnable The associated returnable on contents
     * @param view The view for SAXing contents
     * @param cTypesHelper the {@link ContentTypesHelper}
     * @param contentTypes the allowed content types
     */
    public CalendarContentSaxer(AbstractContentBasedReturnable contentReturnable, String view, ContentTypesHelper cTypesHelper, Collection<String> contentTypes)
    {
        super(contentReturnable, view, cTypesHelper, contentTypes);
    }

    @Override
    public void sax(ContentHandler contentHandler, AmetysObject hit, Logger logger, SearchComponentArguments args) throws SAXException
    {
        Content content = (Content) hit;
        
        View view = _cTypesHelper.getView(_view, content);
        
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", content.getId());
        attrs.addCDATAAttribute("name", content.getName());
        attrs.addCDATAAttribute("title", content.getTitle(null));
        attrs.addCDATAAttribute("createdAt", DateUtils.zonedDateTimeToString(content.getCreationDate()));
        attrs.addCDATAAttribute("creator", UserIdentity.userIdentityToString(content.getCreator()));
        attrs.addCDATAAttribute("lastModifiedAt", DateUtils.zonedDateTimeToString(content.getLastModified()));
        
        Optional.ofNullable(content.getLanguage()).ifPresent(lang -> attrs.addCDATAAttribute("language", lang));
        
        XMLUtils.startElement(contentHandler, "content", attrs);
        
        String[] contentTypes = content.getTypes();
        XMLUtils.startElement(contentHandler, "contentTypes");
        Arrays.asList(contentTypes).forEach(LambdaUtils.wrapConsumer(cType -> XMLUtils.createElement(contentHandler, "contentType", cType)));
        XMLUtils.endElement(contentHandler, "contentTypes");
        
        XMLUtils.startElement(contentHandler, "attributes");
        Locale defaultLocale = getDefaultLocale(args.request(), args);
        content.dataToSAX(contentHandler, view, DataContext.newInstance().withLocale(defaultLocale).withEmptyValues(false));
        XMLUtils.endElement(contentHandler, "attributes");
        
        saxHtmlView(contentHandler, content, args, logger);
        
        saxTags(content, contentHandler);
        
        XMLUtils.endElement(contentHandler, "content");
    }
    
    /**
     * Get the default locale
     * @param request the request
     * @param args the search arguments
     * @return the default locale
     */
    protected Locale getDefaultLocale(Request request, SearchComponentArguments args)
    {
        String lang = args.generatorParameters().getParameter("lang", request.getParameter("lang"));
        return StringUtils.isNotEmpty(lang) ? new Locale(lang) : Locale.getDefault();
    }
    
    /**
     * SAX content html view
     * @param contentHandler the content handler
     * @param content the content
     * @param args the search arguments
     * @param logger the logger
     * @throws SAXException if an error occurred while saxing
     */
    protected void saxHtmlView(ContentHandler contentHandler, Content content, SearchComponentArguments args, Logger logger) throws SAXException
    {
        
        Request request = args.request();
        String currentSiteName = (String) request.getAttribute(WebConstants.REQUEST_ATTR_SITE_NAME);
        Site currentSite = (Site) request.getAttribute(WebConstants.REQUEST_ATTR_SITE);
        String currentSkinName = (String) request.getAttribute(WebConstants.REQUEST_ATTR_SKIN_ID);
        
        try
        {
            // Content of other sites should be rendered with the current skin
            request.setAttribute(GetSiteAction.OVERRIDE_SITE_REQUEST_ATTR, currentSiteName);
            request.setAttribute(GetSiteAction.OVERRIDE_SKIN_REQUEST_ATTR, currentSkinName);
            
            XMLUtils.startElement(contentHandler, "view");
            
            if (_contentReturnable instanceof CalendarContentReturnable)
            {
                
                
                String uri = _contentReturnable.getContentHelper().getContentHtmlViewUrl(content, _view);
                
                SitemapSource src = null;
                
                try
                {
                    src = (SitemapSource) ((CalendarContentReturnable) _contentReturnable)._srcResolver.resolveURI(uri);
                    src.toSAX(new IgnoreRootHandler(contentHandler));
                }
                finally
                {
                    ((CalendarContentReturnable) _contentReturnable)._srcResolver.release(src);
                }
            }
            
            XMLUtils.endElement(contentHandler, "view");
        }
        catch (IOException e)
        {
            logger.error("Unable to sax HTML view for content '{}'", content.getId(), e);
        }
        finally
        {
            // Need to do this as contentFilterHelper redirects to a Cocoon URL that gets through 'GetSiteAction' which changes some request attributes
            request.removeAttribute(GetSiteAction.OVERRIDE_SITE_REQUEST_ATTR);
            request.removeAttribute(GetSiteAction.OVERRIDE_SKIN_REQUEST_ATTR);
            request.setAttribute(WebConstants.REQUEST_ATTR_SITE_NAME, currentSiteName);
            request.setAttribute(WebConstants.REQUEST_ATTR_SITE, currentSite);
            request.setAttribute("siteName", currentSiteName);
            request.setAttribute(WebConstants.REQUEST_ATTR_SKIN_ID, currentSkinName);
        }
    }
    
    /**
     * Generates SAX events for tags.
     * @param content the {@link WebContent}.
     * @param contentHandler the ContentHandler receving SAX events.
     * @throws SAXException if an error occurs during the SAX events generation.
     */
    protected void saxTags(Content content, ContentHandler contentHandler) throws SAXException
    {
        if (_contentReturnable instanceof CalendarContentReturnable)
        {
            XMLUtils.startElement(contentHandler, "tags");

            Set<String> tags = ((TagAwareAmetysObject) content).getTags();
            for (String tagName : tags)
            {
                Map<String, Object> contextParameters = new HashMap<>();
                if (content instanceof WebContent)
                {
                    contextParameters.put("siteName", ((WebContent) content).getSiteName());
                }
                
                Tag tag = ((CalendarContentReturnable) _contentReturnable).getTagProviderExtensionPoint().getTag(tagName, contextParameters);

                if (tag != null)
                {
                    AttributesImpl attrs = new AttributesImpl();
                    if (tag.getParentName() != null)
                    {
                        attrs.addCDATAAttribute("parent", tag.getParentName());
                    }

                    XMLUtils.startElement(contentHandler, tagName, attrs);
                    tag.getTitle().toSAX(contentHandler);
                    XMLUtils.endElement(contentHandler, tagName);
                }
            }

            XMLUtils.endElement(contentHandler, "tags");
        }
        
    }
}
