/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.calendar.search;

import java.time.ZonedDateTime;

import org.ametys.cms.search.query.AndQuery;
import org.ametys.cms.search.query.DateTimeQuery;
import org.ametys.cms.search.query.NotQuery;
import org.ametys.cms.search.query.OrQuery;
import org.ametys.cms.search.query.Query;
import org.ametys.cms.search.query.Query.Operator;
import org.ametys.plugins.calendar.events.EventsFilterHelper.DateTimeRange;
import org.ametys.web.frontoffice.search.requesttime.SearchComponent;
import org.ametys.web.frontoffice.search.requesttime.SearchComponentArguments;
import org.ametys.web.frontoffice.search.requesttime.impl.SearchComponentHelper;


/**
 * {@link SearchComponent} to add date range to query
 */
public class DateRangeCriterionSearchComponent implements SearchComponent
{
    /** The helper for search component */
    protected SearchComponentHelper _searchComponentHelper;
    
    @Override
    public int priority()
    {
        return SEARCH_PRIORITY - 7999; // after CriterionTreeSearchComponent
    }

    @Override
    public boolean supports(SearchComponentArguments args)
    {
        return CalendarSearchService.isFormSubmit(args) && CalendarSearchService.isActive(args);
    }

    @Override
    public void execute(SearchComponentArguments args) throws Exception
    {
        DateTimeRange dateRange = CalendarSearchService.getDateRange(args);
        if (dateRange != null)
        {
            ZonedDateTime fromDateTime = dateRange.fromDate();
            ZonedDateTime untilDateTime = dateRange.untilDate();
            
            Query startBeforeQuery = new DateTimeQuery(CalendarSearchService.START_DATE_ATTRIBUTE_NAME, Operator.LT, untilDateTime);
            Query endAfterExpr = new DateTimeQuery(CalendarSearchService.END_DATE_ATTRIBUTE_NAME, Operator.GE, fromDateTime);
            Query startAfterExpr = new DateTimeQuery(CalendarSearchService.START_DATE_ATTRIBUTE_NAME, Operator.GE, fromDateTime);
            Query noEndExpr = new NotQuery(new DateTimeQuery(CalendarSearchService.END_DATE_ATTRIBUTE_NAME));
            
            // start-date < untilDate AND (end-date >= fromDate OR not(end-date) AND start-date >= fromDate)
            args.searcher()
                .addFilterQuery(new AndQuery(startBeforeQuery, new OrQuery(endAfterExpr,  new AndQuery(noEndExpr, startAfterExpr))));
        }
        else
        {
            // filter on start-date exists
            args.searcher()
                .addFilterQuery(new DateTimeQuery(CalendarSearchService.START_DATE_ATTRIBUTE_NAME));
        }
    }
}
