/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.calendar.search;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.userpref.UserPreferencesManager;
import org.ametys.core.util.JSONUtils;
import org.ametys.web.frontoffice.search.requesttime.SearchComponent;
import org.ametys.web.frontoffice.search.requesttime.SearchComponentArguments;

/**
 * {@link SearchComponent} to save the selected user calendar's filters in the user preferences if the user is authenticated.
 */
public class SaveUserPrefsSearchComponent implements SearchComponent, Serviceable
{
    /** The JSON utils */
    protected JSONUtils _jsonUtils;
    /** The user preferences manager */
    protected UserPreferencesManager _userPrefsManager;
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _jsonUtils = (JSONUtils) manager.lookup(JSONUtils.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _userPrefsManager = (UserPreferencesManager) manager.lookup(UserPreferencesManager.ROLE);
    }

    @Override
    public int priority()
    {
        // Should save in user prefs before sax
        return MAX_PRIORITY + 500;
    }

    @Override
    public boolean supports(SearchComponentArguments args)
    {
        return CalendarSearchService.isFormSubmit(args) && CalendarSearchService.saveUserPrefsEnabled(args) && CalendarSearchService.isActive(args) && CalendarSearchService.hasTagCategories(args.serviceInstance()) && CalendarSearchService.saveFilterInUserPrefs(args.serviceInstance());
    }

    @Override
    public void execute(SearchComponentArguments args) throws Exception
    {
        // If the user is connected
        UserIdentity currentUser = _currentUserProvider.getUser();
        if (currentUser != null)
        {
            List<String> selectedTags = CalendarSearchService.getSelectedTags(args);
            
            String serviceId = args.serviceInstance().getId();

            Map<String, String> values = new HashMap<>();
            
            values.put("search.calendar.tags", _jsonUtils.convertObjectToJson(selectedTags));
            
            _userPrefsManager.setUserPreferences(currentUser, "calendar-search-" + serviceId, new HashMap<>(), values);
        }
    }
    
}
