/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.calendar.search;

import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.excalibur.source.SourceResolver;

import org.ametys.cms.tag.TagProviderExtensionPoint;
import org.ametys.web.contenttype.SkinContentViewHelper;
import org.ametys.web.frontoffice.search.metamodel.AdditionalParameterValueMap;
import org.ametys.web.frontoffice.search.metamodel.Returnable;
import org.ametys.web.frontoffice.search.metamodel.ReturnableSaxer;
import org.ametys.web.frontoffice.search.metamodel.impl.AbstractContentBasedReturnable;

/**
 * {@link Returnable} for contents with a start date
 */
public class CalendarContentReturnable extends AbstractContentBasedReturnable
{
    /** Avalon Role */
    public static final String ROLE = CalendarContentReturnable.class.getName();
    
    private static final String __SORT_DEFINITION_PREFIX_ID = "ContentWithDateReturnable$";
    
    /** The source resolver */
    protected SourceResolver _srcResolver;
    /** The tag provider extenstion point */
    private TagProviderExtensionPoint _tagProvideEP;
    private SkinContentViewHelper _skinContentViewHelper;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _srcResolver = (SourceResolver) smanager.lookup(SourceResolver.ROLE);
        _tagProvideEP = (TagProviderExtensionPoint) smanager.lookup(TagProviderExtensionPoint.ROLE);
        _skinContentViewHelper = (SkinContentViewHelper) smanager.lookup(SkinContentViewHelper.ROLE);
    }
    
    public String getId()
    {
        return CalendarContentReturnable.ROLE;
    }
    
    @Override
    public boolean isPrivate()
    {
        return true;
    }
    
    /**
     * Get the source resolver
     * @return the source resolver
     */
    public SourceResolver getSourceResolver()
    {
        return _srcResolver;
    }
    
    /**
     * Get the tag provider extension point
     * @return the tag provider extension point
     */
    public TagProviderExtensionPoint getTagProviderExtensionPoint()
    {
        return _tagProvideEP;
    }

    public ReturnableSaxer getSaxer(Collection<Returnable> allReturnables, AdditionalParameterValueMap additionalParameterValues)
    {
        return new CalendarContentSaxer(this, getViewForSax(additionalParameterValues), _contentTypesHelper, _skinContentViewHelper, getContentTypeIds(additionalParameterValues));
    }

    @Override
    protected String associatedContentSearchableRole()
    {
        return CalendarContentSearchable.ROLE;
    }
    
    /**
     * Get the model view for sax
     * @param additionalParameterValues The additional parameter values
     * @return the model view to use for saxing results
     */
    protected String getViewForSax(AdditionalParameterValueMap additionalParameterValues)
    {
        return additionalParameterValues.getValue(CalendarSearchService.SERVICE_PARAM_CONTENT_VIEW);
    }

    @Override
    protected Set<String> getContentTypeIds(AdditionalParameterValueMap additionalParameterValues)
    {
        Collection<String> contentTypeIds = additionalParameterValues.getValue(CalendarSearchService.SERVICE_PARAM_CONTENT_TYPES);
        return contentTypeIds != null
                ? new HashSet<>(contentTypeIds)
                : Set.of();
    }

    @Override
    protected String getSortDefinitionPrefix()
    {
        return __SORT_DEFINITION_PREFIX_ID;
    }

}
