/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.calendar.events;

import java.io.IOException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.repository.Content;
import org.ametys.core.right.RightManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.URIUtils;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.authentication.AccessDeniedException;
import org.ametys.runtime.authentication.AuthorizationRequiredException;

/**
 * Sax an event
 */
public class EventGenerator extends AbstractEventGenerator
{
    /** The ametys object resolver. */
    protected AmetysObjectResolver _ametysResolver;
    /** The right manager */
    protected RightManager _rightManager;
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _ametysResolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _currentUserProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
        _rightManager = (RightManager) serviceManager.lookup(RightManager.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        String contentId = parameters.getParameter("contentId", "");
        if (StringUtils.isNotEmpty(contentId))
        {
            contentId = URIUtils.decode(contentId);
        }
        Content content = _ametysResolver.resolveById(contentId);

        // Check 'start-date' attribute exists
        if (!content.hasValue("start-date"))
        {
            throw new IllegalArgumentException("The content must have a metadata named 'start-date' to be imported as an event");
        }
        // Check user access
        _checkUserAccess(content);
        
        contentHandler.startDocument();
        XMLUtils.startElement(contentHandler, "events");
        XMLUtils.startElement(contentHandler, "contents");

        saxContent(contentHandler, content, false, null, false);

        XMLUtils.endElement(contentHandler, "contents");

        XMLUtils.endElement(contentHandler, "events");
        contentHandler.endDocument();
    }
    
    private void _checkUserAccess(Content content)
    {
        UserIdentity user = _currentUserProvider.getUser();
        boolean readAccess = _rightManager.hasReadAccess(user, content);
        
        if (!readAccess && user == null)
        {
            throw new AuthorizationRequiredException();
        }
        else if (!readAccess)
        {
            throw new AccessDeniedException("Access to event '" + content.getId() + "' is not allowed for user " + user);
        }
    }
}
