/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
 
 CalendarSearchService = {
    
    RELATIVE_URL: "/_plugins/calendar/service/search-events/${serviceId}/${resultMode}/${start}/${end}/_results-page.html",
    ICAL_URL: "/_plugins/calendar/service/search-events/${siteName}/${lang}/${serviceId}/${tags}/agenda.ics",
    PDF_URL: "/_plugins/calendar/service/search-events/${siteName}/${lang}/${serviceId}/${start}/${end}/${tags}/agenda.pdf",
    
    DATA_SEARCH_ATTR: 'data-ametys-search-role',
    FORM_SELECTOR: '[data-ametys-search-role=form]',
    FORM_HIDDEN_SELECTOR: '[data-ametys-search-role=form-criteria-hidden]',
    SEARCH_RESULTS_SELECTOR: '[data-ametys-search-role=search-results]',
    TAG_FILTER_SELECTOR: '[data-ametys-search-role=tag-filter]',
    ICS_LINK_SELECTOR: '[data-ametys-search-role=ics-link]',
    PDF_LINK_SELECTOR: '[data-ametys-search-role=pdf-link]',
    DATA_ZONE_LEVEL_ATTR: 'data-ametys-zone-level',
    ZONE_LEVEL_SELECTOR: '[data-ametys-zone-level]',
    LOADING_SELECTOR: '[data-ametys-search-role=loading]',
    
    /**
     * Load events
     * @param {Object} config The object configuration
     * @param {String} config.selector The jquery selector of service
     * @param {String} config.siteUriPrefix The site URI prefix
     * @param {String} config.zoneItemId The zone item id
     * @param {String} config.siteName The site name
     * @param {String} config.lang The current language
     * @param {Boolean} config.isSubmit true if criteria form is submit
     * @param {String} [config.searchResultsSelector] The jquery selector for results to be replaced. By default '[data-ametys-search-role=search-results]'
     * @param {String} start The start date as ISO string
     * @param {String} end The end date as ISO string
     * @param {Function} [successCb] function to call in case of success
     * @param {Function} [errorCb] function to call in case of error
     */
    launchSearch: function(config, start, end, successCb, errorCb)
    {
        var $rootElement = $j(config.selector);
        var $searchResults = $rootElement.find(config.searchResultsSelector || this.SEARCH_RESULTS_SELECTOR);
        var $loading = $rootElement.find(this.LOADING_SELECTOR);
        
        var url = this._computeUrl(config, start, end);
        
        var params = {};
        
        var zoneLevel = $rootElement.closest(this.ZONE_LEVEL_SELECTOR).attr(this.DATA_ZONE_LEVEL_ATTR);
        params["zoneLevel"] = zoneLevel;
        
        var me = this;
        this._handleForm(config, $rootElement, params);
        this._handleTagFilter(config, $rootElement, params);
        
        console.log("Load events between " + start + " and " + end);
        
        $j.ajax({
            url: url,
            method: "POST",
            data: params,
            traditional: true,
            dataType: "html",
            beforeSend: function() {
                $loading.show();
            },
            success: function(response){
                $searchResults.children().remove();
                $searchResults.append(response);

                me.updateICSLink(config, $rootElement);
                // me.updatePDFLink(config, $rootElement, start, end);
                
                if (typeof successCb == 'function')
                {
                    successCb(response);
                }
            },
            error: function(){
                console.error("Failed to load calendar events between " + start + " and " + end);
                
                if (typeof errorCb == 'function')
                {
                    errorCb();
                }
            },
            complete: function() {
                $loading.hide();
            }
        });
    },
    
    _computeUrl: function(config, start, end)
    {
        var url = config.siteUriPrefix + this.RELATIVE_URL;
        
        var serviceId = config.zoneItemId.replace("://", "/");
        
        return url.replace("${serviceId}", serviceId)
                .replace("${resultMode}", config.resultMode || "full")
                .replace("${start}", encodeURIComponent(encodeURIComponent(start)))
                .replace("${end}", encodeURIComponent(encodeURIComponent(end)));
                
    },
    
    _handleForm: function(config, $htmlElement, params)
    {
        var $form = config.isSubmit ? this._retrieveForm(config, $htmlElement) : this._retrieveHiddenForm(config, $htmlElement);
                        
        $form.serializeArray()
            .forEach(function (param) {
                var name = param.name,
                val = param.value;
                if (name && params.hasOwnProperty(name))
                {
                    valuesArray = params[name];
                    valuesArray.push(val);
                }
                else if (name)
                {
                    params[name] = [val];
                }
            });
    },
    
    _retrieveHiddenForm: function(config, $htmlElement)
    {
        var $form = $htmlElement.find(this.FORM_HIDDEN_SELECTOR);
        return $form;
    },
    
    _retrieveForm: function(config, $htmlElement)
    {
        var $form = $htmlElement.find(this.FORM_SELECTOR);
        return $form;
    },
    
    _handleTagFilter: function(config, $htmlElement, params)
    {
        params['tags'] = this._getSelectedTags(config, $htmlElement).join(',');
    },
    
    _getSelectedTags: function(config, $htmlElement)
    {
        var $tagFilter = $htmlElement.find(this.TAG_FILTER_SELECTOR);
        return $tagFilter.find('input[type=checkbox]:checked').map(function() {
            return this.value
        }).get();
    },
    
    updateICSLink: function(config, $htmlElement)
    {
        var $icsLink = $htmlElement.find(this.ICS_LINK_SELECTOR);
        if ($icsLink)
        {
            var url = this._computeIcalUrl(config, $htmlElement);
            $icsLink.attr("href", url);
        }
    },
    
    _computeIcalUrl: function(config, $htmlElement)
	{
	    var url = config.siteUriPrefix + this.ICAL_URL;
        
        var serviceId = config.zoneItemId.replace("://", "/");
        var tags = this._getSelectedTags(config, $htmlElement);
        
        return url.replace("${siteName}", config.siteName)
                .replace("${lang}", config.lang)
                .replace("${serviceId}", serviceId)
                .replace("${tags}", tags.length > 0 ? tags.join(",") : "_ALL");
                
	},
    
    updatePDFLink: function(config, $htmlElement, start, end)
    {
        var $pdfLink = $htmlElement.find(this.PDF_LINK_SELECTOR);
        if ($pdfLink)
        {
            var url = this._computePDFUrl(config, $htmlElement, start, end);
            $pdfLink.attr("href", url);
        }
    },
    
    _computePDFUrl: function(config, $htmlElement, start, end)
    {
        var url = config.siteUriPrefix + this.PDF_URL;
        
        var serviceId = config.zoneItemId.replace("://", "/");
        var tags = this._getSelectedTags(config, $htmlElement);
        
        return url.replace("${siteName}", config.siteName)
                .replace("${lang}", config.lang)
                .replace("${serviceId}", serviceId)
                .replace("${start}", encodeURIComponent(encodeURIComponent(start)))
                .replace("${end}", encodeURIComponent(encodeURIComponent(end)))
                .replace("${tags}", tags.length > 0 ? tags.join(",") : "_ALL");
                
    }
 }