/*
 *  Copyright 2012 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.calendar.events;

import java.io.IOException;
import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoField;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.UUID;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.repository.Content;
import org.ametys.cms.tag.Tag;
import org.ametys.cms.tag.TagProviderExtensionPoint;
import org.ametys.core.util.URIUtils;
import org.ametys.plugins.calendar.icsreader.IcsEventHelper;
import org.ametys.plugins.calendar.icsreader.IcsReader;
import org.ametys.plugins.calendar.icsreader.IcsReader.IcsEvents;
import org.ametys.plugins.calendar.icsreader.LocalVEvent;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareRepeaterEntry;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.WebConstants;
import org.ametys.web.content.GetSiteAction;
import org.ametys.web.filter.WebContentFilter;
import org.ametys.web.filter.WebContentFilter.AccessLimitation;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.SitemapElement;
import org.ametys.web.repository.page.ZoneItem;

import net.fortuna.ical4j.model.Property;
import net.fortuna.ical4j.model.component.VEvent;
import net.fortuna.ical4j.model.property.Created;
import net.fortuna.ical4j.model.property.DtStamp;
import net.fortuna.ical4j.model.property.LastModified;

/**
 * Query and generate news according to many parameters.
 */
public class EventsGenerator extends AbstractEventGenerator
{
    /** The ametys object resolver. */
    protected AmetysObjectResolver _ametysResolver;

    /** The events helper */
    protected EventsFilterHelper _eventsFilterHelper;
    
    /** The ICS Reader */
    protected IcsReader _icsReader;

    /** The tag provider extension point. */
    protected TagProviderExtensionPoint _tagProviderEP;

    private IcsEventHelper _icsEventHelper;

    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _ametysResolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _eventsFilterHelper = (EventsFilterHelper) serviceManager.lookup(EventsFilterHelper.ROLE);
        _icsEventHelper = (IcsEventHelper) serviceManager.lookup(IcsEventHelper.ROLE);
        _icsReader = (IcsReader) serviceManager.lookup(IcsReader.ROLE);
        _tagProviderEP = (TagProviderExtensionPoint) manager.lookup(TagProviderExtensionPoint.ROLE);
    }

    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        @SuppressWarnings("unchecked")
        Map<String, Object> parentContextAttrs = (Map<String, Object>) objectModel.get(ObjectModelHelper.PARENT_CONTEXT);
        if (parentContextAttrs == null)
        {
            parentContextAttrs = Collections.EMPTY_MAP;
        }

        LocalDate today = LocalDate.now();

        // Get site and language in sitemap parameters. Can not be null.
        String siteName = parameters.getParameter("site", (String) request.getAttribute(WebConstants.REQUEST_ATTR_SITE_NAME));
        String lang = parameters.getParameter("lang", (String) request.getAttribute("renderingLanguage"));
        if (StringUtils.isEmpty(lang))
        {
            lang = (String) request.getAttribute(WebConstants.REQUEST_ATTR_SITEMAP_NAME);
        }
        // Get the parameters.
        int monthsBefore = parameters.getParameterAsInteger("months-before", 3);
        int monthsAfter = parameters.getParameterAsInteger("months-after", 3);
        // Type can be "calendar", "single-day" or "agenda".
        String type = parameters.getParameter("type", "calendar");
        String view = parameters.getParameter("view", "");
        int year = parameters.getParameterAsInteger("year", today.getYear());
        int month = parameters.getParameterAsInteger("month", today.getMonthValue());
        int day = parameters.getParameterAsInteger("day", today.getDayOfMonth());
        // Select a single tag or "all".
        String requestedTagsString = parameters.getParameter("tags", "all");
        
        
        Page currentPage = (Page) request.getAttribute(WebConstants.REQUEST_ATTR_PAGE);
        
        // Get the zone item, as a request attribute or from the ID in the
        // parameters.
        ZoneItem zoneItem = (ZoneItem) request.getAttribute(WebConstants.REQUEST_ATTR_ZONEITEM);
        String zoneItemId = parameters.getParameter("zoneItemId", "");
        if (zoneItem == null && StringUtils.isNotEmpty(zoneItemId))
        {
            zoneItemId = URIUtils.decode(zoneItemId);
            zoneItem = (ZoneItem) _ametysResolver.resolveById(zoneItemId);
        }
        
        if (currentPage == null && zoneItem != null)
        {
            // Wrapped page such as _plugins/calendar/page/YEAR/MONTH/DAY/ZONEITEMID/events_1.3.html => get the page from its zone item
            // The page is needed to get restriction
            SitemapElement sitemapElement = zoneItem.getZone().getSitemapElement();
            if (sitemapElement instanceof Page page)
            {
                currentPage = page;
            }
            else
            {
                throw new IllegalStateException("The calendar service cannot be inherited from the sitemap root");
            }
        }
        
        ZonedDateTime dateTime = ZonedDateTime.of(year, month, day, 0, 0, 0, 0, ZoneId.systemDefault());
        String title = _eventsFilterHelper.getTitle(zoneItem);
        String rangeType = parameters.getParameter("rangeType", _eventsFilterHelper.getDefaultRangeType(zoneItem));
        boolean maskOrphan = _eventsFilterHelper.getMaskOrphan(zoneItem);
        boolean pdfDownload = _eventsFilterHelper.getPdfDownload(zoneItem);
        boolean icalDownload = _eventsFilterHelper.getIcalDownload(zoneItem);
        String link = _eventsFilterHelper.getLink(zoneItem);
        String linkTitle = _eventsFilterHelper.getLinkTitle(zoneItem);
        
        boolean doRetrieveView = !StringUtils.equalsIgnoreCase("false", parameters.getParameter("do-retrieve-view", "true"));

        // Get the search context to match, from the zone item or from the parameters.
        @SuppressWarnings("unchecked")
        List<Map<String, Object>> searchContexts = _eventsFilterHelper.getSearchContext(zoneItem, (List<Map<String, Object>>) parentContextAttrs.get("search"));
        
        
        Set<String> tags = _eventsFilterHelper.getTags(zoneItem, searchContexts);
        Set<Tag> categories = _eventsFilterHelper.getTagCategories(zoneItem, searchContexts, siteName);
        Set<Tag> icsTags = _getIcsTags(zoneItem, siteName);
        String pagePath = currentPage != null ? currentPage.getPathInSitemap() : "";
        
        Set<String> filteredCategories = _eventsFilterHelper.getFilteredCategories(null, requestedTagsString.split(","), zoneItem, siteName);
        // Get the date range and deduce the expression (single day or month-before to month-after).
        EventsFilterHelper.DateTimeRange dateRange = _eventsFilterHelper.getDateRange(type, year, month, day, monthsBefore, monthsAfter, rangeType);
        
        EventsFilter eventsFilter = _eventsFilterHelper.generateEventFilter(dateRange, zoneItem, view, type, filteredCategories, searchContexts);
        
        // Get the corresponding contents.
        AmetysObjectIterable<Content> eventContents = eventsFilter.getMatchingContents(siteName, lang, currentPage);
        
        // Read ICS threads
        List<IcsEvents> parsedICS;
        if (icalDownload && zoneItem != null && type.equals("full"))
        {
            // If we are exporting the ICS, we do not want to use maximum events or ICS file size limitation
            parsedICS = _icsEventHelper.getICSEvents(zoneItem, siteName, dateRange, Long.MAX_VALUE, Long.MAX_VALUE);
        }
        else
        {
            parsedICS = _icsEventHelper.getICSEvents(zoneItem, siteName, dateRange);
        }
        
        // CAL-94 (same as CMS-2292 for filtered contents)
        String currentSiteName = (String) request.getAttribute(WebConstants.REQUEST_ATTR_SITE_NAME);
        String currentSkinName = (String) request.getAttribute(WebConstants.REQUEST_ATTR_SKIN_ID);
        String currentTemplateName = (String) request.getAttribute(WebConstants.REQUEST_ATTR_TEMPLATE_ID);
        String currentLanguage = (String) request.getAttribute("renderingLanguage");
        request.setAttribute(GetSiteAction.OVERRIDE_SITE_REQUEST_ATTR, currentSiteName);
        request.setAttribute(GetSiteAction.OVERRIDE_SKIN_REQUEST_ATTR, currentSkinName);

        try
        {
            _sax(today, monthsBefore, monthsAfter, year, month, day, filteredCategories, currentPage, zoneItem, dateTime, title, rangeType, maskOrphan, pdfDownload, icalDownload, link, linkTitle, doRetrieveView, tags, categories, icsTags, pagePath, eventsFilter, dateRange, eventContents, parsedICS);
        }
        finally
        {
            request.removeAttribute(GetSiteAction.OVERRIDE_SITE_REQUEST_ATTR);
            request.removeAttribute(GetSiteAction.OVERRIDE_SKIN_REQUEST_ATTR);
            request.setAttribute(WebConstants.REQUEST_ATTR_SITE_NAME, currentSiteName);
            request.setAttribute("siteName", currentSiteName);
            request.setAttribute(WebConstants.REQUEST_ATTR_SKIN_ID, currentSkinName);
            request.setAttribute(WebConstants.REQUEST_ATTR_TEMPLATE_ID, currentTemplateName);
            request.setAttribute("renderingLanguage", currentLanguage);
        }
    }
    
    private Set<Tag> _getIcsTags(ZoneItem zoneItem, String currentSiteName)
    {
        ModelAwareDataHolder serviceParameters = zoneItem.getServiceParameters();
        Set<Tag> categories = new LinkedHashSet<>();
        
        // Add the categories defined in the ICS fields
        if (serviceParameters.hasValue("ics"))
        {
            ModifiableModelAwareRepeater icsRepeater = serviceParameters.getValue("ics");
            for (ModifiableModelAwareRepeaterEntry repeaterEntry : icsRepeater.getEntries())
            {
                String categoryName = repeaterEntry.getValue("tag");
                Map<String, Object> contextualParameters = new HashMap<>();
                contextualParameters.put("siteName", currentSiteName);
                Tag category = _tagProviderEP.getTag(categoryName, contextualParameters);
                if (category != null)
                {
                    categories.add(category);
                }
            }
        }
        return categories;
    }

    

    private void _sax(LocalDate today, int monthsBefore, int monthsAfter, int year, int month, int day, Set<String> filteredCategoryTags, Page page, ZoneItem zoneItem, ZonedDateTime date,
            String title, String rangeType, boolean maskOrphan, boolean pdfDownload, boolean icalDownload, String link, String linkTitle, boolean doRetrieveView, Set<String> tags,
            Set<Tag> categories, Set<Tag> icsTags, String pagePath, EventsFilter eventsFilter, EventsFilterHelper.DateTimeRange dateRange, AmetysObjectIterable<Content> eventContents, List<IcsEvents> icsEvents)
            throws SAXException, IOException
    {
        AttributesImpl atts = new AttributesImpl();

        atts.addCDATAAttribute("page-path", pagePath);
        atts.addCDATAAttribute("today", DateTimeFormatter.ISO_LOCAL_DATE.format(today));
        if (dateRange != null)
        {
            if (dateRange.fromDate() != null)
            {
                atts.addCDATAAttribute("start", DateTimeFormatter.ISO_LOCAL_DATE.format(dateRange.fromDate()));
            }
            if (dateRange.untilDate() != null)
            {
                atts.addCDATAAttribute("end", DateTimeFormatter.ISO_LOCAL_DATE.format(dateRange.untilDate()));
            }
        }

        atts.addCDATAAttribute("year", Integer.toString(year));
        atts.addCDATAAttribute("month", String.format("%02d", month));
        atts.addCDATAAttribute("day", String.format("%02d", day));
        atts.addCDATAAttribute("months-before", Integer.toString(monthsBefore));
        atts.addCDATAAttribute("months-after", Integer.toString(monthsAfter));

        atts.addCDATAAttribute("title", title);
        atts.addCDATAAttribute("mask-orphan", Boolean.toString(maskOrphan));
        atts.addCDATAAttribute("pdf-download", Boolean.toString(pdfDownload));
        atts.addCDATAAttribute("ical-download", Boolean.toString(icalDownload));
        atts.addCDATAAttribute("link", link);
        atts.addCDATAAttribute("link-title", linkTitle);

        if (zoneItem != null)
        {
            atts.addCDATAAttribute("zoneItemId", zoneItem.getId());
        }
        if (StringUtils.isNotEmpty(rangeType))
        {
            atts.addCDATAAttribute("range", rangeType);
        }
        
        if (!filteredCategoryTags.isEmpty())
        {
            atts.addCDATAAttribute("requested-tags", String.join(",", filteredCategoryTags));
        }

        contentHandler.startDocument();
        XMLUtils.startElement(contentHandler, "events", atts);

        _saxRssUrl(zoneItem);

        // Generate months (used in calendar mode) and days (used in full-page
        // agenda mode).
        _saxMonths(dateRange);
        _saxDays(date, rangeType);

        _saxDaysNew(dateRange, rangeType);

        // Generate tags and categories.
        _saxTags(tags);
        _saxCategories(categories, icsTags);

        Pair<List<LocalVEvent>, String> parsedICSEvents = _icsEventHelper.toLocalIcsEvent(icsEvents, dateRange);
        List<LocalVEvent> localIcsEvents = parsedICSEvents.getLeft();
        String fullICSDistantEvents = parsedICSEvents.getRight();
        
        // Generate the matching contents.
        XMLUtils.startElement(contentHandler, "contents");

        saxMatchingContents(contentHandler, eventsFilter, eventContents, page, doRetrieveView);

        saxIcsEvents(contentHandler, localIcsEvents);

        XMLUtils.endElement(contentHandler, "contents");

        XMLUtils.createElement(contentHandler, "rawICS", fullICSDistantEvents);
        
        // Generate ICS events with errors
        _icsEventHelper.saxICSErrors(icsEvents, contentHandler);
        
        if (icalDownload)
        {
            // Generate VTimeZones for distant events
            String timezones = _icsEventHelper.toVTimeZone(icsEvents, dateRange, List.of());
            XMLUtils.createElement(contentHandler, "timezones", timezones);
        }
        
        XMLUtils.endElement(contentHandler, "events");

        contentHandler.endDocument();
    }

    private void _saxRssUrl(ZoneItem zoneItem) throws SAXException
    {
        if (zoneItem != null)
        {
            ModelAwareDataHolder serviceParameters = zoneItem.getServiceParameters();
            // First check that there is a value because calendar service doesn't define the rss parameter
            if (serviceParameters.hasValue("rss") && (boolean) serviceParameters.getValue("rss"))
            {
                // Only add RSS if there is a search context
                if (serviceParameters.hasValue("search"))
                {
                    ModifiableModelAwareRepeater searchRepeater = serviceParameters.getValue("search");
                    if (searchRepeater.getSize() > 0)
                    {
                        // Split protocol and id
                        String[] zoneItemId = zoneItem.getId().split("://");
                        String url = "_plugins/calendar/" + zoneItemId[1] + "/rss.xml";
                        
                        XMLUtils.createElement(contentHandler, "rssUrl", url);
                    }
                }
            }
        }
    }
    
    /**
     * SAX all contents matching the given filter
     * 
     * @param handler The content handler to SAX into
     * @param filter The filter
     * @param contents iterator on the contents.
     * @param currentPage The current page.
     * @param saxContentItSelf true to sax the content, false will only sax some meta
     * @throws SAXException If an error occurs while SAXing
     * @throws IOException If an error occurs while retrieving content.
     */
    public void saxMatchingContents(ContentHandler handler, WebContentFilter filter, AmetysObjectIterable<Content> contents, Page currentPage, boolean saxContentItSelf) throws SAXException, IOException
    {
        boolean checkUserAccess = filter.getAccessLimitation() == AccessLimitation.USER_ACCESS;
        
        for (Content content : contents)
        {
            if (_filterHelper.isContentValid(content, currentPage, filter))
            {
                saxContent(handler, content, saxContentItSelf, filter, checkUserAccess);
            }
        }
    }
    
    /**
     * Sax a list of events coming from a distant ICS file
     * @param handler The content handler to SAX into
     * @param icsEvents the events to sax
     * @throws SAXException Something went wrong
     */
    public void saxIcsEvents(ContentHandler handler, List<LocalVEvent> icsEvents) throws SAXException
    {
        for (LocalVEvent icsEvent : icsEvents)
        {
            saxIcsEvent(handler, icsEvent);
        }
    }
    
    /**
     * Sax an event coming from a distant ICS file
     * @param handler The content handler to SAX into
     * @param icsEvent an event to sax
     * @throws SAXException Something went wrong
     */
    public void saxIcsEvent(ContentHandler handler, LocalVEvent icsEvent) throws SAXException
    {
        VEvent event = icsEvent.getEvent();
        AttributesImpl attrs = new AttributesImpl();

        String start = org.ametys.core.util.DateUtils.getISODateTimeFormatter().format(icsEvent.getStart());
        String end = org.ametys.core.util.DateUtils.getISODateTimeFormatter().format(icsEvent.getEnd());
        List<String> params = new ArrayList<>();
        
        String title = event.getProperty(Property.SUMMARY).map(Property::getValue).orElse("");
        String id = event.getProperty(Property.UID).map(Property::getValue).orElse(UUID.randomUUID().toString());
        String eventAbstract = event.getProperty(Property.DESCRIPTION).map(Property::getValue).orElse("");
        
        params.add(title);

        if (start != null)
        {
            String startAttr = icsEvent.getStart().format(DateTimeFormatter.ISO_LOCAL_DATE);
            params.add(start);
            attrs.addCDATAAttribute("start", startAttr);
        }
        
        if (end != null)
        {
            String endAttr = icsEvent.getEnd().format(DateTimeFormatter.ISO_LOCAL_DATE);
            params.add(end);
            attrs.addCDATAAttribute("end", endAttr);
        }

        XMLUtils.startElement(handler, "event", attrs);

        String key = end == null ? "CALENDAR_SERVICE_AGENDA_EVENT_TITLE_SINGLE_DAY" : "CALENDAR_SERVICE_AGENDA_FROM_TO";
        I18nizableText description = new I18nizableText(null, key, params);
        description.toSAX(handler, "description");
        
        attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", "ics://" + id);
        attrs.addCDATAAttribute("title", title);
        
        DtStamp dateTimeStamp = event.getDateTimeStamp();
        Instant dtStamp = dateTimeStamp != null ? dateTimeStamp.getDate() : Instant.now();
        
        Created created = event.getCreated();
        Instant createdAtDate = created != null ? created.getDate() : dtStamp;
        
        LastModified lastModified = event.getLastModified();
        Instant lastModifiedDate = lastModified != null ? lastModified.getDate() : dtStamp;

        String createdAt =  org.ametys.core.util.DateUtils.asZonedDateTime(createdAtDate, null).format(DateTimeFormatter.ISO_INSTANT);
        attrs.addCDATAAttribute("createdAt", createdAt);

        String lastModifiedAt =  org.ametys.core.util.DateUtils.asZonedDateTime(lastModifiedDate, null).format(DateTimeFormatter.ISO_INSTANT);
        attrs.addCDATAAttribute("lastModifiedAt", lastModifiedAt);
        
        XMLUtils.startElement(handler, "content", attrs);

        XMLUtils.startElement(handler, "metadata");
        attrs = new AttributesImpl();
        attrs.addCDATAAttribute("typeId", "string");
        attrs.addCDATAAttribute("multiple", "false");
        XMLUtils.createElement(handler, "title", attrs, title);
        XMLUtils.createElement(handler, "abstract", attrs, eventAbstract);
        

        attrs = new AttributesImpl();
        attrs.addCDATAAttribute("typeId", "datetime");
        attrs.addCDATAAttribute("multiple", "false");
        XMLUtils.createElement(handler, "start-date", attrs, start);
        XMLUtils.createElement(handler, "end-date", attrs, end);

        XMLUtils.endElement(handler, "metadata");

        Tag tag = icsEvent.getTag();
        if (tag != null)
        {
            XMLUtils.startElement(handler, "tags");
            attrs = new AttributesImpl();
            attrs.addCDATAAttribute("parent", tag.getParentName());
            XMLUtils.startElement(handler, tag.getName(), attrs);
            tag.getTitle().toSAX(handler);
            XMLUtils.endElement(handler, tag.getName());
            XMLUtils.endElement(handler, "tags");
        }
        
        XMLUtils.endElement(handler, "content");
        XMLUtils.endElement(handler, "event");
    }

    /**
     * SAX information on the months spanning the date range.
     * @param dateRange the date range.
     * @throws SAXException if a error occurs while saxing
     */
    protected void _saxMonths(EventsFilterHelper.DateTimeRange dateRange) throws SAXException
    {
        if (dateRange != null && dateRange.fromDate() != null && dateRange.untilDate() != null)
        {
            AttributesImpl atts = new AttributesImpl();

            XMLUtils.startElement(contentHandler, "months");

            ZonedDateTime date = dateRange.fromDate();
            ZonedDateTime end = dateRange.untilDate();

            while (date.isBefore(end))
            {
                int year = date.getYear();
                int month = date.getMonthValue();
                
                String monthStr = String.format("%d-%02d", year, month);
                String dateStr = org.ametys.core.util.DateUtils.getISODateTimeFormatter().format(date);

                atts.clear();
                atts.addCDATAAttribute("str", monthStr);
                atts.addCDATAAttribute("raw", dateStr);
                XMLUtils.startElement(contentHandler, "month", atts);

                XMLUtils.endElement(contentHandler, "month");

                date = date.plusMonths(1);
            }

            XMLUtils.endElement(contentHandler, "months");
        }
    }

    /**
     * Generate days to build a "calendar" view.
     * 
     * @param dateRange a date belonging to the time span to generate.
     * @param rangeType the range type, "month" or "week".
     * @throws SAXException if an error occurs while saxing
     */
    protected void _saxDaysNew(EventsFilterHelper.DateTimeRange dateRange, String rangeType) throws SAXException
    {
        if (dateRange != null)
        {
            XMLUtils.startElement(contentHandler, "calendar-months");
    
            ZonedDateTime date = dateRange.fromDate();
            ZonedDateTime end = dateRange.untilDate();
            
            while (date.isBefore(end))
            {
                int year = date.getYear();
                int month = date.getMonthValue();
    
                String monthStr = String.format("%d-%02d", year, month);
                String dateStr = org.ametys.core.util.DateUtils.getISODateTimeFormatter().format(date);
    
                AttributesImpl attrs = new AttributesImpl();
                attrs.addCDATAAttribute("str", monthStr);
                attrs.addCDATAAttribute("raw", dateStr);
                attrs.addCDATAAttribute("year", Integer.toString(year));
                attrs.addCDATAAttribute("month", Integer.toString(month));
                XMLUtils.startElement(contentHandler, "month", attrs);
    
                _saxDays(date, "month");
    
                XMLUtils.endElement(contentHandler, "month");
    
                date = date.plusMonths(1);
            }
    
            XMLUtils.endElement(contentHandler, "calendar-months");
        }
    }

    /**
     * Generate days to build a "calendar" view.
     * 
     * @param date a date belonging to the time span to generate.
     * @param type the range type, "month" or "week".
     * @throws SAXException if an error occurs while saxing
     */
    protected void _saxDays(ZonedDateTime date, String type) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();

        int rangeStyle = DateUtils.RANGE_MONTH_MONDAY;
        ZonedDateTime previousDay = null;
        ZonedDateTime nextDay = null;

        // Week.
        if ("week".equals(type))
        {
            rangeStyle = DateUtils.RANGE_WEEK_MONDAY;

            // Get the first day of the week.
            previousDay = date.with(ChronoField.DAY_OF_WEEK, 1);
            // First day of next week.
            nextDay = previousDay.plusWeeks(1);
            // First day of previous week.
            previousDay = previousDay.minusWeeks(1);
        }
        else
        {
            rangeStyle = DateUtils.RANGE_MONTH_MONDAY;

            // Get the first day of the month.
            previousDay = date.with(ChronoField.DAY_OF_MONTH, 1);
            // First day of previous month.
            nextDay = previousDay.plusMonths(1);
            // First day of next month.
            previousDay = previousDay.minusMonths(1);
        }

        addNavAttributes(attrs, date, previousDay, nextDay);

        // Get an iterator on the days to be present on the calendar.
        
        Iterator<Calendar> days = DateUtils.iterator(org.ametys.core.util.DateUtils.asDate(date), rangeStyle);

        XMLUtils.startElement(contentHandler, "calendar", attrs);

        ZonedDateTime previousWeekDay = date.minusWeeks(1);
        ZonedDateTime nextWeekDay = date.plusWeeks(1);

        AttributesImpl weekAttrs = new AttributesImpl();
        addNavAttributes(weekAttrs, date, previousWeekDay, nextWeekDay);

        XMLUtils.startElement(contentHandler, "week", weekAttrs);

        while (days.hasNext())
        {
            Calendar dayCal = days.next();
            
            ZonedDateTime day = dayCal.toInstant().atZone(dayCal.getTimeZone().toZoneId());
            String rawDateStr = org.ametys.core.util.DateUtils.getISODateTimeFormatter().format(day);
            String dateStr = DateTimeFormatter.ISO_LOCAL_DATE.format(day);
            String yearStr = Integer.toString(dayCal.get(Calendar.YEAR));
            String monthStr = Integer.toString(dayCal.get(Calendar.MONTH) + 1);
            String dayStr = Integer.toString(dayCal.get(Calendar.DAY_OF_MONTH));

            AttributesImpl dayAttrs = new AttributesImpl();

            dayAttrs.addCDATAAttribute("raw", rawDateStr);
            dayAttrs.addCDATAAttribute("date", dateStr);
            dayAttrs.addCDATAAttribute("year", yearStr);
            dayAttrs.addCDATAAttribute("month", monthStr);
            dayAttrs.addCDATAAttribute("day", dayStr);

            XMLUtils.createElement(contentHandler, "day", dayAttrs);

            // Break on week on the last day of the week (but not on the last
            // week).
            if (dayCal.get(Calendar.DAY_OF_WEEK) == _eventsFilterHelper.getLastDayOfWeek(dayCal) && days.hasNext())
            {
                previousWeekDay = day.minusDays(6);
                nextWeekDay = day.plusDays(8);
                
                weekAttrs.clear();
                addNavAttributes(weekAttrs, day, previousWeekDay, nextWeekDay);

                XMLUtils.endElement(contentHandler, "week");
                XMLUtils.startElement(contentHandler, "week", weekAttrs);
            }
        }

        XMLUtils.endElement(contentHandler, "week");
        XMLUtils.endElement(contentHandler, "calendar");
    }

    /**
     * Add nav attributes.
     * 
     * @param attrs the attributes object to fill in.
     * @param current the current date.
     * @param previousDay the previous date.
     * @param nextDay the next date.
     */
    protected void addNavAttributes(AttributesImpl attrs, ZonedDateTime current, ZonedDateTime previousDay, ZonedDateTime nextDay)
    {
        attrs.addCDATAAttribute("current", org.ametys.core.util.DateUtils.getISODateTimeFormatter().format(current));

        attrs.addCDATAAttribute("previous", org.ametys.core.util.DateUtils.getISODateTimeFormatter().format(previousDay));
        attrs.addCDATAAttribute("previousYear", Integer.toString(previousDay.getYear()));
        attrs.addCDATAAttribute("previousMonth", Integer.toString(previousDay.getMonthValue()));
        attrs.addCDATAAttribute("previousDay", Integer.toString(previousDay.getDayOfMonth()));

        attrs.addCDATAAttribute("next", org.ametys.core.util.DateUtils.getISODateTimeFormatter().format(nextDay));
        attrs.addCDATAAttribute("nextYear", Integer.toString(nextDay.getYear()));
        attrs.addCDATAAttribute("nextMonth", Integer.toString(nextDay.getMonthValue()));
        attrs.addCDATAAttribute("nextDay", Integer.toString(nextDay.getDayOfMonth()));
    }

    /**
     * Generate the list of selected tags.
     * @param tags the list of tags.
     * @throws SAXException if an error occurs while saxing
     */
    protected void _saxTags(Collection<String> tags) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "tags");
        for (String tag : tags)
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("name", tag);
            XMLUtils.createElement(contentHandler, "tag", attrs);
        }
        XMLUtils.endElement(contentHandler, "tags");
    }

    /**
     * Generate the list of selected tags that act as categories and their descendant tags.
     * @param categories the list of categories to generate.
     * @param icsTags list of tags for the ICS feeds (tags, not parents)
     * @throws SAXException if an error occurs while saxing
     */
    protected void _saxCategories(Collection<Tag> categories, Collection<Tag> icsTags) throws SAXException
    {
        Map<Tag, Set<Tag>> icsTagsToAdd = new HashMap<>();
        
        // Only the tags that are not already in the ones from the search contexts
        for (Tag tag : icsTags)
        {
            Tag parent = tag.getParent();
            if (icsTagsToAdd.containsKey(parent))
            {
                icsTagsToAdd.get(parent).add(tag);
            }
            else if (categories == null || !categories.contains(tag.getParent()))
            {
                icsTagsToAdd.put(parent, new LinkedHashSet<>());
                icsTagsToAdd.get(parent).add(tag);
            }
        }
        
        XMLUtils.startElement(contentHandler, "tag-categories");
        
     // Add the tags from the search contexts
        if (categories != null)
        {
            for (Tag category : categories)
            {
                XMLUtils.startElement(contentHandler, "category");
    
                category.getTitle().toSAX(contentHandler, "title");
    
                _saxTags(_eventsFilterHelper.getAllTags(category));
    
                XMLUtils.endElement(contentHandler, "category");
            }
        }
        
        // Add the tags from the ICS feeds
        for (Entry<Tag, Set<Tag>> entry : icsTagsToAdd.entrySet())
        {
            Tag parent = entry.getKey();
            Set<Tag> tags = entry.getValue();
            
            XMLUtils.startElement(contentHandler, "category");

            // As in the ICS, we select directly a tag and not a category, it is possible that there are no parent.
            // To keep the XML equivalent, the category is still created, possibly with an empty title
            if (parent != null)
            {
                parent.getTitle().toSAX(contentHandler, "title");
            }
            else
            {
                XMLUtils.createElement(contentHandler, "title");
            }

            _saxTags(tags);

            XMLUtils.endElement(contentHandler, "category");
        }
        
        XMLUtils.endElement(contentHandler, "tag-categories");
    }
    
    /**
     * Sax a list of tags
     * @param tags the list of tags to sax
     * @throws SAXException if an error occurs while saxing
     */
    protected void _saxTags(Set<Tag> tags) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "tags");
        for (Tag tag : tags)
        {
            AttributesImpl tagAttrs = new AttributesImpl();
            tagAttrs.addCDATAAttribute("name", tag.getName());
            XMLUtils.startElement(contentHandler, "tag", tagAttrs);

            tag.getTitle().toSAX(contentHandler);

            XMLUtils.endElement(contentHandler, "tag");
        }
        XMLUtils.endElement(contentHandler, "tags");
    }

}
