/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.captchetat.captcha;

import java.io.UnsupportedEncodingException;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.http.HttpEntity;
import org.apache.http.NameValuePair;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;

import org.ametys.core.cache.AbstractCacheManager;
import org.ametys.core.cache.Cache;
import org.ametys.core.util.JSONUtils;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;


/**
 * Captcha implementation with images
 */
public class CaptchEtatHelper extends AbstractLogEnabled implements Component, Initializable, Serviceable
{

    /** Role */
    public static final String ROLE = CaptchEtatHelper.class.getName();
    
    /** id of the token cache */
    public static final String CAPCHETAT_TOKEN_CACHE = CaptchetatReader.class.getName();
    
    private static final String CAPTCHA_CLIENT_ID = "captchetat.client_id";
    private static final String CAPTCHA_CLIENT_SECRET = "captchetat.client_secret";
    private static final String CAPTCHA_ENDPOINT = "captchetat.endpoint";
    
    private static final Map<Endpoint, String> CAPTCHA_ENDPOINT_AUTH = Map.of(
            Endpoint.PRODUCTION, "https://oauth.piste.gouv.fr/api/oauth/token",
            Endpoint.SANDBOX, "https://sandbox-oauth.piste.gouv.fr/api/oauth/token"
    );
    private static final Map<Endpoint, String> CAPTCHA_ENDPOINT_PISTE = Map.of(
            Endpoint.PRODUCTION, "https://api.piste.gouv.fr/piste/captchetat/v2/",
            Endpoint.SANDBOX, "https://sandbox-api.piste.gouv.fr/piste/captchetat/v2/"
            );
    
    /** the endpoint present in configuration */
    protected Endpoint _endpoint;
    
    private AbstractCacheManager _cacheManager;
    private CloseableHttpClient _httpClient;
    private JSONUtils _jsonUtils;

    /**
     * The type of endpoint available for Captchetat
     */
    protected enum Endpoint
    {
        /** To use the production endpoint */
        PRODUCTION,
        /** To use the sandbox endpoint */
        SANDBOX
    }
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _jsonUtils = (JSONUtils) smanager.lookup(JSONUtils.ROLE);
        _cacheManager = (AbstractCacheManager) smanager.lookup(AbstractCacheManager.ROLE);
    }

    /**
     * Initialize
     */
    @Override
    public void initialize()
    {
        // A little bit less than an hour, so the token is still valid after user submit the captcha
        Duration duration = Duration.ofMinutes(50);
        if (!_cacheManager.hasCache(CAPCHETAT_TOKEN_CACHE))
        {
            _cacheManager.createMemoryCache(CAPCHETAT_TOKEN_CACHE,
                    new I18nizableText("plugin.captchetat", "PLUGINS_CAPTCHETAT_CACHE_TOKEN_LABEL"),
                    new I18nizableText("plugin.captchetat", "PLUGINS_CAPTCHETAT_CACHE_TOKEN_DESCRIPTION"),
                    true,
                    duration);
        }

        RequestConfig config = RequestConfig.custom()
                .setConnectTimeout(20000)
                .setConnectionRequestTimeout(20000)
                .setSocketTimeout(20000).build();
        
        _httpClient = HttpClientBuilder.create()
            .setDefaultRequestConfig(config)
            .disableRedirectHandling()
            .useSystemProperties()
            .build();
        
        Config ametysConfig = Config.getInstance();
        if (ametysConfig != null)
        {
            String endpoint = ametysConfig.<String>getValue(CAPTCHA_ENDPOINT);
            // the endpoint is a mandatory config when using captchetat. If its not present, then this helper is
            // unused so it doesn't matter that the component is not fully initialized
            if (StringUtils.isNotBlank(endpoint))
            {
                _endpoint = Endpoint.valueOf(endpoint.toUpperCase());
            }
        }
    }

    private Cache<Pair<String, String>, String> _getCache()
    {
        return _cacheManager.get(CAPCHETAT_TOKEN_CACHE);
    }
    
    private String _getToken(Pair<String, String> key)
    {
        return _getCache().get(key, this::_computeToken);
    }

    private String _computeToken(Pair<String, String> key)
    {
        String url = CAPTCHA_ENDPOINT_AUTH.get(_endpoint);
        HttpPost httpPost = new HttpPost(url);
        List<NameValuePair> pairs = new ArrayList<>();
        pairs.add(new BasicNameValuePair("grant_type", "client_credentials"));
        pairs.add(new BasicNameValuePair("client_id", key.getLeft()));
        pairs.add(new BasicNameValuePair("client_secret", key.getRight()));
        pairs.add(new BasicNameValuePair("scope", "piste.captchetat"));

        try
        {
            httpPost.setEntity(new UrlEncodedFormEntity(pairs));
        }
        catch (UnsupportedEncodingException e)
        {
            throw new RuntimeException("Error while building request, can't compute token", e);
        }
        
        try (CloseableHttpResponse response = _httpClient.execute(httpPost))
        {
            HttpEntity entity = response.getEntity();
            
            String responseString = EntityUtils.toString(entity, "UTF-8");
            switch (response.getStatusLine().getStatusCode())
            {
                case 200:
                    break;
                
                case 403:
                    throw new IllegalStateException("The CMS back-office refused the connection");
                    
                case 500:
                default:
                    throw new IllegalStateException("The captchEtat token generator server returned an error " + response.getStatusLine().getStatusCode() + ": " + responseString);
            }
            Map<String, Object> result = _jsonUtils.convertJsonToMap(responseString);
            return (String) result.get("access_token");
        }
        catch (Exception e)
        {
            throw new RuntimeException("Error during request, can't compute token", e);
        }
    }

    /**
     * get Token
     * @return token
     */
    public String getToken()
    {
        return _getToken(Pair.of(Config.getInstance().getValue(CAPTCHA_CLIENT_ID), Config.getInstance().getValue(CAPTCHA_CLIENT_SECRET)));
    }

    /**
     * get Token
     * @return token
     */
    public String getEndpoint()
    {
        return CAPTCHA_ENDPOINT_PISTE.get(_endpoint);
    }
    
    /**
     * Check if the captcha is correct
     * @param key the key
     * @param value the value
     * @return true if correct
     */
    public boolean checkAndInvalidateCaptcha(String key, String value)
    {
        String url = getEndpoint() + "valider-captcha";
        HttpPost httpPost = new HttpPost(url);
        String token = getToken();
        httpPost.addHeader("Authorization", "Bearer " + token);
        httpPost.addHeader("accept", "application/json");
        httpPost.addHeader("Content-Type", "application/json");
        
        Map<String, Object> parameters = new HashMap<>();
        parameters.put(_getIdentifierParameterName(), key);
        parameters.put("code", value);
        String json = _jsonUtils.convertObjectToJson(parameters);
        HttpEntity jsonEntity = new StringEntity(json, ContentType.create("application/json", StandardCharsets.UTF_8));
        
        httpPost.setEntity(jsonEntity);
        
        try (CloseableHttpResponse response = _httpClient.execute(httpPost))
        {
            switch (response.getStatusLine().getStatusCode())
            {
                case 200:
                    break;
                case 403:
                    throw new IllegalStateException("The CMS back-office refused the connection");
                case 500:
                default:
                    throw new IllegalStateException("The captchEtat verification server returned an error");
            }
            HttpEntity entity = response.getEntity();
            String responseString = EntityUtils.toString(entity, "UTF-8");
            return BooleanUtils.toBoolean(responseString);
        }
        catch (Exception e)
        {
            getLogger().error("Error during request, can't verify captcha", e);
        }
        
        return false;
    }
    
    /**
     * Get the name of the parameter for identifier when contacting the API
     * @return the parameter name
     */
    protected String _getIdentifierParameterName()
    {
        return "uuid";
    }
    
}
