/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.captchetat.captcha;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.util.Map;

import org.apache.avalon.excalibur.pool.Recyclable;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.cocoon.reading.Reader;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.Header;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.xml.sax.SAXException;

/**
 * Reader implementation that add header for Captchetat requests
 */
public class CaptchetatReader extends AbstractLogEnabled implements Reader, Recyclable, Serviceable
{
    /** The CaptchEtat helper */
    protected CaptchEtatHelper _captchEtatHelper;

    private OutputStream _os;

    private CloseableHttpClient _httpClient;
    private CloseableHttpResponse _response;


    public void service(ServiceManager manager) throws ServiceException
    {
        _captchEtatHelper = (CaptchEtatHelper) manager.lookup(CaptchEtatHelper.ROLE);
    }
    
    public void setup(SourceResolver resolver, Map objectModel, String src, Parameters par) throws ProcessingException, SAXException, IOException
    {
        RequestConfig config = RequestConfig.custom()
                .setConnectTimeout(20000)
                .setConnectionRequestTimeout(20000)
                .setSocketTimeout(20000).build();
        
        _httpClient = HttpClientBuilder.create()
            .setDefaultRequestConfig(config)
            .disableRedirectHandling()
            .useSystemProperties()
            .build();

        Request request = ObjectModelHelper.getRequest(objectModel);
        
        String url = _captchEtatHelper.getEndpoint() + "simple-captcha-endpoint?" + StringUtils.defaultString(request.getQueryString());
        
        HttpGet httpGet = new HttpGet(url);
        String token = _captchEtatHelper.getToken();
        httpGet.addHeader("Authorization", "Bearer " + token);

        _response = _httpClient.execute(httpGet);
        switch (_response.getStatusLine().getStatusCode())
        {
            case 200:
                break;
            
            case 403:
                throw new IllegalStateException("CaptchEtat refused the connection");
                
            case 500:
            default:
                throw new IllegalStateException("CaptchEtat returned an error: " + org.apache.commons.io.IOUtils.toString(_response.getEntity().getContent(), StandardCharsets.UTF_8));
        }
    }

    public void setOutputStream(OutputStream out) throws IOException
    {
        _os = out;
    }

    public String getMimeType()
    {
        Header contentType = _response.getEntity().getContentType();
        return contentType != null ? contentType.getValue() : null;
    }

    public boolean shouldSetContentLength()
    {
        return false;
    }

    public void generate() throws IOException, SAXException, ProcessingException
    {
        _response.getEntity().writeTo(_os);
    }

    public long getLastModified()
    {
        return 0;
    }

    public void recycle()
    {
        try
        {
            _response.close();
        }
        catch (IOException e)
        {
            getLogger().error("Can't close response", e);
        }
    }
}
