/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.captchetat.captcha;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.util.Map;

import org.apache.avalon.framework.activity.Disposable;
import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.cocoon.reading.Reader;
import org.apache.commons.lang3.StringUtils;
import org.apache.hc.client5.http.classic.methods.HttpGet;
import org.apache.hc.client5.http.config.RequestConfig;
import org.apache.hc.client5.http.impl.classic.CloseableHttpClient;
import org.apache.hc.core5.http.ClassicHttpResponse;
import org.apache.hc.core5.http.HttpEntity;
import org.apache.hc.core5.http.ParseException;
import org.apache.hc.core5.http.io.entity.EntityUtils;
import org.apache.hc.core5.io.CloseMode;
import org.xml.sax.SAXException;

import org.ametys.core.util.HttpUtils;

/**
 * Reader implementation that add header for Captchetat requests
 */
public class CaptchetatReader extends AbstractLogEnabled implements Reader, Serviceable, Initializable, Disposable
{
    /** The CaptchEtat helper */
    protected CaptchEtatHelper _captchEtatHelper;

    private OutputStream _os;

    private CloseableHttpClient _httpClient;
    private ClassicHttpResponse _response;


    public void service(ServiceManager manager) throws ServiceException
    {
        _captchEtatHelper = (CaptchEtatHelper) manager.lookup(CaptchEtatHelper.ROLE);
    }
    
    public void initialize() throws Exception
    {
        _httpClient = HttpUtils.createHttpClient(0, 20, false);
    }
    
    public void dispose()
    {
        _httpClient.close(CloseMode.GRACEFUL);
    }
    
    public void setup(SourceResolver resolver, Map objectModel, String src, Parameters par) throws ProcessingException, SAXException, IOException
    {

        Request request = ObjectModelHelper.getRequest(objectModel);
        
        String url = _captchEtatHelper.getEndpoint() + "simple-captcha-endpoint?" + StringUtils.defaultString(request.getQueryString());
        
        HttpGet httpGet = new HttpGet(url);
        RequestConfig config = RequestConfig.custom()
                .setRedirectsEnabled(false)
                .build();
        httpGet.setConfig(config);
        
        String token = _captchEtatHelper.getToken();
        httpGet.addHeader("Authorization", "Bearer " + token);

        _response = _httpClient.executeOpen(null, httpGet, null);
        switch (_response.getCode())
        {
            case 200:
                break;
            
            case 403:
                _response.close();
                throw new IllegalStateException("CaptchEtat refused the connection");
                
            case 500:
            default:
                try
                {
                    throw new IllegalStateException("CaptchEtat returned an error: " + EntityUtils.toString(_response.getEntity(), StandardCharsets.UTF_8));
                }
                catch (ParseException e)
                {
                    throw new IllegalStateException("CaptchEtat returned an error");
                }
                finally
                {
                    _response.close();
                }
        }
    }

    public void setOutputStream(OutputStream out) throws IOException
    {
        _os = out;
    }

    public String getMimeType()
    {
        HttpEntity entity = _response.getEntity();
        return entity != null ? entity.getContentType() : null;
    }

    public boolean shouldSetContentLength()
    {
        return false;
    }

    public void generate() throws IOException, SAXException, ProcessingException
    {
        try
        {
            _response.getEntity().writeTo(_os);
        }
        finally
        {
            _response.close();
        }
    }

    public long getLastModified()
    {
        return 0;
    }
}
