/*
 *  Copyright 2013 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.cart;

import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.List;

import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.core.user.UserIdentity;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Implementation of a {@link CartElement} wrapping a {@link Content}
 *
 */
public class ContentElement implements CartElement
{
    private Content _content;
    private ContentTypesHelper _cTypesHelper;
    private ContentTypeExtensionPoint _cTypesEP;
    
    /**
     * Constructor
     * @param content The content
     * @param cTypesHelper The content types helper
     * @param cTypesEP The extension point for content types
     */
    public ContentElement (Content content, ContentTypesHelper cTypesHelper, ContentTypeExtensionPoint cTypesEP)
    {
        _content = content;
        _cTypesHelper = cTypesHelper;
        _cTypesEP = cTypesEP;
    }
    
    @Override
    public String getId()
    {
        return _content.getId();
    }

    @Override
    public I18nizableText getTitle()
    {
        return new I18nizableText(_content.getTitle());
    }
    
    @Override
    public I18nizableText getDescription()
    {
        return new I18nizableText("");
    }
    
    @Override
    public List<I18nizableText> getGroups ()
    {
        List<I18nizableText> cTypesLabel = new ArrayList<>();
        
        for (String cTypeId : _content.getTypes())
        {
            cTypesLabel.add(_cTypesEP.getExtension(cTypeId).getLabel());
        }
        return cTypesLabel;
    }
    
    @Override
    public ZonedDateTime getCreationDate()
    {
        return _content.getCreationDate();
    }
    
    @Override
    public UserIdentity getCreator()
    {
        return _content.getCreator();
    }
    
    @Override
    public ZonedDateTime getLastModified()
    {
        return _content.getLastModified();
    }

    @Override
    public UserIdentity getLastContributor()
    {
        return _content.getLastContributor();
    }

    @Override
    public String getType()
    {
        return "content";
    }
    
    @Override
    public String getGlyphIcon()
    {
        return _cTypesHelper.getIconGlyph(_content);
    }
    
    @Override
    public String getSmallIcon()
    {
        return _cTypesHelper.getSmallIcon(_content);
    }

    @Override
    public String getMediumIcon()
    {
        return _cTypesHelper.getMediumIcon(_content);
    }

    /**
     * Get the content
     * @return The content
     */
    public Content getContent()
    {
        return _content;
    }
}
