/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

Ametys.namespace('Cart', (function() {
	
	//imports
	const Arrays = Java.type('java.util.Arrays');
	const Collectors = Java.type('java.util.stream.Collectors');
	const CollectionIterable = Java.type('org.ametys.plugins.repository.CollectionIterable');
		
	//lookup
	const cartsDAO = Ametys.serviceManager.lookup('org.ametys.plugins.cart.CartsDAO');
	
	function getContents(cartId) 
	{
		const cart = cartsDAO.getCart(cartId);
		const contentList = cart.getContentCartElements()
					.stream()
					.map(ce => ce.getContent())
	                .collect(Collectors.toList());
		return new CollectionIterable(contentList);
    }

    function getQueries(cartId) 
	{
		const cart = cartsDAO.getCart(cartId);
		return cart.getQueryCartElements()
					.stream()
					.map(re => re.getId())
	                .collect(Collectors.toList());
    }

    function getResources(cartId) 
	{
		const cart = cartsDAO.getCart(cartId);
		const contentList = cart.getResourceCartElements()
					.stream()
					.map(re => re.getResource())
	                .collect(Collectors.toList());
		return new CollectionIterable(contentList);
    }

    function query(cartId, queryId) 
	{
		const jsonUtils = Ametys.serviceManager.lookup('org.ametys.core.util.JSONUtils')
		const contentSearcherFactory = Ametys.serviceManager.lookup('org.ametys.cms.search.content.ContentSearcherFactory')
		const getQueryFromJSONHelper = Ametys.serviceManager.lookup('org.ametys.cms.search.GetQueryFromJSONHelper')
		
		const cart = cartsDAO.getCart(cartId);
		if (cart == null)
		{
			throw new Error(`No cart found with id "${cartId}"`)
		}
	
		var content = null;
		
	  	cart.getQueryCartElements().forEach(function(tempQuery) 
		{
			if(tempQuery.getId() == queryId)
			{
				content = jsonUtils.convertJsonToMap(tempQuery.getDescription().toString());
			}
		});
		if (content == null)
		{
			throw new Error(`No query found with id "${queryId}"`)
		}
	    switch (content.get("toolId"))
	    {
	        case "uitool-solrsearch":
				return Solr.query(content.get("toolParams").get("values").get("query"));
	        case "uitool-web-search":
				const model = getQueryFromJSONHelper.getSearchUIModel(content.get("exportParams"));
				
				const sortJson = jsonUtils.convertObjectToJson(content.get("toolParams").get("sort"))
				const sort = getQueryFromJSONHelper.getSort(sortJson, "");
				
				const values = content.get("exportParams").get("values")
				const facetValues = content.get("exportParams").get("facetValues")
				const contextualParameters = content.get("exportParams").get("contextualParameters") 
				const type = content.get("toolParams").get("searchMode") || "simple"
	            const results = contentSearcherFactory.create(model).withSearchMode(type)
	                                                             .withSort(sort)
	                                                             .withLimits(0, 50)
	                                                             .searchWithFacets(values, facetValues, contextualParameters);
				return results.getObjects();
	        case "uitool-script":
	          	throw new Error("Forbidden to execute a script. Only simple, advanced and solr search are allowed.");
			default:
	          	throw new Error("Unknow query type '" + type + "'. Unable to execute this query.");
        }
	}
	
	return {
		getContents,
		getQueries,
		getResources,
		query
	}
})());