/*
 *  Copyright 2013 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.cart.generators;

import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.components.source.impl.SitemapSource;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.SourceResolver;
import org.xml.sax.SAXException;

import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.DateUtils;
import org.ametys.core.util.IgnoreRootHandler;
import org.ametys.core.util.JSONUtils;
import org.ametys.core.util.URIUtils;
import org.ametys.plugins.cart.Cart;
import org.ametys.plugins.cart.CartElement;
import org.ametys.plugins.cart.CartsDAO;
import org.ametys.plugins.cart.ContentElement;
import org.ametys.plugins.cart.QueryElement;
import org.ametys.plugins.cart.QueryFromDirectoryElement;
import org.ametys.plugins.cart.ResourceElement;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * SAX elements of a cart
 */
public class CartElementDetailsGenerator extends ServiceableGenerator
{
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The current user provider */
    protected CurrentUserProvider _userProvider;
    /** The Json utils */
    private JSONUtils _jsonUtils;
    /** The source resolver */
    private SourceResolver _sourceResolver;
    /** The user manager */
    private UserManager _userManager;
    /** The carts DAO */
    private CartsDAO _cartsDAO;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _jsonUtils = (JSONUtils) manager.lookup(JSONUtils.ROLE);
        _sourceResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _userManager = (UserManager) smanager.lookup(UserManager.ROLE);
        _userProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
        _cartsDAO = (CartsDAO) smanager.lookup(CartsDAO.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {

        Request request = ObjectModelHelper.getRequest(objectModel);
        String id = request.getParameter("cartId");
        
        Cart cart = _resolver.resolveById(id);
        UserIdentity user = _userProvider.getUser();
        
        contentHandler.startDocument();
        
        if (_cartsDAO.canRead(user, cart))
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", cart.getId());
            
            XMLUtils.startElement(contentHandler, "cart", attrs);
            
            XMLUtils.createElement(contentHandler, "label", cart.getTitle());
            XMLUtils.createElement(contentHandler, "description", cart.getDescription());
            
            try
            {
                List<QueryElement> queries = cart.getQueryCartElements();
                _saxQueries(queries);
            }
            catch (Exception e)
            {
                throw new SAXException("Unable to sax queries", e);
            }
            
            try
            {
                List<QueryFromDirectoryElement> queries = cart.getQueryFromDirectoryCartElements();
                _saxQueries(queries);
            }
            catch (Exception e)
            {
                throw new SAXException("Unable to sax queries", e);
            }
            
            try
            {
                List<ContentElement> contents = cart.getContentCartElements();
                _saxContents(contents);
            }
            catch (Exception e)
            {
                throw new SAXException("Unable to sax contents", e);
            }
            
            try
            {
                List<ResourceElement> resources = cart.getResourceCartElements();
                _saxResources(resources);
            }
            catch (Exception e)
            {
                throw new SAXException("Unable to sax resources", e);
            }
                    
            XMLUtils.endElement(contentHandler, "cart");
        }
        else
        {
            XMLUtils.createElement(contentHandler, "not-allowed");
        }
        
        contentHandler.endDocument();
    }
    
    /**
     * SAX resources of the cart
     * @param resources The resources of the cart
     * @throws IOException if an error occurred
     * @throws SAXException if an error occurred while saxing
     */
    protected void _saxResources (List<ResourceElement> resources) throws IOException, SAXException
    {
        XMLUtils.startElement(contentHandler, "resources");
        for (ResourceElement resource : resources)
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", resource.getId());
            XMLUtils.startElement(contentHandler, "resource", attrs);
            
            _saxCartElement(resource, resource.getGroups().get(0));
            
            XMLUtils.endElement(contentHandler, "resource");
        }
        XMLUtils.endElement(contentHandler, "resources");
    }
    
    /**
     * SAX the attributes of a cart element
     * @param cartElement The element to sax
     * @param group The group to sax with
     * @throws SAXException if an error occurred while saxing
     */
    protected void _saxCartElement(CartElement cartElement, I18nizableText group) throws SAXException
    {
        cartElement.getTitle().toSAX(contentHandler, "title");
        
        XMLUtils.createElement(contentHandler, "creation", DateUtils.zonedDateTimeToString(cartElement.getCreationDate()));
        
        UserIdentity creatorIdentity = cartElement.getCreator();
        User creator = _userManager.getUser(creatorIdentity);
        XMLUtils.createElement(contentHandler, "creator", creator != null ? creator.getFullName() : StringUtils.EMPTY);

        XMLUtils.createElement(contentHandler, "lastModification", DateUtils.zonedDateTimeToString(cartElement.getLastModified()));
        
        UserIdentity lastContributorIdentity = cartElement.getLastContributor();
        User lastContributor = _userManager.getUser(lastContributorIdentity.getPopulationId(), lastContributorIdentity.getLogin());
        
        XMLUtils.createElement(contentHandler, "lastContributor", lastContributor != null ? lastContributor.getFullName() : StringUtils.EMPTY);
        
        group.toSAX(contentHandler, "group");
    }
    
    /**
     * SAX contents of the cart
     * @param contents The contents of the cart
     * @throws IOException if an error occurred
     * @throws SAXException if an error occurred while saxing
     */
    protected void _saxContents (List<ContentElement> contents) throws IOException, SAXException
    {
        SitemapSource sitemapSource = null;
        try
        {
            XMLUtils.startElement(contentHandler, "contents");
            for (ContentElement content : contents)
            {
                List<I18nizableText> groups = content.getGroups();
                for (I18nizableText group : groups)
                {
                    XMLUtils.startElement(contentHandler, "content");
                    _saxCartElement(content, group);
                    
                    String cocoonUrl = _contentToCocoonUrl(content);
                    sitemapSource = (SitemapSource) _sourceResolver.resolveURI(cocoonUrl);
                    sitemapSource.toSAX(new IgnoreRootHandler(contentHandler));
                    XMLUtils.endElement(contentHandler, "content");
                }
               
            }
            XMLUtils.endElement(contentHandler, "contents");
        }
        finally
        {
            resolver.release(sitemapSource);
        }
    }
    
    /**
     * Returns the Cocoon pipeline URL that returns content details
     * @param content The content
     * @return The cocoon URL that returns the content details
     */
    protected String _contentToCocoonUrl(ContentElement content)
    {
        return "cocoon://_content.html?contentId=" + URIUtils.encodeParameter(content.getId());
    }
    
    /**
     * SAX queries of the cart
     * @param queries The queries of the cart
     * @throws IOException if an error occurred
     * @throws SAXException if an error occurred while saxing
     */
    protected void _saxQueries (List<? extends QueryElement> queries) throws IOException, SAXException
    {
        SitemapSource sitemapSource = null;
        try
        {
            XMLUtils.startElement(contentHandler, "queries");
            for (QueryElement query : queries)
            {
                XMLUtils.startElement(contentHandler, "query");
                _saxCartElement(query, query.getGroups().get(0));
                
                String queryContent = query.getQueryContent();
                Map<String, Object> queryDetails = _jsonUtils.convertJsonToMap(queryContent);
                
                String printPluginUrl = queryDetails.containsKey("printUrlPlugin") ? (String) queryDetails.get("printUrlPlugin") : "cms";
                String printUrl = queryDetails.containsKey("printUrl") ? (String) queryDetails.get("printUrl") : "search/print.html";
                @SuppressWarnings("unchecked")
                Map<String, Object> exportParams = (Map<String, Object>) queryDetails.get("exportParams");
                Map<String, String> params = new HashMap<>();
                params.put("parameters", _jsonUtils.convertObjectToJson(exportParams));
                
                sitemapSource = (SitemapSource) _sourceResolver.resolveURI("cocoon://_plugins/" + printPluginUrl + "/" + printUrl + "?parameters=" + _jsonUtils.convertObjectToJson(exportParams), null, new HashMap<>());
                sitemapSource.toSAX(new IgnoreRootHandler(contentHandler));
                XMLUtils.endElement(contentHandler, "query");
            }
            XMLUtils.endElement(contentHandler, "queries");
        }
        finally
        {
            resolver.release(sitemapSource);
        }
    }
        
}
