/*
 *  Copyright 2013 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.cart;

import java.time.ZonedDateTime;
import java.util.Collections;
import java.util.List;

import org.ametys.core.user.UserIdentity;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Implementation of a {@link CartElement} wrapping a search query
 *
 */
public class QueryElement implements CartElement
{
    private String _id;
    private String _query;
    private String _title;
    private UserIdentity _author;
    private ZonedDateTime _creationDate;
    private UserIdentity _contributor;
    private ZonedDateTime _lastModified;

    /**
     * Constructor
     * @param id The id
     * @param query The query
     * @param author The author
     * @param creationDate The creation date
     * @param contributor The last contributor
     * @param lastModified The last modification date
     * @param title The query title
     */
    public QueryElement (String id, String query, UserIdentity author, ZonedDateTime creationDate, UserIdentity contributor, ZonedDateTime lastModified, String title)
    {
        _id = id;
        _query = query;
        _author = author;
        _creationDate = creationDate;
        _contributor = contributor;
        _lastModified = lastModified;
        _title = title;
    }
    
    @Override
    public String getId()
    {
        return _id;
    }
    
    @Override
    public I18nizableText getTitle()
    {
        return new I18nizableText(_title);
    }

    @Override
    public I18nizableText getDescription()
    {
        return new I18nizableText("");
    }
    
    /**
     * Get the content of the query
     * @return the content as a JSON string
     */
    public String getQueryContent()
    {
        return _query;
    }
    
    @Override
    public List<I18nizableText> getGroups()
    {
        return Collections.singletonList(new I18nizableText("plugin.cart", "PLUGINS_CART_UITOOL_CART_QUERIES_GROUP"));
    }
    
    @Override
    public ZonedDateTime getCreationDate()
    {
        return _creationDate;
    }
    
    @Override
    public UserIdentity getCreator()
    {
        return _author;
    }
    
    @Override
    public ZonedDateTime getLastModified()
    {
        return _lastModified;
    }

    @Override
    public UserIdentity getLastContributor()
    {
        return _contributor;
    }
    
    @Override
    public String getType()
    {
        return "cartQuery";
    }

    public String getGlyphIcon()
    {
        return "ametysicon-magnifier12";
    }
    
    @Override
    public String getSmallIcon()
    {
        return null;
    }
    
    @Override
    public String getMediumIcon()
    {
        return null;
    }
}
