/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.cart.actions;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.lang3.ArrayUtils;

import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.core.cocoon.JSonReader;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.cart.Cart;
import org.ametys.plugins.cart.CartElement;
import org.ametys.plugins.cart.CartsDAO;
import org.ametys.plugins.cart.ContentElement;
import org.ametys.plugins.cart.QueryElement;
import org.ametys.plugins.cart.QueryFromDirectoryElement;
import org.ametys.plugins.core.user.UserHelper;
import org.ametys.plugins.queriesdirectory.QueryDAO;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.version.VersionAwareAmetysObject;
import org.ametys.plugins.workflow.store.AmetysStep;
import org.ametys.plugins.workflow.support.WorkflowHelper;
import org.ametys.plugins.workflow.support.WorkflowProvider;
import org.ametys.plugins.workflow.support.WorkflowProvider.AmetysObjectWorkflow;
import org.ametys.runtime.i18n.I18nizableText;

import com.opensymphony.workflow.WorkflowException;
import com.opensymphony.workflow.spi.Step;

/**
 * SAX a {@link Cart}
 *
 */
public class GetCartElementsAction extends ServiceableAction
{
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The workflow */
    protected WorkflowProvider _workflowProvider;
    /** The workflow helper */
    protected WorkflowHelper _workflowHelper;
    /** The current user provider */
    protected CurrentUserProvider _userProvider;
    /** The user manager */
    protected UserManager _userManager;
    /** The user helper */
    protected UserHelper _userHelper;
    /** The carts DAO */
    protected CartsDAO _cartsDAO;
    /** The query DAO */
    protected QueryDAO _queryDAO;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _workflowProvider = (WorkflowProvider) serviceManager.lookup(WorkflowProvider.ROLE);
        _workflowHelper = (WorkflowHelper) serviceManager.lookup(WorkflowHelper.ROLE);
        _userManager = (UserManager) serviceManager.lookup(UserManager.ROLE);
        _userProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
        _userHelper = (UserHelper) serviceManager.lookup(UserHelper.ROLE);
        _cartsDAO = (CartsDAO) serviceManager.lookup(CartsDAO.ROLE);
        _queryDAO = (QueryDAO) serviceManager.lookup(QueryDAO.ROLE);
    }
    
    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        @SuppressWarnings("unchecked")
        Map jsParameters = (Map<String, Object>) objectModel.get(ObjectModelHelper.PARENT_CONTEXT);
        
        String id = (String) jsParameters.get("cartId");
        
        Map<String, Object> result = new HashMap<>();
        List<Map<String, Object>> nodes = new ArrayList<>();
        int total = 0;
        
        try
        {
            Cart cart = _resolver.resolveById(id);
            
            UserIdentity user = _userProvider.getUser();
            if (_cartsDAO.canRead(user, cart))
            {
                result.put("label", cart.getTitle());
                result.put("description", cart.getDescription());
                
                List<CartElement> elements = cart.getElements();
                for (CartElement elmt : elements)
                {
                    boolean isQueryFromDirectory = elmt instanceof QueryFromDirectoryElement;
                    if (!isQueryFromDirectory || _queryDAO.canRead(user, ((QueryFromDirectoryElement) elmt).getQuery()))
                    {
                        nodes.addAll(elementToJSON(elmt));
                    }
                }
                total = elements.size();
            }
            
            result.put("cartElements", nodes);
            result.put("total", total);
        }
        catch (UnknownAmetysObjectException e)
        {
            result.put("unknown-cart", true);
        }
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        request.setAttribute(JSonReader.OBJECT_TO_READ, result);
        
        return EMPTY_MAP;
    }
    
    /**
     * Gets a cart element to JSON format
     * @param elmt The elemetnt
     * @return The element's properties
     */
    protected List<Map<String, Object>> elementToJSON (CartElement elmt)
    {
        List<Map<String, Object>> elmts = new ArrayList<>();
        
        HashMap<String, Object> infos = new HashMap<>();
        
        infos.put("elmtId", elmt.getId());
        infos.put("title", elmt.getTitle());
        infos.put("description", elmt.getDescription());
        
        infos.put("creation", DateUtils.zonedDateTimeToString(elmt.getCreationDate()));
        infos.put("creator", _userHelper.user2json(elmt.getCreator(), true));
        
        infos.put("lastModification", DateUtils.zonedDateTimeToString(elmt.getLastModified()));
        infos.put("lastContributor", _userHelper.user2json(elmt.getLastContributor(), true));
        
        infos.put("iconGlyph", elmt.getGlyphIcon());
        infos.put("smallIcon", elmt.getSmallIcon());
        infos.put("mediumIcon", elmt.getMediumIcon());
        
        infos.put("type", elmt.getType());
        
        if (elmt instanceof ContentElement)
        {
            Content content = _resolver.resolveById(elmt.getId());
            infos.put("workflow-step", workflowStepToJSON(content));
        }
        else if (elmt instanceof QueryElement queryElmt)
        {
            infos.put("queryContent", queryElmt.getQueryContent());
            if (queryElmt instanceof QueryFromDirectoryElement queryDirectoryElmt)
            {
                infos.put("documentation", queryDirectoryElmt.getQuery().getDocumentation());
                infos.put("queryType", queryDirectoryElmt.getQuery().getType());
            }
        }
        
        List<I18nizableText> groups = elmt.getGroups();
        for (I18nizableText group : groups)
        {
            @SuppressWarnings("unchecked")
            HashMap<String, Object> clonedInfos = (HashMap<String, Object>) infos.clone();
            clonedInfos.put("group", group);
            elmts.add(clonedInfos);
        }
        
        return elmts;
    }
    
    /**
     * Gets the workflow step if the content is a <code>WorkflowAwareContent</code>
     * @param content The content
     * @return The workflow step's properties
     */
    protected Map<String, Object> workflowStepToJSON (Content content)
    {
        Map<String, Object> infos = new HashMap<>();
        
        if (content instanceof WorkflowAwareContent)
        {
            WorkflowAwareContent waContent = (WorkflowAwareContent) content;
            
            try
            {
                AmetysObjectWorkflow workflow = _workflowProvider.getAmetysObjectWorkflow(waContent);
                long workflowId = waContent.getWorkflowId();
                String workflowName = workflow.getWorkflowName(workflowId);
                
                Step currentStep = _getCurrentStep(waContent, workflow);
                
                int currentStepId = currentStep.getStepId();
                
                I18nizableText workflowStepName = new I18nizableText("application",  _workflowHelper.getStepName(workflowName, currentStepId));
                
                infos.put("id", String.valueOf(currentStepId));
                infos.put("name", workflowStepName);
                if ("application".equals(workflowStepName.getCatalogue()))
                {
                    infos.put("icon-small", "/plugins/cms/resources_workflow/" + workflowStepName.getKey() + "-small.png");
                    infos.put("icon-medium", "/plugins/cms/resources_workflow/" + workflowStepName.getKey() + "-medium.png");
                    infos.put("icon-large", "/plugins/cms/resources_workflow/" + workflowStepName.getKey() + "-large.png");
                }
                else
                {
                    String pluginName = workflowStepName.getCatalogue().substring("plugin.".length());
                    infos.put("icon-small", "/plugins/" + pluginName + "/resources/img/workflow/" + workflowStepName.getKey() + "-small.png");
                    infos.put("icon-medium", "/plugins/" + pluginName + "/resources/img/workflow/" + workflowStepName.getKey() + "-medium.png");
                    infos.put("icon-large", "/plugins/" + pluginName + "/resources/img/workflow/" + workflowStepName.getKey() + "-large.png");
                }
            }
            catch (AmetysRepositoryException e)
            {
                // Current step id was not positioned
            }
            catch (WorkflowException e)
            {
                // Ignore, just don't SAX the workflow step.
            }
        }
        
        return infos;
    }
    
    /**
     * Get a content's step, wherever it works on the base version or not.
     * @param content the content.
     * @param workflow the workflow instance bound to this content
     * @return the content's workflow step.
     * @throws WorkflowException if an error occurs.
     */
    protected Step _getCurrentStep(WorkflowAwareContent content, AmetysObjectWorkflow workflow) throws WorkflowException
    {
        long workflowId = content.getWorkflowId();
        Step currentStep = (Step) workflow.getCurrentSteps(workflowId).get(0);
        
        if (content instanceof VersionAwareAmetysObject)
        {
            VersionAwareAmetysObject vaContent = (VersionAwareAmetysObject) content;
            String currentRevision = vaContent.getRevision();
            
            if (currentRevision != null)
            {
                
                String[] allRevisions = vaContent.getAllRevisions();
                int currentRevIndex = ArrayUtils.indexOf(allRevisions, currentRevision);
                
                if (currentRevIndex > -1 && currentRevIndex < (allRevisions.length - 1))
                {
                    String nextRevision = allRevisions[currentRevIndex + 1];
                    
                    Date currentRevTimestamp = vaContent.getRevisionTimestamp();
                    Date nextRevTimestamp = vaContent.getRevisionTimestamp(nextRevision);
                    
                    // Get all steps between the two revisions.
                    List<Step> steps = _workflowHelper.getStepsBetween(workflow, workflowId, currentRevTimestamp, nextRevTimestamp);
                    
                    // In the old workflow structure
                    // We take the second, which is current revision's last step.
                    if (steps.size() > 0 && steps.get(0) instanceof AmetysStep)
                    {
                        AmetysStep amStep = (AmetysStep) steps.get(0);
                        if (amStep.getProperty("actionFinishDate") != null)
                        {
                            // New workflow structure detected: cut the first workflow step
                            // in the list, as it belongs to the next version.
                            steps = steps.subList(1, steps.size());
                        }
                    }
                    
                    // Order by step descendant.
                    Collections.sort(steps, new Comparator<Step>()
                    {
                        public int compare(Step step1, Step step2)
                        {
                            return -Long.valueOf(step1.getId()).compareTo(step2.getId());
                        }
                    });
                    
                    // The first step in the list is the current version's last workflow step.
                    if (steps.size() > 0)
                    {
                        currentStep = steps.get(0);
                    }
                }
            }
        }
        return currentStep;
    }

}
