/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

// Upgrade carts to kernel rights: read old attributes to grant rights for each cart.

const GroupIdentity = Java.type('org.ametys.core.group.GroupIdentity');
const UserIdentity = Java.type('org.ametys.core.user.UserIdentity');


const VISIBILITY = "ametys-internal:visibility";

const PROFILE_READ_ACCESS = "ametys-internal:read-access";

const PROFILE_WRITE_ACCESS = "ametys-internal:write-access";
    
const USERS = "ametys-internal:users";

const GROUPS = "ametys-internal:groups";

const TITLE = "ametys-internal:label";
const TYPE = "ametys-internal:type";
const DESCRIPTION = "ametys-internal:description";
const DOCUMENTATION = "ametys-internal:documentation";
const CONTENTS = "ametys-internal:contents";
const RESOURCES = "ametys-internal:resources";
const QUERIES = "ametys-internal:queries";
const QUERY_ID_PROPERTY = "ametys-internal:id";
const QUERY_DESCRIPTION_PROPERTY = "ametys-internal:description";
const QUERY_AUTHOR_PROPERTY = "ametys-internal:author";
const QUERY_TITLE_PROPERTY = "ametys-internal:title";
const QUERY_DATE_PROPERTY = "ametys-internal:date";



const AUTHOR = "ametys-internal:author";
const LASTMODIFICATIONDATE = "ametys-internal:lastModificationDate";

const READER = "READER";

const CARTMIGRATIONPROFILELABEL = "Cart migration";
const CARTMIGRATIONPROFILEID = "cart-migration";
        
const _profileAssignmentStorageExtensionPoint = Ametys.serviceManager.lookup("org.ametys.core.right.ProfileAssignmentStorageExtensionPoint");

const _rightProfileDAO = Ametys.serviceManager.lookup("org.ametys.core.right.RightProfilesDAO");

//Replace cart node types to read old attributes
Repository.registerNodeType("carts");
Repository.registerNodeType("cart");

let cartProfile = _rightProfileDAO.getProfile(CARTMIGRATIONPROFILEID);

if (cartProfile == null)
{
    cartProfile = _rightProfileDAO.addProfile(CARTMIGRATIONPROFILELABEL);
}
  
_rightProfileDAO.updateRights(cartProfile, ["Cart_Rights_Admin"]);

Repository.query("//element(*, ametys:cart)").forEach(function(cart)
{
    let node = cart.getNode();
    
    if (node.hasNode(AUTHOR))
    {
        var authorNode = node.getNode(AUTHOR);
        var author = new UserIdentity(authorNode.getProperty("ametys:login").getString(), authorNode.getProperty("ametys:population").getString());
        cart.setAuthor(author);
        authorNode.remove();
    }

    if (node.hasProperty(TITLE))
    {
        cart.setTitle(node.getProperty(TITLE).getValue().getString());
        node.getProperty(TITLE).remove();
    }
    
    if (node.hasProperty(DESCRIPTION))
    {
        cart.setDescription(node.getProperty(DESCRIPTION).getValue().getString());
        node.getProperty(DESCRIPTION).remove();
    }
    
    if (node.hasProperty(CONTENTS))
    {
        let contents = node.getProperty(CONTENTS).getValues();
        for (const content in contents) 
        {
            cart.addContent(contents[content].getString())
        }
    }
    
    if (node.hasProperty(RESOURCES))
    {
        let resources = node.getProperty(RESOURCES).getValues();
        for (const resource in resources) 
        {
            cart.addResource(resources[resource].getString())
        }
    }
    
    if (node.hasNode(QUERIES))
    {
        let queries = node.getNode(QUERIES);
        let iterator = queries.getNodes();
        while(iterator.hasNext())
        {
            let queryNode = iterator.nextNode();
            let id = queryNode.getProperty(QUERY_ID_PROPERTY).getString();
            let authorLogin = queryNode.getNode(QUERY_AUTHOR_PROPERTY).getProperty("ametys:login").getString();
            let authorPopulation = queryNode.getNode(QUERY_AUTHOR_PROPERTY).getProperty("ametys:population").getString();
            let author = new UserIdentity(authorLogin, authorPopulation);
            let title = queryNode.getProperty(QUERY_TITLE_PROPERTY).getString();
            let query = queryNode.getProperty(QUERY_DESCRIPTION_PROPERTY).getString();
            let creationDate = queryNode.getProperty(QUERY_DATE_PROPERTY).getValue().getDate().getTime();
            cart.addQuery(id, author, title, query, creationDate);
        }
    }
    
    if (cart.getNode().hasProperty(VISIBILITY))
    {
        var visibility = cart.getNode().getProperty(VISIBILITY).getValue().getString();
        
        if (visibility == "PUBLIC")
        {
            // grant access to reader profile to every connected users
            _profileAssignmentStorageExtensionPoint.removeDeniedProfileFromAnyConnectedUser(READER, cart);
            _profileAssignmentStorageExtensionPoint.allowProfileToAnyConnectedUser(READER, cart);
            
            _profileAssignmentStorageExtensionPoint.removeDeniedProfileFromAnyConnectedUser(CARTMIGRATIONPROFILEID, cart);
            _profileAssignmentStorageExtensionPoint.allowProfileToAnyConnectedUser(CARTMIGRATIONPROFILEID, cart);
        }
        else if (visibility == "SHARED")
        {
            if (cart.getNode().hasNode(PROFILE_READ_ACCESS))
            {
                var profileNode = cart.getNode().getNode(PROFILE_READ_ACCESS);
                
                if (profileNode.hasNode(USERS))
                {
                    var usersIterator = profileNode.getNodes(USERS);
                    
                    // grant access to reader profile to each user that had reader right
                    while (usersIterator.hasNext())
                    {
                        var user = usersIterator.next();
                        
                        var userIdentity = new UserIdentity(user.getProperty("ametys:login").getString(), user.getProperty("ametys:population").getString());
                        
                        _profileAssignmentStorageExtensionPoint.removeDeniedProfileFromUser(userIdentity, READER, cart);
                        _profileAssignmentStorageExtensionPoint.allowProfileToUser(userIdentity, READER, cart);
                    }
                    
                    var groupsIterator = profileNode.getNodes(GROUPS);
                    
                    // grant access to reader profile to each group that had reader right
                    while (groupsIterator.hasNext())
                    {
                        var group = groupsIterator.next();
                        var groupIdentity = new GroupIdentity(group.getProperty("ametys:groupId").getString(), group.getProperty("ametys:groupDirectory").getString());
                        
                        _profileAssignmentStorageExtensionPoint.removeDeniedProfileFromGroup(groupIdentity, READER, cart);
                        _profileAssignmentStorageExtensionPoint.allowProfileToGroup(groupIdentity, READER, cart);
                    }
                }
            }
            if (cart.getNode().hasNode(PROFILE_WRITE_ACCESS))
            {
                var profileNode = cart.getNode().getNode(PROFILE_WRITE_ACCESS);
                
                if (profileNode.hasNode(USERS))
                {
                    var usersIterator = profileNode.getNodes(USERS);
                    
                    // grant access to reader and cart-migration profile to each user that had read/write right
                    while (usersIterator.hasNext())
                    {
                        var user = usersIterator.next();
                        
                        var userIdentity = new UserIdentity(user.getProperty("ametys:login").getString(), user.getProperty("ametys:population").getString());

                        _profileAssignmentStorageExtensionPoint.removeDeniedProfileFromUser(userIdentity, READER, cart);
                        _profileAssignmentStorageExtensionPoint.allowProfileToUser(userIdentity, READER, cart);
                        
                        _profileAssignmentStorageExtensionPoint.removeDeniedProfileFromUser(userIdentity, CARTMIGRATIONPROFILEID, cart);
                        _profileAssignmentStorageExtensionPoint.allowProfileToUser(userIdentity, CARTMIGRATIONPROFILEID, cart);
                    }
                    
                    var groupsIterator = profileNode.getNodes(GROUPS);
                    
                    // grant access to reader and cart-migration profile to each group that had read/write right
                    while (groupsIterator.hasNext())
                    {
                        var group = groupsIterator.next();
                        var groupIdentity = new GroupIdentity(group.getProperty("ametys:groupId").getString(), group.getProperty("ametys:groupDirectory").getString());

                        _profileAssignmentStorageExtensionPoint.removeDeniedProfileFromGroup(groupIdentity, READER, cart);
                        _profileAssignmentStorageExtensionPoint.allowProfileToGroup(groupIdentity, READER, cart);
                        
                        _profileAssignmentStorageExtensionPoint.removeDeniedProfileFromGroup(groupIdentity, CARTMIGRATIONPROFILEID, cart);
                        _profileAssignmentStorageExtensionPoint.allowProfileToGroup(groupIdentity, CARTMIGRATIONPROFILEID, cart);
                    }
                }
            }
            // grant acces to given user and group
        }
    }

    cart.saveChanges();
});
