/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Singleton class defining the file import function.
 * @private
 */
Ext.define('Ametys.plugins.contentio.ImportFromFile', {
    singleton: true,
    
    /**
     * @property {Ext.form.Panel} _formPanel The dialog box form panel.
     * @private
     */
    _formPanel: null,
    
    /**
     * @property {Boolean} _initialized True when the dialog is initialized.
     * @private
     */
    _initialized: false,
    
    /**
     * @property {Ametys.window.DialogBox} _box The action dialog box.
     * @private
     */
    _box: null,
    
    /**
     * Called when the button is pressed
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller for the ui debug button
     * @param {Boolean} state When the button is a toggle button, the new press-state of the button, null otherwise
     */
    act: function(controller, state)
    {
        if (!this._delayedInitialize())
        {
            return;
        }
        
        this._box.show();        
    },
    
    /**
     * Create the query dialog box.
     * @private
     */
    _delayedInitialize: function()
    {
        if (this._initialized)
        {
            return true;
        }
        
        this._formPanel = this._createFormPanel();
        
        this._box = Ext.create('Ametys.window.DialogBox', {
            title: "{{i18n PLUGINS_CONTENTIO_IMPORTFILE_DIALOG_TITLE}}",
            iconCls: 'ametysicon-documents12',
            
            layout: 'fit',
            width: 400,
            
            items: [this._formPanel],
            
            closeAction: 'hide',
            buttons: [{
                text: "{{i18n PLUGINS_CONTENTIO_IMPORTFILE_DIALOG_OK}}",
                handler: this._ok,
                scope: this
            }, {
                text: "{{i18n PLUGINS_CONTENTIO_IMPORTFILE_DIALOG_CANCEL}}",
                handler: this._cancel,
                scope: this
            }]
        });
        
        this._initialized = true;
        
        return true;
    },
    
    /**
     * Create the upload dialog's form panel.
     * @return {String} The upload dialog's form panel.
     * @private
     */
    _createFormPanel: function()
    {
        return Ext.create('Ext.form.Panel', {
            bodyStyle: 'padding:5px',
            
            defaultType: 'textfield',
            fieldDefaults: {
                labelAlign: 'right',
                labelWidth: 60,
                labelSeparator: '',
                msgTarget: 'side'
            },
            
            border: false,
            scollable: true,
            items: [{
                xtype: 'fileuploadfield',
                id: 'form-file',
                name: 'file',
                fieldLabel: "{{i18n PLUGINS_CONTENTIO_IMPORTFILE_DIALOG_FILEFIELD_LABEL}}",
                emptyText: "{{i18n PLUGINS_CONTENTIO_IMPORTFILE_DIALOG_FILEFIELD_EMPTY}}",
                buttonText: "{{i18n PLUGINS_CONTENTIO_IMPORTFILE_DIALOG_FILEFIELD_BUTTON}}",
                allowBlank: false,
                width: 350,
                listeners: {
                    change: {fn: this._onFileChange, scope: this}
                }
            }]
        });
    },
    
    /**
     * Handle the OK button by submitting the form.
     * @private
     */
    _ok: function()
    {
        var form = this._formPanel.getForm();
        
        if (!form.isValid())
        {
            return;
        }
        
        form.submit({
            url : Ametys.getPluginDirectPrefix('contentio') + '/import-file',
            params: {
//                name: name
            },
            
            waitMsg: "{{i18n PLUGINS_CONTENTIO_IMPORTFILE_DIALOG_WAIT_MSG}}",
            waitTitle: "{{i18n PLUGINS_CONTENTIO_IMPORTFILE_DIALOG_WAIT_TITLE}}",
            
            success: this._success,
            failure: this._failure,
            scope: this
        });
    },
    
    /**
     * Callback fired when the property has been successfully added.
     * @param {Ext.form.Basic} form The form that requested the action.
     * @param {Ext.form.action.Action} action The Action class.
     * @private
     */
    _success: function(form, action)
    {
        this._box.hide();
        
        var importedCount = action.result ? action.result.importedCount : '0';
        
        if (importedCount > 0)
        {
            var importedIds = action.result ? action.result.importedIds : '';
            Ext.create('Ametys.message.Message', {
                type: Ametys.message.Message.CREATED,
                targets: {
                    type: 'content',
                    parameters: { ids: importedIds.split('|') }
                }
            });
        }
        
        Ametys.Msg.show({
            title: "{{i18n PLUGINS_CONTENTIO_IMPORTFILE_SUCCESS_TITLE}}",
            msg: "{{i18n PLUGINS_CONTENTIO_IMPORTFILE_SUCCESS_TEXT_1}}" + importedCount + "{{i18n PLUGINS_CONTENTIO_IMPORTFILE_SUCCESS_TEXT_2}}",
            buttons: Ext.Msg.OK,
            icon: Ext.Msg.INFO
        });
        
        if (typeof this._callbackFn == 'function')
        {
            this._callbackFn(this._node);
        }
    },
    
    /**
     * Callback fired when the property creation has failed.
     * @param {Ext.form.Basic} form The form that requested the action.
     * @param {Ext.form.action.Action} action The Action class.
     * @private
     */
    _failure: function(form, action)
    {
        var error = action.result ? action.result.error : '';
        
        if (error == 'no-importer')
        {
            Ametys.log.ErrorDialog.display({
                title: "{{i18n PLUGINS_CONTENTIO_IMPORTFILE_ERROR_TITLE}}",
                text: "{{i18n PLUGINS_CONTENTIO_IMPORTFILE_ERROR_NOIMPORTER}}",
                details: action.result ? action.result.message : ''
            });
        }
        else
        {
            Ametys.log.ErrorDialog.display({
                title: "{{i18n PLUGINS_CONTENTIO_IMPORTFILE_ERROR_TITLE}}",
                text: "{{i18n PLUGINS_CONTENTIO_IMPORTFILE_ERROR_TEXT}}",
                details: action.result ? action.result.message : ''
            });
        }
    },
    
    /**
     * Handle the cancel button by hiding the dialog box.
     * @private
     */
    _cancel: function()
    {
        this._box.hide();
    }
    
});
