/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.archive;

import java.io.IOException;
import java.nio.file.Path;
import java.util.Collection;
import java.util.List;
import java.util.zip.ZipOutputStream;

import org.ametys.runtime.i18n.I18nizableText;

/**
 * Component responsible to archive data.
 */
public interface Archiver
{
    /**
     * Get the {@link ManifestReaderWriter}
     * @return the {@link ManifestReaderWriter}
     */
    public ManifestReaderWriter getManifestReaderWriter();
    
    /**
     * Export data into the destination archive
     * @param zos the output data stream.
     * @throws IOException if an error occurs while writing entries to the archive.
     */
    public void export(ZipOutputStream zos) throws IOException;

    /**
     * Returns the partial imports, among the given ones, that this {@link Archiver} is able to manage.
     * @param partialImports The partial imports to filter
     * @return The managed partial imports
     */
    public Collection<String> managedPartialImports(Collection<String> partialImports);
    
    /**
     * Import data (limited to the given partial imports) from the source archive
     * @param zipPath the input zip path
     * @param partialImports The partial imports to do. The passed collection must already be filtered out by {@link #managedPartialImports}
     * @param merger The {@link Merger}
     * @param deleteBefore <code>true</code> if objects need to be deleted before applying the partial imports
     * @return The {@link ImportReport}
     * @throws IOException if an error occurs while reading archive entries.
     */
    public ImportReport partialImport(Path zipPath, Collection<String> partialImports, Merger merger, boolean deleteBefore) throws IOException;
    
    /**
     * Potential additional information to append to the success mail of the import process
     * @return the mail additional information
     */
    public default List<I18nizableText> additionalSuccessImportMail()
    {
        return List.of(
            new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_ARCHIVE_IMPORT_SCHEDULABLE_MAIL_SUCCESS_BODY_END")
        );
    }
}
