/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.archive;

import java.io.IOException;
import java.nio.file.Path;
import java.util.Collection;
import java.util.Collections;
import java.util.Optional;
import java.util.stream.Stream;
import java.util.zip.ZipOutputStream;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.RepositoryException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.slf4j.Logger;

import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.collection.AmetysObjectCollection;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Archives contents in /ametys:root/ametys:contents
 */
public class ContentArchiver extends AbstractLogEnabled implements Archiver, Serviceable
{
    /** Archiver id. */
    public static final String ID = "contents";
    
    private AmetysObjectResolver _resolver;
    private ContentsArchiverHelper _contentsArchiverHelper;
    
    private ManifestReaderWriter _manifestReaderWriter = new ContentArchiverManifestReaderWriter(getLogger());
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _contentsArchiverHelper = (ContentsArchiverHelper) manager.lookup(ContentsArchiverHelper.ROLE);
    }
    
    @Override
    public ManifestReaderWriter getManifestReaderWriter()
    {
        return _manifestReaderWriter;
    }
    
    @Override
    public void export(ZipOutputStream zos) throws IOException
    {
        AmetysObjectCollection rootContents = _rootContents();
        _contentsArchiverHelper.exportContents("contents/", rootContents, zos);
    }
    
    private AmetysObjectCollection _rootContents()
    {
        return _resolver.resolveByPath("/ametys:contents");
    }
    
    private void _deleteBeforePartialImport(AmetysObjectCollection rootContents) throws IOException
    {
        try
        {
            Node rootContentsNode = rootContents.getNode();
            NodeIterator rootChildren = rootContentsNode.getNodes();
            while (rootChildren.hasNext())
            {
                rootChildren.nextNode().remove();
            }
            rootContentsNode.getSession().save();
        }
        catch (RepositoryException e)
        {
            throw new IOException(e);
        }
    }
    
    @Override
    public Collection<String> managedPartialImports(Collection<String> partialImports)
    {
        return partialImports.contains(ID)
                ? Collections.singletonList(ID)
                : Collections.EMPTY_LIST;
    }
    
    @Override
    public ImportReport partialImport(Path zipPath, Collection<String> partialImports, Merger merger, boolean deleteBefore) throws IOException
    {
        AmetysObjectCollection rootContents = _rootContents();
        if (deleteBefore)
        {
            _deleteBeforePartialImport(rootContents);
        }
        return _contentsArchiverHelper.importContents("contents/", rootContents, zipPath, merger, Collections.EMPTY_SET);
    }
    
    private static final class ContentArchiverManifestReaderWriter implements ManifestReaderWriter
    {
        private static final String __DATA = "contents";
        
        private Logger _logger;
        
        public ContentArchiverManifestReaderWriter(Logger logger)
        {
            _logger = logger;
        }
        
        @Override
        public Object getData()
        {
            return __DATA;
        }
        
        @Override
        public Stream<PartialImport> toPartialImports(Object data)
        {
            return Optional.ofNullable(data)
                    .filter(String.class::isInstance)
                    .map(String.class::cast)
                    .filter(__DATA::equals)
                    .map(Stream::of)
                    .orElseGet(() ->
                    {
                        _logger.warn("Unexpected manifest data '{}', we would expect '{}', the ZIP archive probably comes from a different version of Ametys.", data, __DATA);
                        return Stream.empty();
                    })
                    .map(contentsStr -> PartialImport.of(contentsStr, new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_ARCHIVE_IMPORT_CONTENT_ARCHIVER_OPTION_ALL")));
        }
    }
}
