/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.archive;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import org.ametys.core.schedule.Runnable;
import org.ametys.core.user.UserIdentity;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * A {@link Runnable} which schedules a {@link ImportArchiveSchedulable} for importing data from an archive.
 */
public class ImportArchiveRunnable implements Runnable
{
    private static final String __ID_PREFIX = ImportArchiveRunnable.class.getName();
    
    private String _id;
    private String _archiveName;
    private Optional<Collection<String>> _elements;
    private MergePolicy _mergePolicy;
    private UserIdentity _userIdentity;
    
    /**
     * Constructor
     * @param archiveName The name of the archive to import
     * @param elements The elements to import
     * @param mergePolicy The {@link MergePolicy}
     * @param user The user identity
     */
    public ImportArchiveRunnable(String archiveName, Optional<Collection<String>> elements, MergePolicy mergePolicy, UserIdentity user)
    {
        _setId();
        _archiveName = archiveName;
        _elements = elements;
        _mergePolicy = mergePolicy;
        _userIdentity = user;
    }
    
    private void _setId()
    {
        _id = __ID_PREFIX + "$" + System.currentTimeMillis();
    }
    
    @Override
    public String getId()
    {
        return _id;
    }

    @Override
    public I18nizableText getLabel()
    {
        return new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_ARCHIVE_IMPORT_RUNNABLE_LABEL");
    }

    @Override
    public I18nizableText getDescription()
    {
        return new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_ARCHIVE_IMPORT_RUNNABLE_DESC");
    }

    @Override
    public FireProcess getFireProcess()
    {
        return FireProcess.NOW;
    }

    @Override
    public String getCronExpression()
    {
        return null;
    }

    @Override
    public String getSchedulableId()
    {
        return ImportArchiveSchedulable.ID;
    }

    @Override
    public boolean isRemovable()
    {
        return false;
    }

    @Override
    public boolean isModifiable()
    {
        return false;
    }

    @Override
    public boolean isDeactivatable()
    {
        return false;
    }

    @Override
    public MisfirePolicy getMisfirePolicy()
    {
        return null;
    }

    @Override
    public boolean isVolatile()
    {
        return false;
    }

    @Override
    public Map<String, Object> getParameterValues()
    {
        Map<String, Object> values = new HashMap<>();
        values.put(ImportArchiveSchedulable.ARCHIVE_KEY, _archiveName);
        values.put(ImportArchiveSchedulable.MERGE_POLICY_KEY, _mergePolicy);
        _elements.ifPresent(els -> values.put(ImportArchiveSchedulable.ELEMENTS_KEY, els));
        return values;
    }

    @Override
    public UserIdentity getUserIdentity()
    {
        return _userIdentity;
    }

}
