/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.archive;

import java.util.LinkedHashMap;
import java.util.Map;

import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.Enumerator;
import org.ametys.runtime.model.StaticEnumerator;

/**
 * The policy to apply when trying to import an object with same identifier as an existing object.
 * <br>This is the view of the different policies of merge available to the application, for the business logic, see {@link Merger} (and its implementations in {@link Mergers}).
 */
public enum MergePolicy
{
    /** Delete all existing objects before the import process */
    DELETE_BEFORE
    {
        @Override
        public Merger getMerger()
        {
            return Mergers.DELETE_BEFORE;
        }
    },
    
    /** Ignore the objects that already exist */
    IGNORE
    {
        @Override
        public Merger getMerger()
        {
            return Mergers.IGNORE;
        }
    },
    /** Replace the objects that already exist */
    REPLACE
    {
        @Override
        public Merger getMerger()
        {
            return Mergers.REPLACE;
        }
    },
    
    /** Fail when an object try to be imported but already exist */
    FAIL
    {
        @Override
        public Merger getMerger()
        {
            return Mergers.FAIL;
        }
    };
    
    static Map<String, MergePolicy> _ids = new LinkedHashMap<>();
    static
    {
        for (MergePolicy mergePolicy : MergePolicy.values())
        {
            _ids.put(mergePolicy._id, mergePolicy);
        }
    }
    
    private String _id;
    private I18nizableText _label;
    private I18nizableText _description;
    
    private MergePolicy()
    {
        final String catalogue = "plugin.contentio";
        String label = String.format("PLUGINS_CONTENTIO_ARCHIVE_IMPORT_SCHEDULABLE_PARAM_MERGE_POLICY_OPTION_%s", name());
        _label = new I18nizableText(catalogue, label);
        String desc = String.format("PLUGINS_CONTENTIO_ARCHIVE_IMPORT_SCHEDULABLE_PARAM_MERGE_POLICY_OPTION_%s_DESC", name());
        _description = new I18nizableText(catalogue, desc);
        _setId();
    }
    
    private void _setId()
    {
        _id = name()
                .toLowerCase()
                .replace('_', '-');
    }
    
    /**
     * Gets the {@link Merger}
     * @return the {@link Merger}
     */
    public abstract Merger getMerger();
    
    static MergePolicy forId(String id)
    {
        return _ids.computeIfAbsent(id, MergePolicy::_throwIllegalId);
    }
    
    private static MergePolicy _throwIllegalId(String id)
    {
        throw new IllegalArgumentException(String.format("Wrong MergePolicy id: '%s'", id));
    }
    
    /**
     * Gets the id
     * @return the id
     */
    public String getId()
    {
        return _id;
    }
    
    Map<String, Object> toJson()
    {
        return Map.of(
                "value", _id, 
                "label", _label, 
                "desc", _description);
    }
    
    /**
     * {@link Enumerator} for listing merge policies for import.
     */
    public static class MergePolicyEnumerator extends StaticEnumerator<String>
    {
        /**
         * Constructor
         */
        public MergePolicyEnumerator()
        {
            super();
            MergePolicy._ids.forEach((id, mergePolicy) -> add(mergePolicy._label, id));
        }
    }
}
