/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.archive;

import java.io.IOException;
import java.io.InputStream;

import javax.jcr.ImportUUIDBehavior;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

/**
 * A merger, which is responsible to handle UUID collision when encountered.
 */
public interface Merger
{
    /**
     * Returns <code>true</code> if all existing objects concerned by the {@link Archiver} must be deleted before {@link Archiver#partialImport importing}
     * @return <code>true</code> if existing objects must be deleted
     */
    default boolean deleteBeforePartialImport()
    {
        return false;
    }
    
    /**
     * Returns <code>true</code> if the object with the given id needs to be merged
     * @param id The id
     * @return <code>true</code> if the object with the given id needs to be merged
     */
    boolean needsMerge(String id);
    
    /**
     * The result of {@link Merger#merge}.
     * <br>It tells whether to continue or stop the process of importing the given id.
     */
    public static enum AfterMerge
    {
        /** Stop importing of the object after the merge was processed. */
        STOP_PROCESS,
        /** Continue importing the object after the merge was processed. */
        CONTINUE_PROCESS,
    }
    
    /**
     * Merges the existing object with given id
     * @param id The id
     * @return What the caller <b>must</b> do after this method is finished.
     * @throws MergeException If the merge cannot be done. This exception must propagate and stop the global import process.
     */
    AfterMerge merge(String id) throws MergeException;
    
    /**
     * Gets the UUID Behavior for JCR import
     * <br>See {@link ImportUUIDBehavior} and {@link Session#importXML(String, java.io.InputStream, int)}
     * @return the UUID Behavior
     */
    int getImportUuidBehavior();
    
    /**
     * Does a JCR import from the XML read from the given {@link InputStream}
     * <br>The save operation must be made by the caller.
     * <br>The {@link InputStream} must be closed by the caller.
     * <br>Default implementation uses {@link Session#importXML}
     * @param session The JCR {@link Session}
     * @param parentAbsPath the absolute path of the node below which the deserialized subgraph is added.
     * @param in The {@link InputStream} from which the XML to be deserialized is read.
     * @throws RepositoryException if another error occurs.
     * @throws IOException if an error during an I/O operation occurs.
     */
    default void jcrImportXml(Session session, String parentAbsPath, InputStream in) throws RepositoryException, IOException
    {
        session.importXML(parentAbsPath, in, getImportUuidBehavior());
    }
}
