/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.archive;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.LocalDate;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import javax.jcr.Binary;
import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.sax.TransformerHandler;
import javax.xml.transform.stream.StreamResult;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.Constants;
import org.apache.cocoon.environment.Context;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.jackrabbit.JcrConstants;
import org.apache.xpath.XPathAPI;
import org.slf4j.Logger;
import org.w3c.dom.Document;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.contentio.archive.Archivers.AmetysObjectNotImportedException;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.explorer.resources.ResourceCollection;
import org.ametys.plugins.explorer.resources.jcr.JCRResource;
import org.ametys.plugins.explorer.resources.jcr.JCRResourceFactory;
import org.ametys.plugins.explorer.resources.jcr.JCRResourcesCollection;
import org.ametys.plugins.explorer.resources.jcr.JCRResourcesCollectionFactory;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectFactoryExtensionPoint;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.dublincore.DublinCoreAwareAmetysObject;
import org.ametys.plugins.repository.dublincore.ModifiableDublinCoreAwareAmetysObject;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Export a resources collection as individual files.
 */
public class ResourcesArchiverHelper extends AbstractLogEnabled implements Component, Serviceable, Contextualizable
{
    /** Avalon role. */
    public static final String ROLE = ResourcesArchiverHelper.class.getName();
    
    private static final String __PROPERTIES_METADATA_XML_FILE_NAME_SUFFIX = "properties.xml";
    private static final String __DC_METADATA_XML_FILE_NAME_SUFFIX = "dc.xml";
    
    private static final String __DC_METADATA_XML_EXPORT_TAG_NAME_ROOT = "dublin-core-metadata";
    private static final String __DC_METADATA_XML_EXPORT_TAG_NAME_TITLE = "title";
    private static final String __DC_METADATA_XML_EXPORT_TAG_NAME_CREATOR = "creator";
    private static final String __DC_METADATA_XML_EXPORT_TAG_NAME_SUBJECT = "subject";
    private static final String __DC_METADATA_XML_EXPORT_TAG_NAME_DESCRIPTION = "description";
    private static final String __DC_METADATA_XML_EXPORT_TAG_NAME_PUBLISHER = "publisher";
    private static final String __DC_METADATA_XML_EXPORT_TAG_NAME_CONTRIBUTOR = "contributor";
    private static final String __DC_METADATA_XML_EXPORT_TAG_NAME_DATE = "date";
    private static final String __DC_METADATA_XML_EXPORT_TAG_NAME_TYPE = "type";
    private static final String __DC_METADATA_XML_EXPORT_TAG_NAME_FORMAT = "format";
    private static final String __DC_METADATA_XML_EXPORT_TAG_NAME_IDENTIFIER = "identifier";
    private static final String __DC_METADATA_XML_EXPORT_TAG_NAME_SOURCE = "source";
    private static final String __DC_METADATA_XML_EXPORT_TAG_NAME_LANGUAGE = "language";
    private static final String __DC_METADATA_XML_EXPORT_TAG_NAME_RELATION = "relation";
    private static final String __DC_METADATA_XML_EXPORT_TAG_NAME_COVERAGE = "coverage";
    private static final String __DC_METADATA_XML_EXPORT_TAG_NAME_RIGHTS = "rights";
    
    private JCRResourcesCollectionFactory _jcrResourcesCollectionFactory;
    private JCRResourceFactory _jcrResourceFactory;
    
    private Context _cocoonContext;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        AmetysObjectFactoryExtensionPoint ametysObjectFactoryEP = (AmetysObjectFactoryExtensionPoint) manager.lookup(AmetysObjectFactoryExtensionPoint.ROLE);
        _jcrResourcesCollectionFactory = (JCRResourcesCollectionFactory) ametysObjectFactoryEP.getExtension(JCRResourcesCollectionFactory.class.getName());
        _jcrResourceFactory = (JCRResourceFactory) ametysObjectFactoryEP.getExtension(JCRResourceFactory.class.getName());
    }
    
    @Override
    public void contextualize(org.apache.avalon.framework.context.Context context) throws ContextException
    {
        _cocoonContext = (Context) context.get(Constants.CONTEXT_ENVIRONMENT_CONTEXT);
    }
    
    /**
     * Exports a {@link ResourceCollection} as folders and files inside the ZIP archive.
     * @param collection the root {@link ResourceCollection}
     * @param zos the ZIP OutputStream.
     * @param prefix the prefix for the ZIP archive.
     * @throws IOException if an error occurs while archiving
     */
    public void exportCollection(ResourceCollection collection, ZipOutputStream zos, String prefix) throws IOException
    {
        if (collection == null)
        {
            return;
        }
        
        zos.putNextEntry(new ZipEntry(StringUtils.appendIfMissing(prefix, "/"))); // even if there is no child, at least export the collection folder
        
        try (AmetysObjectIterable<AmetysObject> objects = collection.getChildren();)
        {
            for (AmetysObject object : objects)
            {
                _exportChild(object, zos, prefix);
            }
        }
        
        try
        {
            // process metadata for the collection
            _exportCollectionMetadataEntry(collection, zos, prefix);
            // process ACL for the collection
            Node collectionNode = ((JCRAmetysObject) collection).getNode();
            Archivers.exportAcl(collectionNode, zos, ArchiveHandler.METADATA_PREFIX + prefix + "acl.xml");
        }
        catch (RepositoryException e)
        {
            throw new RuntimeException("Unable to SAX some information for collection '" + collection.getPath() + "' for archiving", e);
        }
    }
    
    private void _exportCollectionMetadataEntry(ResourceCollection collection, ZipOutputStream zos, String prefix) throws IOException
    {
        String metadataFullPrefix = ArchiveHandler.METADATA_PREFIX + prefix;
        ZipEntry contentEntry = new ZipEntry(metadataFullPrefix + __PROPERTIES_METADATA_XML_FILE_NAME_SUFFIX);
        zos.putNextEntry(contentEntry);
        
        try
        {
            TransformerHandler contentHandler = Archivers.newTransformerHandler();
            contentHandler.setResult(new StreamResult(zos));
            
            contentHandler.startDocument();
            _saxSystemMetadata(collection, contentHandler);
            contentHandler.endDocument();
        }
        catch (SAXException | TransformerConfigurationException e)
        {
            throw new RuntimeException("Unable to SAX properties for collection '" + collection.getPath() + "' for archiving", e);
        }
    }
    
    private void _exportChild(AmetysObject child, ZipOutputStream zos, String prefix) throws IOException
    {
        if (child instanceof ResourceCollection)
        {
            String newPrefix = prefix + child.getName() + "/";
            exportCollection((ResourceCollection) child, zos, newPrefix);
        }
        else if (child instanceof Resource)
        {
            exportResource((Resource) child, zos, prefix);
        }
    }
    
    /**
     * Exports a {@link Resource} as  file inside the ZIP archive.
     * @param resource the {@link Resource}.
     * @param zos the ZIP OutputStream.
     * @param prefix the prefix for the ZIP archive.
     * @throws IOException if an error occurs while archiving
     */
    public void exportResource(Resource resource, ZipOutputStream zos, String prefix) throws IOException
    {
        ZipEntry resourceEntry = new ZipEntry(prefix + resource.getName());
        zos.putNextEntry(resourceEntry);

        try (InputStream is = resource.getInputStream())
        {
            IOUtils.copy(is, zos);
        }
        
        // dublin core properties
        _exportResourceMetadataEntry(resource, zos, prefix, __DC_METADATA_XML_FILE_NAME_SUFFIX, this::_saxDublinCoreMetadata, "Dublin Core metadata");
        
        // other properties (id, creator...)
        _exportResourceMetadataEntry(resource, zos, prefix, __PROPERTIES_METADATA_XML_FILE_NAME_SUFFIX, this::_saxSystemMetadata, "properties");
    }
    
    private void _exportResourceMetadataEntry(Resource resource, ZipOutputStream zos, String prefix, String metadataFileNameSuffix, ResourceMetadataSaxer metadataSaxer, String debugName) throws IOException
    {
        String metadataFullPrefix = ArchiveHandler.METADATA_PREFIX + prefix + resource.getName() + "_";
        ZipEntry contentEntry = new ZipEntry(metadataFullPrefix + metadataFileNameSuffix);
        zos.putNextEntry(contentEntry);
        
        try
        {
            TransformerHandler contentHandler = Archivers.newTransformerHandler();
            contentHandler.setResult(new StreamResult(zos));
            
            contentHandler.startDocument();
            metadataSaxer.sax(resource, contentHandler);
            contentHandler.endDocument();
        }
        catch (SAXException | TransformerConfigurationException e)
        {
            throw new RuntimeException("Unable to SAX " + debugName + " for resource '" + resource.getPath() + "' for archiving", e);
        }
    }
    
    @FunctionalInterface
    private static interface ResourceMetadataSaxer
    {
        void sax(Resource resource, ContentHandler contentHandler) throws SAXException;
    }
    
    private void _saxDublinCoreMetadata(DublinCoreAwareAmetysObject dcObject, ContentHandler contentHandler) throws SAXException
    {
        XMLUtils.startElement(contentHandler, __DC_METADATA_XML_EXPORT_TAG_NAME_ROOT);
        _saxIfNotNull(__DC_METADATA_XML_EXPORT_TAG_NAME_TITLE, dcObject.getDCTitle(), contentHandler);
        _saxIfNotNull(__DC_METADATA_XML_EXPORT_TAG_NAME_CREATOR, dcObject.getDCCreator(), contentHandler);
        _saxIfNotNull(__DC_METADATA_XML_EXPORT_TAG_NAME_SUBJECT, dcObject.getDCSubject(), contentHandler);
        _saxIfNotNull(__DC_METADATA_XML_EXPORT_TAG_NAME_DESCRIPTION, dcObject.getDCDescription(), contentHandler);
        _saxIfNotNull(__DC_METADATA_XML_EXPORT_TAG_NAME_PUBLISHER, dcObject.getDCPublisher(), contentHandler);
        _saxIfNotNull(__DC_METADATA_XML_EXPORT_TAG_NAME_CONTRIBUTOR, dcObject.getDCContributor(), contentHandler);
        _saxLocalDateIfNotNull(__DC_METADATA_XML_EXPORT_TAG_NAME_DATE, dcObject.getDCDate(), contentHandler);
        _saxIfNotNull(__DC_METADATA_XML_EXPORT_TAG_NAME_TYPE, dcObject.getDCType(), contentHandler);
        _saxIfNotNull(__DC_METADATA_XML_EXPORT_TAG_NAME_FORMAT, dcObject.getDCFormat(), contentHandler);
        _saxIfNotNull(__DC_METADATA_XML_EXPORT_TAG_NAME_IDENTIFIER, dcObject.getDCIdentifier(), contentHandler);
        _saxIfNotNull(__DC_METADATA_XML_EXPORT_TAG_NAME_SOURCE, dcObject.getDCSource(), contentHandler);
        _saxIfNotNull(__DC_METADATA_XML_EXPORT_TAG_NAME_LANGUAGE, dcObject.getDCLanguage(), contentHandler);
        _saxIfNotNull(__DC_METADATA_XML_EXPORT_TAG_NAME_RELATION, dcObject.getDCRelation(), contentHandler);
        _saxIfNotNull(__DC_METADATA_XML_EXPORT_TAG_NAME_COVERAGE, dcObject.getDCCoverage(), contentHandler);
        _saxIfNotNull(__DC_METADATA_XML_EXPORT_TAG_NAME_RIGHTS, dcObject.getDCRights(), contentHandler);
        XMLUtils.endElement(contentHandler, __DC_METADATA_XML_EXPORT_TAG_NAME_ROOT);
    }
    
    private void _saxSystemMetadata(Resource resource, ContentHandler contentHandler) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "resource");
        _saxIfNotNull("id", resource.getId(), contentHandler);
        _saxIfNotNull("name", resource.getName(), contentHandler);
        _saxIfNotNull("creator", resource.getCreator(), contentHandler);
        _saxZonedDateTimeIfNotNull("creationDate", resource.getCreationDate(), contentHandler);
        _saxIfNotNull("contributor", resource.getLastContributor(), contentHandler);
        _saxZonedDateTimeIfNotNull("lastModified", resource.getLastModified(), contentHandler);
        XMLUtils.endElement(contentHandler, "resource");
    }
    
    private void _saxSystemMetadata(ResourceCollection collection, ContentHandler contentHandler) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "resource-collection");
        _saxIfNotNull("id", collection.getId(), contentHandler);
        _saxIfNotNull("name", collection.getName(), contentHandler);
        XMLUtils.endElement(contentHandler, "resource-collection");
    }

    private void _saxIfNotNull(String name, String value, ContentHandler contentHandler) throws SAXException
    {
        if (value != null)
        {
            XMLUtils.createElement(contentHandler, name, value);
        }
    }
    
    private void _saxIfNotNull(String name, UserIdentity value, ContentHandler contentHandler) throws SAXException
    {
        if (value != null)
        {
            XMLUtils.createElement(contentHandler, name, UserIdentity.userIdentityToString(value));
        }
    }
    
    private void _saxIfNotNull(String name, String[] values, ContentHandler contentHandler) throws SAXException
    {
        if (values != null)
        {
            for (String value : values)
            {
                XMLUtils.createElement(contentHandler, name, value);
            }
        }
    }
    
    private void _saxLocalDateIfNotNull(String name, Date value, ContentHandler contentHandler) throws SAXException
    {
        if (value != null)
        {
            LocalDate ld = DateUtils.asLocalDate(value);
            XMLUtils.createElement(contentHandler, name, ld.format(DateTimeFormatter.ISO_LOCAL_DATE));
        }
    }
    
    private void _saxZonedDateTimeIfNotNull(String name, Date value, ContentHandler contentHandler) throws SAXException
    {
        if (value != null)
        {
            ZonedDateTime zdt = DateUtils.asZonedDateTime(value, ZoneId.systemDefault());
            XMLUtils.createElement(contentHandler, name, zdt.format(DateUtils.getISODateTimeFormatter()));
        }
    }
    
    /**
     * Imports folders and files from the given ZIP archive and path, under the given {@link ResourceCollection}
     * @param commonPrefix The common prefix in the ZIP archive
     * @param parentOfRootResources the parent of the root {@link ResourceCollection} (the root will also be created)
     * @param zipPath the input zip path
     * @param merger The {@link Merger}
     * @return The {@link ImportReport}
     * @throws IOException if an error occurs while importing archive
     */
    public ImportReport importCollection(String commonPrefix, Node parentOfRootResources, Path zipPath, Merger merger) throws IOException
    {
        Importer importer;
        List<JCRResource> importedResources;
        try
        {
            importer = new Importer(commonPrefix, parentOfRootResources, zipPath, merger, getLogger());
            importer.importRoot();
            importedResources = importer.getImportedResource();
        }
        catch (ParserConfigurationException e)
        {
            throw new IOException(e);
        }
        _saveImported(parentOfRootResources);
        _checkoutImported(importedResources);
        return importer._report;
    }
    
    private void _saveImported(Node parentOfRoot)
    {
        try
        {
            Session session = parentOfRoot.getSession();
            if (session.hasPendingChanges())
            {
                getLogger().warn(Archivers.WARN_MESSAGE_ROOT_HAS_PENDING_CHANGES, parentOfRoot);
                session.save();
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to save changes", e);
        }
    }
    
    private void _checkoutImported(List<JCRResource> importedResources)
    {
        for (JCRResource resource : importedResources)
        {
            resource.checkpoint();
        }
    }
    
    private class Importer
    {
        final ImportReport _report = new ImportReport();
        private final String _commonPrefix;
        private final Node _parentOfRoot;
        private final Path _zipArchivePath;
        private final Merger _merger;
        private final Logger _logger;
        private final DocumentBuilder _builder;
        private final List<JCRResource> _importedResources = new ArrayList<>();
        private JCRResourcesCollection _root;
        private final UnitaryCollectionImporter _unitaryCollectionImporter = new UnitaryCollectionImporter();
        private final UnitaryResourceImporter _unitaryResourceImporter = new UnitaryResourceImporter();
        
        Importer(String commonPrefix, Node parentOfRoot, Path zipArchivePath, Merger merger, Logger logger) throws ParserConfigurationException
        {
            _commonPrefix = commonPrefix;
            _parentOfRoot = parentOfRoot;
            _zipArchivePath = zipArchivePath;
            _merger = merger;
            _logger = logger;
            _builder = DocumentBuilderFactory.newInstance()
                    .newDocumentBuilder();
        }
        
        void importRoot() throws IOException
        {
            if (ZipEntryHelper.zipEntryFolderExists(_zipArchivePath, _commonPrefix))
            {
                Path rootFolderToImport = ZipEntryHelper.zipFileRoot(_zipArchivePath)
                        .resolve(_commonPrefix);
                
                _importResourceCollectionAndChildren(rootFolderToImport);
            }
        }
        
        List<JCRResource> getImportedResource()
        {
            return _importedResources;
        }
        
        private void _createResourceCollectionAcl(Node collectionNode, String folderPath) throws IOException
        {
            String zipEntryPath = new StringBuilder()
                    .append(ArchiveHandler.METADATA_PREFIX)
                    .append(StringUtils.strip(folderPath, "/"))
                    .append("/acl.xml")
                    .toString();
            try
            {
                _logger.debug("Trying to import ACL node for ResourcesCollection '{}', from ACL XML file '{}', if it exists", collectionNode, zipEntryPath);
                Archivers.importAcl(collectionNode, _zipArchivePath, _merger, zipEntryPath, _logger);
            }
            catch (RepositoryException e)
            {
                throw new IOException(e);
            }
        }
        
        private void _importChildren(Path folder) throws IOException
        {
            String pathPrefix = _relativePath(folder);
            
            DirectoryStream<Path> childFiles = _getDirectFileChildren(pathPrefix);
            try (childFiles)
            {
                for (Path childFile : childFiles)
                {
                    _importResource(childFile)
                            .ifPresent(_importedResources::add);
                }
            }
            
            DirectoryStream<Path> childFolders = _getDirectFolderChildren(pathPrefix);
            try (childFolders)
            {
                for (Path childFolder : childFolders)
                {
                    _importResourceCollectionAndChildren(childFolder);
                }
            }
        }
        
        private DirectoryStream<Path> _getDirectFolderChildren(String pathPrefix) throws IOException
        {
            return ZipEntryHelper.children(
                _zipArchivePath,
                Optional.of(_commonPrefix + pathPrefix),
                p -> Files.isDirectory(p));
        }
        
        private DirectoryStream<Path> _getDirectFileChildren(String pathPrefix) throws IOException
        {
            return ZipEntryHelper.children(
                _zipArchivePath,
                Optional.of(_commonPrefix + pathPrefix),
                p -> !Files.isDirectory(p));
        }
        
        private void _importResourceCollectionAndChildren(Path folder) throws IOException
        {
            Optional<Node> optionalCollectionNode = _importResourceCollection(folder);
            if (optionalCollectionNode.isPresent())
            {
                Node collectionNode = optionalCollectionNode.get();
                _createResourceCollectionAcl(collectionNode, folder.toString());
                _importChildren(folder);
            }
        }
        
        private Optional<Node> _importResourceCollection(Path folder) throws ImportGlobalFailException
        {
            return _unitaryCollectionImporter.unitaryImport(_zipArchivePath, folder, _merger, _logger);
        }
        
        private Document _getFolderPropertiesXml(Path folder) throws IOException
        {
            String zipEntryPath = new StringBuilder()
                    .append(ArchiveHandler.METADATA_PREFIX)
                    .append(StringUtils.strip(folder.toString(), "/"))
                    .append("/")
                    .append(__PROPERTIES_METADATA_XML_FILE_NAME_SUFFIX)
                    .toString();
            try (InputStream stream = ZipEntryHelper.zipEntryFileInputStream(_zipArchivePath, zipEntryPath))
            {
                Document doc = _builder.parse(stream);
                return doc;
            }
            catch (SAXException e)
            {
                throw new IOException(e);
            }
        }
        
        private Node _createResourceCollection(Path folder, String id, Document propertiesXml) throws IOException, AmetysObjectNotImportedException, TransformerException
        {
            boolean isRoot = _isRoot(folder);
            Node parentNode = _retrieveParentJcrNode(folder, isRoot);
            String uuid = StringUtils.substringAfter(id, "://");
            String collectionName = Archivers.xpathEvalNonEmpty("resource-collection/name", propertiesXml);
            _logger.info("Creating a ResourcesCollection object for '{}' folder (id={})", folder, id);
            
            try
            {
                Node resourceCollection = _createChildResourceCollection(parentNode, uuid, collectionName);
                if (isRoot)
                {
                    _root = _jcrResourcesCollectionFactory.getAmetysObject(resourceCollection, null);
                }
                return resourceCollection;
            }
            catch (RepositoryException e)
            {
                throw new IOException(e);
            }
        }
        
        private boolean _isRoot(Path folder)
        {
            String folderPath = StringUtils.strip(folder.toString(), "/");
            String commonPrefixToCompare = StringUtils.strip(_commonPrefix, "/");
            return commonPrefixToCompare.equals(folderPath);
        }
        
        private Node _createChildResourceCollection(Node parentNode, String uuid, String collectionName) throws RepositoryException
        {
            // Create a Node with JCR primary type "ametys:resources-collection"
            // But then call 'replaceNodeWithDesiredUuid' to have it with the desired UUID (srcNode will be removed)
            Node srcNode = parentNode.addNode(collectionName, JCRResourcesCollectionFactory.RESOURCESCOLLECTION_NODETYPE);
            Node nodeWithDesiredUuid = Archivers.replaceNodeWithDesiredUuid(srcNode, uuid);
            return nodeWithDesiredUuid;
        }
        
        private String _relativePath(Path folderOrFile)
        {
            // for instance, _commonPrefix="resources/"
            // relPath=folderOrFile.toString()="/resources/foo/bar"
            // it should return "foo/bar"
            
            // for instance, _commonPrefix="resources/"
            // relPath=folderOrFile.toString()="/resources"
            // it should return ""
            
            String commonPrefixToRemove = "/" + StringUtils.strip(_commonPrefix, "/");
            String relPath = folderOrFile.toString();
            relPath = relPath.startsWith(commonPrefixToRemove)
                    ? StringUtils.substringAfter(relPath, commonPrefixToRemove)
                    : relPath;
            relPath = StringUtils.strip(relPath, "/");
            return relPath;
        }
        
        private Node _retrieveParentJcrNode(Path fileOrFolder, boolean isRoot)
        {
            if (isRoot)
            {
                // is the root, thus return the parent of the root
                return _parentOfRoot;
            }
            
            if (_root == null)
            {
                throw new IllegalStateException("Unexpected error, the root must have been created before.");
            }
            
            Path parent = fileOrFolder.getParent();
            String parentRelPath = _relativePath(parent);
            return parentRelPath.isEmpty()
                    ? _root.getNode()
                    : _jcrResourcesCollectionFactory.<JCRResourcesCollection>getChild(_root, parentRelPath).getNode();
        }
        
        private Optional<JCRResource> _importResource(Path file) throws ImportGlobalFailException
        {
            return _unitaryResourceImporter.unitaryImport(_zipArchivePath, file, _merger, _logger);
        }
        
        private Document _getFilePropertiesXml(Path file) throws IOException
        {
            String zipEntryPath = new StringBuilder()
                    .append(ArchiveHandler.METADATA_PREFIX)
                    .append(StringUtils.strip(file.toString(), "/"))
                    .append("_")
                    .append(__PROPERTIES_METADATA_XML_FILE_NAME_SUFFIX)
                    .toString();
            try (InputStream stream = ZipEntryHelper.zipEntryFileInputStream(_zipArchivePath, zipEntryPath))
            {
                Document doc = _builder.parse(stream);
                return doc;
            }
            catch (SAXException e)
            {
                throw new IOException(e);
            }
        }
        
        private JCRResource _createdResource(Path file, String id, Document propertiesXml) throws IOException, AmetysObjectNotImportedException
        {
            Node parentNode = _retrieveParentJcrNode(file, false);
            String uuid = StringUtils.substringAfter(id, "://");
            String resourceName = file.getFileName().toString();
            _logger.info("Creating a Resource object for '{}' file (id={})", file, id);
            
            try
            {
                Node resourceNode = _createChildResource(parentNode, uuid, resourceName);
                _setResourceData(resourceNode, file, propertiesXml);
                _setResourceProperties(resourceNode, propertiesXml);
                _setResourceMetadata(resourceNode, file);
                
                JCRResource createdResource = _resolveResource(resourceNode);
                return createdResource;
            }
            catch (TransformerException | RepositoryException e)
            {
                throw new IOException(e);
            }
        }
        
        private Node _createChildResource(Node parentNode, String uuid, String resourceName) throws RepositoryException
        {
            // Create a Node with JCR primary type "ametys:resource"
            // But then call 'replaceNodeWithDesiredUuid' to have it with the desired UUID (srcNode will be removed)
            Node srcNode = parentNode.addNode(resourceName, "ametys:resource");
            Node nodeWithDesiredUuid = Archivers.replaceNodeWithDesiredUuid(srcNode, uuid);
            return nodeWithDesiredUuid;
        }
        
        private JCRResource _resolveResource(Node resourceNode)
        {
            return _jcrResourceFactory.getAmetysObject(resourceNode, null);
        }
        
        private void _setResourceData(Node resourceNode, Path file, Document propertiesXml) throws RepositoryException, IOException, TransformerException
        {
            Node resourceContentNode = resourceNode.addNode(JcrConstants.JCR_CONTENT, JcrConstants.NT_RESOURCE);
            
            String mimeType = _getMimeType(file);
            resourceContentNode.setProperty(JcrConstants.JCR_MIMETYPE, mimeType);
            
            try (InputStream stream = ZipEntryHelper.zipEntryFileInputStream(_zipArchivePath, file.toString()))
            {
                Binary binary = resourceNode.getSession()
                        .getValueFactory()
                        .createBinary(stream);
                resourceContentNode.setProperty(JcrConstants.JCR_DATA, binary);
            }
            
            Date lastModified = Objects.requireNonNull(DomNodeHelper.nullableDatetimeValue(propertiesXml, "resource/lastModified"));
            Calendar lastModifiedCal = new GregorianCalendar();
            lastModifiedCal.setTime(lastModified);
            resourceContentNode.setProperty(JcrConstants.JCR_LASTMODIFIED, lastModifiedCal);
        }
        
        private void _setResourceProperties(Node resourceNode, Document propertiesXml) throws TransformerException, AmetysObjectNotImportedException, RepositoryException
        {
            UserIdentity contributor = UserIdentity.stringToUserIdentity(Archivers.xpathEvalNonEmpty("resource/contributor", propertiesXml));
            Node lastContributorNode = resourceNode.addNode(RepositoryConstants.NAMESPACE_PREFIX + ":" + JCRResource.CONTRIBUTOR_NODE_NAME, RepositoryConstants.USER_NODETYPE);
            lastContributorNode.setProperty(RepositoryConstants.NAMESPACE_PREFIX + ":login", contributor.getLogin());
            lastContributorNode.setProperty(RepositoryConstants.NAMESPACE_PREFIX + ":population", contributor.getPopulationId());
            
            UserIdentity creator = UserIdentity.stringToUserIdentity(Archivers.xpathEvalNonEmpty("resource/creator", propertiesXml));
            Node creatorNode = resourceNode.addNode(RepositoryConstants.NAMESPACE_PREFIX + ":" + JCRResource.CREATOR_NODE_NAME, RepositoryConstants.USER_NODETYPE);
            creatorNode.setProperty(RepositoryConstants.NAMESPACE_PREFIX + ":login", creator.getLogin());
            creatorNode.setProperty(RepositoryConstants.NAMESPACE_PREFIX + ":population", creator.getPopulationId());
            
            Date creationDate = Objects.requireNonNull(DomNodeHelper.nullableDatetimeValue(propertiesXml, "resource/creationDate"));
            Calendar creationDateCal = new GregorianCalendar();
            creationDateCal.setTime(creationDate);
            resourceNode.setProperty(RepositoryConstants.NAMESPACE_PREFIX + ":" + JCRResource.CREATION_DATE, creationDateCal);
        }
        
        private void _setResourceMetadata(Node resourceNode, Path file) throws IOException
        {
            ModifiableDublinCoreAwareAmetysObject dcObject = _jcrResourceFactory.getAmetysObject(resourceNode, null);
            _setDublinCoreMetadata(dcObject, file);
        }
        
        private void _setDublinCoreMetadata(ModifiableDublinCoreAwareAmetysObject dcObject, Path file) throws IOException
        {
            String zipEntryPath = new StringBuilder()
                    .append(ArchiveHandler.METADATA_PREFIX)
                    .append(StringUtils.strip(file.toString(), "/"))
                    .append("_")
                    .append(__DC_METADATA_XML_FILE_NAME_SUFFIX)
                    .toString();
            try (InputStream stream = ZipEntryHelper.zipEntryFileInputStream(_zipArchivePath, zipEntryPath))
            {
                Document doc = _builder.parse(stream);
                _setDublinCoreMetadata(dcObject, doc);
            }
            catch (SAXException | TransformerException e)
            {
                throw new IOException(e);
            }
        }
        
        private void _setDublinCoreMetadata(ModifiableDublinCoreAwareAmetysObject dcObject, Document doc) throws TransformerException
        {
            org.w3c.dom.Node dcNode = XPathAPI.selectSingleNode(doc, __DC_METADATA_XML_EXPORT_TAG_NAME_ROOT);
            dcObject.setDCTitle(DomNodeHelper.nullableStringValue(dcNode, __DC_METADATA_XML_EXPORT_TAG_NAME_TITLE));
            dcObject.setDCCreator(DomNodeHelper.nullableStringValue(dcNode, __DC_METADATA_XML_EXPORT_TAG_NAME_CREATOR));
            dcObject.setDCSubject(DomNodeHelper.stringValues(dcNode, __DC_METADATA_XML_EXPORT_TAG_NAME_SUBJECT));
            dcObject.setDCDescription(DomNodeHelper.nullableStringValue(dcNode, __DC_METADATA_XML_EXPORT_TAG_NAME_DESCRIPTION));
            dcObject.setDCPublisher(DomNodeHelper.nullableStringValue(dcNode, __DC_METADATA_XML_EXPORT_TAG_NAME_PUBLISHER));
            dcObject.setDCContributor(DomNodeHelper.nullableStringValue(dcNode, __DC_METADATA_XML_EXPORT_TAG_NAME_CONTRIBUTOR));
            dcObject.setDCDate(DomNodeHelper.nullableDateValue(dcNode, __DC_METADATA_XML_EXPORT_TAG_NAME_DATE));
            dcObject.setDCType(DomNodeHelper.nullableStringValue(dcNode, __DC_METADATA_XML_EXPORT_TAG_NAME_TYPE));
            dcObject.setDCFormat(DomNodeHelper.nullableStringValue(dcNode, __DC_METADATA_XML_EXPORT_TAG_NAME_FORMAT));
            dcObject.setDCIdentifier(DomNodeHelper.nullableStringValue(dcNode, __DC_METADATA_XML_EXPORT_TAG_NAME_IDENTIFIER));
            dcObject.setDCSource(DomNodeHelper.nullableStringValue(dcNode, __DC_METADATA_XML_EXPORT_TAG_NAME_SOURCE));
            dcObject.setDCLanguage(DomNodeHelper.nullableStringValue(dcNode, __DC_METADATA_XML_EXPORT_TAG_NAME_LANGUAGE));
            dcObject.setDCRelation(DomNodeHelper.nullableStringValue(dcNode, __DC_METADATA_XML_EXPORT_TAG_NAME_RELATION));
            dcObject.setDCCoverage(DomNodeHelper.nullableStringValue(dcNode, __DC_METADATA_XML_EXPORT_TAG_NAME_COVERAGE));
            dcObject.setDCRights(DomNodeHelper.nullableStringValue(dcNode, __DC_METADATA_XML_EXPORT_TAG_NAME_RIGHTS));
        }
        
        private String _getMimeType(Path file)
        {
            return Optional.of(file)
                    .map(Path::getFileName)
                    .map(Path::toString)
                    .map(String::toLowerCase)
                    .map(_cocoonContext::getMimeType)
                    .orElse("application/unknown");
        }
        
        private final class UnitaryCollectionImporter implements UnitaryImporter<Node>
        {
            @Override
            public String objectNameForLogs()
            {
                return "Resource collection";
            }

            @Override
            public Document getPropertiesXml(Path zipEntryPath) throws Exception
            {
                return _getFolderPropertiesXml(zipEntryPath);
            }

            @Override
            public String retrieveId(Document propertiesXml) throws Exception
            {
                return Archivers.xpathEvalNonEmpty("resource-collection/id", propertiesXml);
            }

            @Override
            public Node create(Path zipEntryPath, String id, Document propertiesXml) throws AmetysObjectNotImportedException, Exception
            {
                Node node = _createResourceCollection(zipEntryPath, id, propertiesXml);
                Archivers.unitarySave(node, _logger);
                return node;
            }
            
            @Override
            public ImportReport getReport()
            {
                return _report;
            }
        }
        
        private final class UnitaryResourceImporter implements UnitaryImporter<JCRResource>
        {
            @Override
            public String objectNameForLogs()
            {
                return "Resource";
            }

            @Override
            public Document getPropertiesXml(Path zipEntryPath) throws Exception
            {
                return _getFilePropertiesXml(zipEntryPath);
            }

            @Override
            public String retrieveId(Document propertiesXml) throws Exception
            {
                return Archivers.xpathEvalNonEmpty("resource/id", propertiesXml);
            }

            @Override
            public JCRResource create(Path zipEntryPath, String id, Document propertiesXml) throws AmetysObjectNotImportedException, Exception
            {
                JCRResource createdResource = _createdResource(zipEntryPath, id, propertiesXml);
                Archivers.unitarySave(createdResource.getNode(), _logger);
                return createdResource;
            }
            
            @Override
            public ImportReport getReport()
            {
                return _report;
            }
        }
    }
}
