/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.archive;

import java.nio.file.Path;
import java.util.Optional;

import org.slf4j.Logger;
import org.w3c.dom.Document;

import org.ametys.plugins.contentio.archive.Archivers.AmetysObjectNotImportedException;
import org.ametys.plugins.contentio.archive.ImportReport.ImportError;
import org.ametys.plugins.contentio.archive.Merger.AfterMerge;

/**
 * Unitary importer of an abstract object into Ametys.
 * @param <O> The type of object to import
 */
public interface UnitaryImporter<O>
{
    /**
     * The name for the object to import in error logs
     * @return The name for the object to import in error logs
     */
    String objectNameForLogs();
    
    /**
     * Gets the properties XML file from its ZIP path
     * @param zipEntryPath The ZIP entry path of the XML file
     * @return The properties XML file
     * @throws Exception if an error occurs
     */
    Document getPropertiesXml(Path zipEntryPath) throws Exception;
    
    /**
     * Retrieves the id of the object to import from the XML file
     * @param propertiesXml The properties XML file
     * @return The id of the object to import
     * @throws Exception if an error occurs
     */
    String retrieveId(Document propertiesXml) throws Exception;
    
    /**
     * Creates the object with the given id and XML properties file.
     * <br>Note that the implementation must create the object with the given id, but it can also create sub-objects for this unitary object (for instance a Resource as an attachment for a Content, etc.).
     * @param zipEntryPath The ZIP entry path of the XML file
     * @param id The id of the object to import
     * @param propertiesXml The properties XML file
     * @return The created object
     * @throws AmetysObjectNotImportedException If the object was not imported due to an exception (for instance an attribute is missing in the XML file). Then the object is ignored and the import will continue to the next object.
     * @throws ImportGlobalFailException If the global import process must stop due to any fatal exception.
     * @throws Exception if another unexpected error occurs
     */
    O create(Path zipEntryPath, String id, Document propertiesXml) throws AmetysObjectNotImportedException, ImportGlobalFailException, Exception;
    
    /**
     * The {@link ImportReport}
     * @return The {@link ImportReport}
     */
    ImportReport getReport();
    
    /**
     * The optional process to do in a `finally` block in {@link #unitaryImport}
     */
    default void unitaryImportFinalize()
    {
        // Do nothing by default
    }
    
    /**
     * Does the unitary import.
     * <br>Do not override the default behavior.
     * @param zipArchivePath The ZIP file
     * @param zipEntryPath The ZIP entry path of the XML file
     * @param merger The {@link Merger}
     * @param logger The logger
     * @return The optionally created object
     * @throws ImportGlobalFailException If the global import process must stop due to any fatal exception.
     */
    default Optional<O> unitaryImport(Path zipArchivePath, Path zipEntryPath, Merger merger, Logger logger) throws ImportGlobalFailException
    {
        try
        {
            Document propertiesXml = getPropertiesXml(zipEntryPath);
            
            String id = retrieveId(propertiesXml);
            if (merger.needsMerge(id))
            {
                AfterMerge afterMerge = merger.merge(id);
                switch (afterMerge)
                {
                    case STOP_PROCESS:
                        // stop the object creation, process next object id
                        return Optional.empty();
                    case CONTINUE_PROCESS:
                    default:
                        // continue the object creation
                        break;
                }
            }
            
            O created = create(zipEntryPath, id, propertiesXml);
            return Optional.of(created);
        }
        catch (ImportGlobalFailException e)
        {
            getReport().addError(new ImportError(e));
            throw e;
        }
        catch (AmetysObjectNotImportedException e)
        {
            logger.error("{} for '{}!{}' could not be imported.", objectNameForLogs(), zipArchivePath, zipEntryPath, e);
            getReport().addError(new ImportError(e));
            return Optional.empty();
        }
        catch (Exception e)
        {
            logger.error("An unexpected exception occured when trying to import {} for '{}!{}'.", objectNameForLogs(), zipArchivePath, zipEntryPath, e);
            getReport().addError(new ImportError(e));
            return Optional.empty();
        }
        finally
        {
            unitaryImportFinalize();
        }
    }
}

