/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.csv;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.LocaleUtils;
import org.apache.commons.lang3.StringUtils;
import org.supercsv.io.ICsvListReader;
import org.supercsv.util.Util;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.contenttype.ContentAttributeDefinition;
import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.data.type.BaseMultilingualStringElementType;
import org.ametys.cms.data.type.impl.MultilingualStringRepositoryElementType;
import org.ametys.cms.indexing.solr.SolrIndexHelper;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ModifiableDefaultContent;
import org.ametys.cms.repository.ModifiableWorkflowAwareContent;
import org.ametys.cms.workflow.AbstractContentWorkflowComponent;
import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.cms.workflow.CreateContentFunction;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.contentio.csv.SynchronizeModeEnumerator.ImportMode;
import org.ametys.plugins.contentio.in.ContentImportException;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeaterEntry;
import org.ametys.plugins.repository.data.holder.group.Repeater;
import org.ametys.plugins.repository.data.holder.values.SynchronizableRepeater;
import org.ametys.plugins.repository.data.holder.values.SynchronizableValue;
import org.ametys.plugins.repository.data.holder.values.SynchronizableValue.Mode;
import org.ametys.plugins.repository.data.type.ModelItemTypeConstants;
import org.ametys.plugins.repository.metadata.MultilingualString;
import org.ametys.plugins.repository.metadata.MultilingualStringHelper;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.ExpressionContext;
import org.ametys.plugins.repository.query.expression.MultilingualStringExpression;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ModelViewItemGroup;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.ViewElement;
import org.ametys.runtime.model.ViewElementAccessor;
import org.ametys.runtime.model.ViewItem;
import org.ametys.runtime.model.type.ElementType;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import com.opensymphony.workflow.WorkflowException;

/**
 * Import contents from an uploaded CSV file.
 */
public class CSVImporter extends AbstractLogEnabled implements Component, Serviceable
{
    
    /** Avalon Role */
    public static final String ROLE = CSVImporter.class.getName();

    /** Result key containing updated content's ids */
    public static final String RESULT_CONTENT_IDS = "contentIds";
    /** Result key containing number of errors */
    public static final String RESULT_NB_ERRORS = "nbErrors";
    /** Result key containing number of warnings */
    public static final String RESULT_NB_WARNINGS = "nbWarnings";

    private ContentWorkflowHelper _contentWorkflowHelper;

    private ContentTypeExtensionPoint _contentTypeEP;
    
    private AmetysObjectResolver _resolver;

    private I18nUtils _i18nUtils;

    private SolrIndexHelper _solrIndexHelper;
    
    public void service(ServiceManager smanager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _contentTypeEP = (ContentTypeExtensionPoint) smanager.lookup(ContentTypeExtensionPoint.ROLE);
        _contentWorkflowHelper = (ContentWorkflowHelper) smanager.lookup(ContentWorkflowHelper.ROLE);
        _i18nUtils = (I18nUtils) smanager.lookup(I18nUtils.ROLE);
        _solrIndexHelper = (SolrIndexHelper) smanager.lookup(SolrIndexHelper.ROLE);
    }

    /**
     * Extract contents from CSV file
     * @param mapping mapping of content attributes and CSV file header
     * @param view View of importing content
     * @param contentType content type to import
     * @param listReader mapReader to parse CSV file
     * @param createAction creation action id
     * @param editAction edition action id
     * @param workflowName workflow name
     * @param language language of created content.
     * @param importMode The import mode
     * @param parentContent Optional parent content
     * @param additionalTransientVars The optional additional transient vars to add to the transient vars
     * @return list of created contents
     * @throws IOException IOException while reading CSV
     */
    public Map<String, Object> importContentsFromCSV(Map<String, Object> mapping, View view, ContentType contentType, ICsvListReader listReader, int createAction, int editAction, String workflowName, String language, ImportMode importMode, Optional< ? extends Content> parentContent, Map<String, Object> additionalTransientVars) throws IOException
    {
        try
        {
            _solrIndexHelper.pauseSolrCommitForEvents(_getIndexationEvents());
            
            List<String> contentIds = new ArrayList<>();
            String[] columns = listReader.getHeader(true);
            int nbErrors = 0;
            int nbWarnings = 0;
            List<String> row = null;
            
            while ((row = listReader.read()) != null)
            {
                try
                {
                    if (listReader.length() != columns.length)
                    {
                        getLogger().error("[{}] Import from CSV file: content skipped because of invalid row: {}", contentType.getId(), row);
                        nbErrors++;
                        continue;
                    }
    
                    Map<String, String> rowMap = new HashMap<>();
                    Util.filterListToMap(rowMap, columns, row);
                    List<ViewItem> errors = new ArrayList<>();
                    SynchronizeResult synchronizeResult = processContent(view, rowMap, contentType, mapping, createAction, editAction, workflowName, language, errors, importMode, parentContent, additionalTransientVars);
                    Optional<ModifiableWorkflowAwareContent> content = synchronizeResult.content();
                    
                    // If we are in CREATE_ONLY mode but content was not created, do not add it to the contents list
                    if (content.isPresent() && (importMode != ImportMode.CREATE_ONLY || synchronizeResult.isCreated()))
                    {
                        contentIds.add(content.get().getId());
                    }
                    
                    if (!errors.isEmpty())
                    {
                        nbWarnings++;
                    }
                }
                catch (Exception e)
                {
                    nbErrors++;
                    getLogger().error("[{}] Import from CSV file: error importing the content on line {}", contentType.getId(), listReader.getLineNumber(), e);
                }
            }
            
            Map<String, Object> results = new HashMap<>();
            results.put(RESULT_CONTENT_IDS, contentIds);
            results.put(RESULT_NB_ERRORS, nbErrors);
            results.put(RESULT_NB_WARNINGS, nbWarnings);
            return results;
        }
        finally
        {
            _solrIndexHelper.restartSolrCommitForEvents(_getIndexationEvents());
        }
    }
    
    private String[] _getIndexationEvents()
    {
        return new String[] {
            ObservationConstants.EVENT_CONTENT_ADDED,
            ObservationConstants.EVENT_CONTENT_MODIFIED,
            ObservationConstants.EVENT_CONTENT_WORKFLOW_CHANGED,
            ObservationConstants.EVENT_CONTENT_VALIDATED
        };
    }
    
    /**
     * Process content for CSV import or synchronization
     * @param view The view
     * @param row The row containing the values
     * @param contentType The content type
     * @param mapping The mapping
     * @param createAction The create action
     * @param editAction The edit action
     * @param workflowName The wokflow name
     * @param language The language
     * @param errors The errors
     * @param importMode The import mode
     * @param parentContent The parent content
     * @param additionalTransientVars The additional transient vars
     * @return The synchronize result
     * @throws Exception If an error occurs
     */
    protected SynchronizeResult processContent(View view, Map<String, String> row, ContentType contentType, Map<String, Object> mapping, int createAction, int editAction, String workflowName, String language, List<ViewItem> errors, ImportMode importMode, Optional< ? extends Content> parentContent, Map<String, Object> additionalTransientVars) throws Exception
    {
        @SuppressWarnings("unchecked")
        List<String> attributeIdNames = (List<String>) mapping.get(ImportCSVFileHelper.NESTED_MAPPING_ID);
        @SuppressWarnings("unchecked")
        Map<String, Object> mappingValues = (Map<String, Object>) mapping.get(ImportCSVFileHelper.NESTED_MAPPING_VALUES);

        SynchronizeResult synchronizeResult = _synchronizeContent(row, contentType, view, attributeIdNames, mappingValues, createAction, editAction, workflowName, language, errors, importMode, parentContent, additionalTransientVars);
        return synchronizeResult;
    }

    private void _editContent(int editAction, View view, Map<String, Object> values, ModifiableWorkflowAwareContent content) throws WorkflowException
    {
        Collection<ModelItem> viewItemsDiff = content.getDifferences(view, values);
        if (!viewItemsDiff.isEmpty())
        {
            _contentWorkflowHelper.editContent(content, values, editAction, View.of(viewItemsDiff.toArray(ModelItem[]::new)));
        }
    }
    
    private Object _getValue(Optional<? extends Content> parentContent, ViewItem viewItem, Map<String, Object> mapping, Map<String, String> row, int createAction, int editAction, String language, List<ViewItem> errors, String prefix, ImportMode importMode, Map<String, Object> additionalTransientVars) throws Exception
    {
        if (viewItem instanceof ViewElement viewElement)
        {
            ElementDefinition elementDefinition = viewElement.getDefinition();
            if (elementDefinition instanceof ContentAttributeDefinition contentAttributeDefinition && viewElement instanceof ViewElementAccessor viewElementAccessor)
            {
                return _getContentAttributeDefinitionValues(parentContent, mapping, row, createAction, editAction, language, viewElementAccessor, contentAttributeDefinition, errors, importMode, additionalTransientVars);
            }
            else
            {
                return _getAttributeDefinitionValues(parentContent, mapping, row, elementDefinition, language, prefix);
            }
        }
        else if (viewItem instanceof ModelViewItemGroup modelViewItemGroup)
        {
            List<ViewItem> children = modelViewItemGroup.getViewItems();
            @SuppressWarnings("unchecked")
            Map<String, Object> nestedMap = (Map<String, Object>) mapping.get(viewItem.getName());
            @SuppressWarnings("unchecked")
            Map<String, Object> nestedMapValues = (Map<String, Object>) (nestedMap.get(ImportCSVFileHelper.NESTED_MAPPING_VALUES));
            if (ModelItemTypeConstants.REPEATER_TYPE_ID.equals(modelViewItemGroup.getDefinition().getType().getId()))
            {
                return _getRepeaterValues(parentContent, modelViewItemGroup, row, createAction, editAction, language, children, nestedMap, errors, prefix, importMode, additionalTransientVars);
            }
            else
            {
                return _getCompositeValues(parentContent, viewItem, row, createAction, editAction, language, children, nestedMapValues, errors, prefix, importMode, additionalTransientVars);
            }
        }
        else
        {
            errors.add(viewItem);
            throw new RuntimeException("Import from CSV file: unsupported type of ViewItem for view: " + viewItem.getName());
        }
    }

    private Map<String, Object> _getCompositeValues(Optional<? extends Content> parentContent, ViewItem viewItem, Map<String, String> row, int createAction, int editAction,
            String language, List<ViewItem> children, Map<String, Object> nestedMapValues, List<ViewItem> errors, String prefix, ImportMode importMode, Map<String, Object> additionalTransientVars)
    {
        Map<String, Object> compositeValues = new HashMap<>();
        for (ViewItem child : children)
        {
            try
            {
                compositeValues.put(child.getName(), _getValue(parentContent, child, nestedMapValues, row, createAction, editAction, language, errors, prefix + viewItem.getName() + ModelItem.ITEM_PATH_SEPARATOR, importMode, additionalTransientVars));
            }
            catch (Exception e)
            {
                errors.add(viewItem);
                getLogger().error("Import from CSV file: error while trying to get values for view: {}", viewItem.getName(), e);
            }
        }
        return compositeValues;
    }

    private SynchronizableRepeater _getRepeaterValues(Optional<? extends Content> parentContent, ModelViewItemGroup viewItem, Map<String, String> row, int createAction, int editAction, String language,
            List<ViewItem> children, Map<String, Object> nestedMap, List<ViewItem> errors, String prefix, ImportMode importMode, Map<String, Object> additionalTransientVars)
    {
        @SuppressWarnings("unchecked")
        Map<String, Object> mappingValues = (Map<String, Object>) nestedMap.get(ImportCSVFileHelper.NESTED_MAPPING_VALUES);
        @SuppressWarnings("unchecked")
        List<String> attributeIdNames = (List<String>) nestedMap.getOrDefault(ImportCSVFileHelper.NESTED_MAPPING_ID, List.of());
        Map<String, Object> repeaterValues = new HashMap<>();
        List<Map<String, Object>> repeaterValuesList = new ArrayList<>();
        List<Integer> indexList = new ArrayList<>();

        Optional<ModelAwareRepeater> repeater = parentContent.map(p -> p.getRepeater(prefix + viewItem.getName()));
        
        if (repeater.isPresent() && !attributeIdNames.isEmpty() && _allAttributesFilled(mappingValues, attributeIdNames))
        {
            indexList = repeater.get()
                .getEntries()
                .stream()
                .filter(entry ->
                {
                    return attributeIdNames.stream()
                        .allMatch(attributeName ->
                        {
                            // Get the entry value
                            Object entryValue = entry.getValue(attributeName);
                            
                            // Get the row value for the attribute (from CSV file)
                            Object rowValue = Optional.of(attributeName)
                                    .map(mappingValues::get)
                                    .map(String.class::cast)
                                    .map(row::get)
                                    .orElse(null);
                            
                            // Transform the row value to the right type then compare entry value and row value
                            return Optional.of(attributeName)
                                .map(viewItem::getModelViewItem)
                                .map(ViewElement.class::cast)
                                .map(ViewElement::getDefinition)
                                .map(ElementDefinition::getType)
                                .map(def -> def.castValue(rowValue))
                                .map(value -> value.equals(entryValue))
                                .orElse(false);
                        });
                })
                .map(ModelAwareRepeaterEntry::getPosition)
                .collect(Collectors.toList());
        }
        
        // If entries match with attribute ids, we replace the first entries
        // Else if the repeater exists, we assume we are on the next model item after the last one
        // Else (the repeater doesn't exist, we set the rowIndex to first index
        Integer rowIndex = indexList.isEmpty() ? repeater.map(Repeater::getSize).orElse(1) : indexList.get(0);
        
        for (ViewItem child : children)
        {
            try
            {
                Object entryValues = _getValue(parentContent, child, mappingValues, row, createAction, editAction, language, errors, prefix + viewItem.getName() + "[" + rowIndex + "]" + ModelItem.ITEM_PATH_SEPARATOR, importMode, additionalTransientVars);
                if (entryValues != null)
                {
                    repeaterValues.put(child.getName(), entryValues);
                }
            }
            catch (Exception e)
            {
                errors.add(viewItem);
                getLogger().error("Import from CSV file: error while trying to get values for view: {}", viewItem.getName(), e);
            }
        }
        repeaterValuesList.add(repeaterValues);
        
        if (indexList.isEmpty())
        {
            return SynchronizableRepeater.appendOrRemove(repeaterValuesList, Set.of());
        }
        else
        {
            // If several rows match the id, only replace the first but add a warning
            if (indexList.size() > 1)
            {
                errors.add(viewItem);
            }
            return SynchronizableRepeater.replace(repeaterValuesList, List.of(rowIndex));
        }
    }
    
    private Object _getContentAttributeDefinitionValues(Optional<? extends Content> parentContent, Map<String, Object> mapping, Map<String, String> row,
            int createAction, int editAction, String language, ViewElementAccessor viewElementAccessor, ContentAttributeDefinition contentAttributeDefinition, List<ViewItem> errors, ImportMode importMode, Map<String, Object> additionalTransientVars) throws Exception
    {
        String contentTypeId = contentAttributeDefinition.getContentTypeId();
        ContentType contentType = _contentTypeEP.getExtension(contentTypeId);
        @SuppressWarnings("unchecked")
        Map<String, Object> nestedMap = (Map<String, Object>) mapping.get(viewElementAccessor.getName());
        @SuppressWarnings("unchecked")
        Map<String, Object> mappingValues = (Map<String, Object>) nestedMap.get(ImportCSVFileHelper.NESTED_MAPPING_VALUES);
        @SuppressWarnings("unchecked")
        List<String> attributeIdNames = (List<String>) nestedMap.get(ImportCSVFileHelper.NESTED_MAPPING_ID);
        if (_allAttributesFilled(mappingValues, attributeIdNames))
        {
            View view = new View();
            view.addViewItems(viewElementAccessor.getViewItems());
            SynchronizeResult synchronizeResult = _synchronizeContent(row, contentType, view, attributeIdNames, mappingValues, createAction, editAction, null, language, errors, importMode, parentContent, additionalTransientVars);
            
            Optional<ModifiableWorkflowAwareContent> attachedContent = synchronizeResult.content();
            
            if (synchronizeResult.hasKey && attachedContent.isEmpty() && importMode == ImportMode.UPDATE_ONLY)
            {
                errors.add(viewElementAccessor);
            }
            
            // If content is multiple, we keep the old value list and we check if content was already inside, and add it otherwise
            if (attachedContent.isPresent() && contentAttributeDefinition.isMultiple())
            {
                Optional<ContentValue[]> multipleContents = parentContent.map(c -> c.getValue(contentAttributeDefinition.getPath()));
                if (!_containsContent(attachedContent.get(), multipleContents))
                {
                    // If there is no list or if it is empty, add it. Otherwise, we have to check if it is inside.
                    SynchronizableValue syncValue = new SynchronizableValue(List.of(attachedContent.get()));
                    syncValue.setMode(Mode.APPEND);
                    return syncValue;
                }
                else if (multipleContents.isPresent())
                {
                    // return existing values as otherwise the values would be erased
                    SynchronizableValue syncValue = new SynchronizableValue(Arrays.asList(multipleContents.get()));
                    syncValue.setMode(Mode.REPLACE);
                    return syncValue;
                }
            }
            else
            {
                return attachedContent.orElse(null);
            }
        }
        return null;
    }
    
    private boolean _containsContent(ModifiableWorkflowAwareContent attachedContent, Optional<ContentValue[]> multipleContents)
    {
        return multipleContents
                .map(Arrays::stream)
                .orElseGet(Stream::empty)
                .map(ContentValue::getContentId)
                .anyMatch(valueFromContent -> valueFromContent.equals(attachedContent.getId()));
    }
    
    private Object _getAttributeDefinitionValues(Optional<? extends Content> parentContent, Map<String, Object> mapping, Map<String, String> row, ElementDefinition elementDefinition, String language, String prefix)
    {
        ElementType elementType = elementDefinition.getType();
        String elementName = elementDefinition.getName();
        String elementColumn = (String) mapping.get(elementName);
        String valueAsString = row.get(elementColumn);
        
        Object value;
        if (elementType instanceof BaseMultilingualStringElementType && !MultilingualStringHelper.matchesMultilingualStringPattern(valueAsString))
        {
            MultilingualString multilingualString = new MultilingualString();
            multilingualString.add(LocaleUtils.toLocale(language), valueAsString);
            value = multilingualString;
        }
        else
        {
            value = elementType.castValue(valueAsString);
        }
        
        if (elementDefinition.isMultiple())
        {
            // Build path with index for repeaters.
            String pathWithIndex = prefix + elementDefinition.getName();
            
            Optional<Object[]> values = parentContent.map(c -> c.getValue(pathWithIndex));
            if (!_containsValue(value, parentContent.map(c -> c.getValue(pathWithIndex))))
            {
                // If there is no list or if it is empty, add it. Otherwise, we have to check if it is inside.
                // If there is no parentContent, still append as we want to fill the values map anyway.
                SynchronizableValue syncValue = new SynchronizableValue(value != null ? List.of(value) : List.of());
                syncValue.setMode(Mode.APPEND);
                return syncValue;
            }
            else if (values.isPresent())
            {
                // return existing values as otherwise the values would be erased
                SynchronizableValue syncValue = new SynchronizableValue(Arrays.asList(values.get()));
                syncValue.setMode(Mode.REPLACE);
                return syncValue;
            }
        }
        else
        {
            return value;
        }
        
        return null;
    }
    
    private boolean _containsValue(Object value, Optional<Object[]> multipleValues)
    {
        return multipleValues
                .map(Arrays::stream)
                .orElseGet(Stream::empty)
                .anyMatch(valueFromContent -> valueFromContent.equals(value));
    }

    private SynchronizeResult _synchronizeContent(Map<String, String> row, ContentType contentType, View view, List<String> attributeIdNames, Map<String, Object> mappingValues, int createAction, int editAction, String workflowName, String language, List<ViewItem> errors, ImportMode importMode, Optional< ? extends Content> parentContent, Map<String, Object> additionalTransientVars) throws Exception
    {
        SynchronizeResult synchronizeResult = _getOrCreateContent(mappingValues, row, contentType, Optional.ofNullable(workflowName), createAction, language, attributeIdNames, parentContent, importMode, additionalTransientVars);
        Optional<ModifiableWorkflowAwareContent> content = synchronizeResult.content();
        
        // If we are on CREATE_ONLY mode and content already exists, or if we are on UPDATE_ONLY mode and content does not exists, stop recursivity
        if (importMode == ImportMode.CREATE_ONLY && !synchronizeResult.isCreated() || importMode == ImportMode.UPDATE_ONLY && content.isEmpty())
        {
            return synchronizeResult;
        }
        
        Map<String, Object> values = _getValues(content, row, view, mappingValues, createAction, editAction, language, errors, importMode, additionalTransientVars);
        if (!values.isEmpty())
        {
            if (content.isEmpty())
            {
                // Throw this exception only when values are filled, as an empty content should not trigger any warning
                throw new ContentImportException("Can't create and fill content of content type '" + contentType.getId() + "' and following values '" + values + "' : at least one of those identifiers is null : " + attributeIdNames);
            }
            else
            {
                try
                {
                    _editContent(editAction, view, values, content.get());
                }
                catch (WorkflowException e)
                {
                    errors.addAll(view.getViewItems());
                    getLogger().error("[{}] Import from CSV file: error editing the content [{}] after import, some values have not been set", contentType.getId(), content.get().getId(), e);
                }
            }
        }
        
        return synchronizeResult;
    }
    
    /**
     * Returns a list of additional expression to retrieve the existing content
     * @param mapping The mapping
     * @param row The row
     * @param contentType The content type
     * @param parentContent The parent content
     * @param additionalTransientVars Additional transients vars
     * @param valuesUsedAsIds The values used as ids, if some values are added in expression, they can be added in this list to be printed in error logs
     * @return a list of additional expression to retrieve the existing content
     */
    protected List<Expression> additionalExpressionToRetrieveContent(Map<String, Object> mapping, Map<String, String> row, ContentType contentType, Optional<? extends Content> parentContent, Map<String, Object> additionalTransientVars, List<String> valuesUsedAsIds)
    {
        // No additional expression by default
        return List.of();
    }
    
    private SynchronizeResult _getOrCreateContent(Map<String, Object> mapping, Map<String, String> row, ContentType contentType, Optional<String> workflowName, int createAction, String language, List<String> attributeIdNames, Optional<? extends Content> parentContent, ImportMode importMode, Map<String, Object> additionalTransientVars) throws ContentImportException, WorkflowException
    {
        AndExpression expression = new AndExpression();
        List<String> values = new ArrayList<>();
        
        for (String attributeName : attributeIdNames)
        {
            ModelItem modelItem = contentType.getModelItem(attributeName);
            String attributePath = (String) mapping.get(attributeName);
            String value = row.get(attributePath);
            values.add(value);
            
            if (value == null)
            {
                return new SynchronizeResult(false, Optional.empty(), false);
            }
            
            // Get content
            if (modelItem.getType() instanceof MultilingualStringRepositoryElementType)
            {
                expression.add(new MultilingualStringExpression(attributeName, Operator.EQ, value, language));
            }
            else
            {
                expression.add(new StringExpression(attributeName, Operator.EQ, value));
            }
        }
        
        expression.addAll(additionalExpressionToRetrieveContent(mapping, row, contentType, parentContent, additionalTransientVars, values));

        expression.add(_contentTypeEP.createHierarchicalCTExpression(contentType.getId()));
        
        if (!contentType.isMultilingual())
        {
            expression.add(new StringExpression("language", Operator.EQ, language, ExpressionContext.newInstance().withInternal(true)));
        }
        
        String xPathQuery = ContentQueryHelper.getContentXPathQuery(expression);
        AmetysObjectIterable<ModifiableDefaultContent> matchingContents = _resolver.query(xPathQuery);
        if (matchingContents.getSize() > 1)
        {
            throw new ContentImportException("More than one content found for type " + contentType.getId() + " with "
                    + attributeIdNames + " as identifier and " + values + " as value");
        }
        else if (matchingContents.getSize() == 1)
        {
            return new SynchronizeResult(false, Optional.of(matchingContents.iterator().next()), true);
        }
        else if (importMode == ImportMode.UPDATE_ONLY)
        {
            return new SynchronizeResult(false, Optional.empty(), true);
        }
        
        // Create content

        if (contentType.isAbstract())
        {
            throw new ContentImportException("Can not create content for type " + contentType.getId() + " with "
                    + attributeIdNames + " as identifier and " + values + " as value, the content type is abstract");
        }
        
        Map<String, Object> result;
        String title;
        if (mapping.containsKey("title"))
        {
            title = row.get(mapping.get("title"));
        }
        else
        {
            title = _i18nUtils.translate(contentType.getDefaultTitle(), language);
        }
        
        
        String finalWorkflowName = workflowName.or(contentType::getDefaultWorkflowName)
                                               .orElseThrow(() -> new ContentImportException("No workflow specified for content type " + contentType.getId() + " with "
                                                       + attributeIdNames + " as identifier and " + values + " as value"));

        Map<String, Object> inputs = new HashMap<>();
        inputs.put(CreateContentFunction.INITIAL_VALUE_SUPPLIER, new Function<List<String>, Object>()
        {
            public Object apply(List<String> keys)
            {
                // Browse the mapping to find the column related to the attribute
                Object nestedValue = mapping;
                for (String key : keys)
                {
                    nestedValue = ((Map) nestedValue).get(key);
                    // If nestedValue is null, the attribute is absent from the map, no value can be found
                    if (nestedValue == null)
                    {
                        return null;
                    }
                    // If nestedValue is a map, the key is a complex element such a content or a composite,
                    // we need to keep browsing the map to find the column
                    if (nestedValue instanceof Map)
                    {
                        nestedValue = ((Map) nestedValue).get(ImportCSVFileHelper.NESTED_MAPPING_VALUES);
                    }
                }
                
                // Get the value of the attribute for the current row
                return row.get(nestedValue.toString());
            }
        });
        
        parentContent.ifPresent(content -> inputs.put(CreateContentFunction.PARENT_CONTEXT_VALUE, content.getId()));
        
        // Add all additional transient vars
        inputs.putAll(additionalTransientVars);
        
        // CONTENTIO-253 To avoid issue with title starting with a non letter character, we prefix the name with the contentTypeId
        String prefix = StringUtils.substringAfterLast(contentType.getId(), ".").toLowerCase();
        String contentName = prefix + "-" + title;
        
        if (contentType.isMultilingual())
        {
            inputs.put(CreateContentFunction.CONTENT_LANGUAGE_KEY, language);
            result = _contentWorkflowHelper.createContent(finalWorkflowName, createAction, contentName, Map.of(language, title), new String[] {contentType.getId()}, null, inputs);
        }
        else
        {
            result = _contentWorkflowHelper.createContent(finalWorkflowName, createAction, contentName, title, new String[] {contentType.getId()}, null, language, inputs);
        }
        
        ModifiableWorkflowAwareContent content = (ModifiableWorkflowAwareContent) result.get(AbstractContentWorkflowComponent.CONTENT_KEY);
        return new SynchronizeResult(true, Optional.of(content), true);
    }
    
    private Map<String, Object> _getValues(Optional<ModifiableWorkflowAwareContent> content, Map<String, String> row, View view, Map<String, Object> mappingValues, int createAction, int editAction, String language, List<ViewItem> errors, ImportMode importMode, Map<String, Object> additionalTransientVars)
    {
        Map<String, Object> values = new HashMap<>();
        
        for (ViewItem viewItem : view.getViewItems())
        {
            try
            {
                Object value = _getValue(content, viewItem, mappingValues, row, createAction, editAction, language, errors, StringUtils.EMPTY, importMode, additionalTransientVars);
                if (value != null)
                {
                    values.put(viewItem.getName(), value);
                }
            }
            catch (Exception e)
            {
                errors.add(viewItem);
                getLogger().error("Import from CSV file: error while trying to get values for item '{}'", viewItem.getName(), e);
            }
        }
        
        return values;
    }
    
    private boolean _allAttributesFilled(Map<String, Object> mappingValues, List<String> attributeNames)
    {
        return mappingValues.entrySet()
            .stream()
            .filter(entry -> attributeNames.contains(entry.getKey()))
            .map(Entry::getValue)
            .allMatch(Objects::nonNull);
    }
    
    /**
     * Synchronize result
     * @param isCreated If the content was created
     * @param content The content
     * @param hasKey has key
     */
    protected record SynchronizeResult(boolean isCreated, Optional<ModifiableWorkflowAwareContent> content, boolean hasKey) { /* empty */ }
    
}
