/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.export.sql;

import java.io.IOException;
import java.util.ArrayList;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.quartz.JobExecutionContext;

import org.ametys.core.schedule.Schedulable;
import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.mail.SendMailHelper;
import org.ametys.plugins.core.impl.schedule.AbstractStaticSchedulable;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;

import jakarta.mail.MessagingException;

/**
 * A {@link Schedulable} job which delete classified ads after an defined number of days
 */
public class ContentExportSchedulable extends AbstractStaticSchedulable
{
    /** The export manager */
    protected ExportManager _exportManager;
    
    /** The i18n translator */
    protected I18nUtils _i18nTranslator;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _exportManager = (ExportManager) manager.lookup(ExportManager.ROLE);
        _i18nTranslator = (I18nUtils) manager.lookup(I18nUtils.ROLE);
    }
    
    @Override
    public void execute(JobExecutionContext context, ContainerProgressionTracker progressionTracker) throws Exception
    {
        long time_0 = System.currentTimeMillis();
        try
        {
            _exportManager.export();
            
            if (getLogger().isInfoEnabled())
            {
                ArrayList<String> i18nParams = new ArrayList<>();
                i18nParams.add(String.valueOf(System.currentTimeMillis() - time_0));
                getLogger().info(_i18nTranslator.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_EXPORT_LOG_END", i18nParams)));
            }
            
            _sendMail(false);
        }
        catch (Exception ex)
        {
            _sendMail(true);
            
            throw ex;
        }
    }
    
    /**
     * Send the mail report
     * @param error true if some error occurred
     * @throws MessagingException if a messaging error occurred
     * @throws IOException if an I/O error occurred during mail building
     */
    protected void _sendMail(boolean error) throws MessagingException, IOException
    {
        String language = _userLanguagesManager.getDefaultLanguage();
        String subject = _i18nTranslator.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_EXPORT_REPORT_MAIL_SUBJECT"), language);
        String email = Config.getInstance().getValue("org.ametys.plugins.contentio.content.export.admin.mail");
        String body = _i18nTranslator.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_EXPORT_REPORT_MAIL_BODY" + (error ? "_ERROR" : "")), language);
        
        try
        {
            SendMailHelper.newMail()
                          .withSubject(subject)
                          .withTextBody(body)
                          .withRecipient(email)
                          .sendMail();
        }
        catch (MessagingException e)
        {
            if (getLogger().isWarnEnabled())
            {
                getLogger().warn(_i18nTranslator.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_EXPORT_LOG_SEND_MAIL_ERROR"), language), e);
            }
        }
    }
}
