/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.export.sql;

import java.io.IOException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.data.type.ModelItemTypeConstants;
import org.ametys.cms.languages.LanguagesManager;
import org.ametys.core.datasource.ConnectionHelper;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.repository.model.CompositeDefinition;
import org.ametys.plugins.repository.model.RepeaterDefinition;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ModelItemContainer;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 *  Create SQl Table Component
 */
public class CreateSqlTableComponent extends AbstractLogEnabled implements Component, Serviceable
{
    /** The component role */
    public static final String ROLE = CreateSqlTableComponent.class.getName();
    
    /** Code of default language for comments */
    public static final String DEFAULT_LANGUAGE_CODE_FOR_COMMENTS = "en";
    
    /** The engine */
    public static final String MYSQL_CONTENT_EXPORT_ENGINE = "MyISAM";
    
    /** The encoding */
    public static final String MYSQL_CONTENT_EXPORT_CHARSET = "UTF8MB4";
    
    /** Prefix for parent table column */
    public static final String COLUMN_PARENT_TABLE_PREFIX = "PID_";
    
    private static final Pattern _MYSQL_VERSION_NUMBER_EXTRACT = Pattern.compile("^([0-9]+).*$");
    
    /** Content type extension point. */
    protected ContentTypeExtensionPoint _contentTypeExtensionPoint;
    
    /** The i18n translator. */
    protected I18nUtils _i18nTranslator;
    
    /** The normalise name component. */
    protected NormalizeNameComponent _normalizeNameComponent;
    
    /** The language manager */
    protected LanguagesManager _languageManager;
    
    private Connection _connection;
    private String _databaseType;
    
    private LinkedHashMap<String, ExportTableInfo> _tablesInfos;
    private int _commentTableMaxLength;
    private int _commentColumnMaxLength;
    private String _sqlTablePrefix;
    private String _sqlPrefixConf;
    private Map<String, Map<String, String>> _mapping;
    private Map<String, Map<String, String>> _reservedWords;
    private String _mappingPolicy;
    private ArrayList<String> _mappingTablesQueries;
    private boolean _exportNoMultiValuedTable;
    
    private int _fkIndice;
    private int _pkIndice;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _i18nTranslator = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _contentTypeExtensionPoint = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
        _normalizeNameComponent = (NormalizeNameComponent) manager.lookup(NormalizeNameComponent.ROLE);
        _languageManager = (LanguagesManager) manager.lookup(LanguagesManager.ROLE);
    } 
    
    /**
     * Create sql tables for contents
     * @param exportConfiguration the content export configuration
     * @return tablesInfos
     * @throws SQLException if a sql error occurred
     * @throws IOException if an IO error occurred
     */
    public synchronized Map<String, ExportTableInfo> createTables(ExportConfiguration exportConfiguration) throws SQLException, IOException
    {
        // Get from configuration
        _sqlPrefixConf = exportConfiguration.getTablePrefix();
        _sqlTablePrefix = exportConfiguration.getTablePrefix();
        _mapping = exportConfiguration.getMappingSql();
        _mappingPolicy = exportConfiguration.getMappingPolicy();
        _reservedWords = exportConfiguration.getReservedWords();
        _exportNoMultiValuedTable = exportConfiguration.exportNoMultiValuedTable();
        
        // Initialization
        _fkIndice = 1;
        _pkIndice = 1;
        _tablesInfos = new LinkedHashMap<>();
        _mappingTablesQueries = new ArrayList<>();
        
        try
        {
            String datasourceId = Config.getInstance().getValue("org.ametys.plugins.contentio.content.export.datasource");
            _connection = ConnectionHelper.getConnection(datasourceId); 
            _databaseType = ConnectionHelper.getDatabaseType(_connection);
            
            String productVersion = _connection.getMetaData().getDatabaseProductVersion();
            initialize(productVersion);
            
            boolean isInfoEnabled = getLogger().isInfoEnabled();
            if (isInfoEnabled)
            {
                getLogger().info(_i18nTranslator.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_EXPORT_LOG_ANALYZE_TABLE_CREATE_BEGIN")));
            }
            
            createTablesInfos(exportConfiguration.getContentTypesToExport());
            
            if (isInfoEnabled)
            {
                getLogger().info(_i18nTranslator.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_EXPORT_LOG_ANALYZE_TABLE_CREATE_END")));
                getLogger().info(_i18nTranslator.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_EXPORT_LOG_ANALYZE_TABLE_CREATE_MAPPING_BEGIN")));
            }
            
            createMappingTables();
            
            if (isInfoEnabled)
            {
                getLogger().info(_i18nTranslator.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_EXPORT_LOG_ANALYZE_TABLE_CREATE_MAPPING_END")));
            }
            
            createRichTextDataTable();
            createContentTable();
            
            executeQueries();
        }
        finally
        {
            ConnectionHelper.cleanup(_connection);
        }
        
        return _tablesInfos;
    }

    /**
     * Initialization
     * @param productVersion The database product version
     */
    protected void initialize(String productVersion)
    {
        // Get the maximun number of authorized characters for table and column comments
        if (_databaseType.equals(ConnectionHelper.DATABASE_MYSQL))
        {
            String[] parts = productVersion.split("\\.");
            
            int majorVersion = Integer.parseInt(parts[0]);
            int minorVersion = 0;
            int patchVersion = 0;
            if (parts.length > 1)
            {
                Matcher matcher = _MYSQL_VERSION_NUMBER_EXTRACT.matcher(parts[1]);
                if (matcher.matches())
                {
                    minorVersion = Integer.parseInt(matcher.group(1));
                }
            }
            
            if (parts.length > 2)
            {
                Matcher matcher = _MYSQL_VERSION_NUMBER_EXTRACT.matcher(parts[2]);
                if (matcher.matches())
                {
                    patchVersion = Integer.parseInt(matcher.group(1));
                }
            }
            
            if (majorVersion > 5 || majorVersion >= 5 && minorVersion > 5 || majorVersion >= 5 && minorVersion >= 5 && patchVersion >= 3)
            {
                // Version 5.5.3 or later
                _commentTableMaxLength = 2048;
                _commentColumnMaxLength = 1024;
            }
            else
            {
                // Version before 5.5.3
                _commentTableMaxLength = 60;
                _commentColumnMaxLength = 255;
            }
        }
        else
        {
            // No max
            _commentTableMaxLength = 2048;
            _commentColumnMaxLength = 1024;
        }
    }
    
    /**
     * Created all tables informations
     * @param contents to export
     * @throws SQLException if a sql error occurred
     */
    protected void createTablesInfos(Map<String, String> contents) throws SQLException
    {
        for (Entry<String, String> entry : contents.entrySet()) 
        {
            String contentTypeId = entry.getKey();

            ContentType contentType = _contentTypeExtensionPoint.getExtension(contentTypeId);
            if (!contentType.isAbstract())
            {
                String tableName = entry.getValue();
                String comment = _i18nTranslator.translate(contentType.getLabel(), DEFAULT_LANGUAGE_CODE_FOR_COMMENTS) + ": " + _i18nTranslator.translate(contentType.getDescription(), DEFAULT_LANGUAGE_CODE_FOR_COMMENTS);
                createQueriesForTableCreation(contentType, tableName, null, comment, false);
            }
            
            _sqlTablePrefix = _sqlPrefixConf;
        }
    }
    
    /**
     * Create SQL queries to create necessary SQL tables for content type export
     * @param modelItemContainer the current {@link ModelItemContainer}
     * @param tableName the table name The SQL table name
     * @param tableParentName  The SQL parent table name
     * @param comment the comment
     * @param isSortTable true if table's rows have to be ordered
     */
    protected void createQueriesForTableCreation(ModelItemContainer modelItemContainer, String tableName, String tableParentName, String comment, boolean isSortTable)
    {
        ExportTableInfo tableInfo = new ExportTableInfo(tableName);
        tableInfo.incrementNbColumns();
        
        String tableNameNormalized = _normalizeNameComponent.normalizedTableName(_sqlTablePrefix, _mappingPolicy, tableName, _connection);
        _tablesInfos.put(tableName, tableInfo);
        
        StringBuilder currentCreateTableSQLQuery = new StringBuilder();
        
        currentCreateTableSQLQuery.append("CREATE TABLE ");
        currentCreateTableSQLQuery.append(tableNameNormalized);
        currentCreateTableSQLQuery.append(" (");
        currentCreateTableSQLQuery.append(_normalizeNameComponent.normalizedColumnName(_mappingPolicy, "id_" + tableName, tableName, _reservedWords, _connection));
        currentCreateTableSQLQuery.append(" VARCHAR(250)");
        currentCreateTableSQLQuery.append(createPrimaryKeyQuery());
        currentCreateTableSQLQuery.append(createCommentQuery(tableName, "id_" + tableName, "Ametys ID"));
        
        if (StringUtils.isNotEmpty(tableParentName))
        {
            currentCreateTableSQLQuery.append(", ");
            addColumnParentId(currentCreateTableSQLQuery, tableParentName, tableName);
        }
        
        if (isSortTable)
        {
            currentCreateTableSQLQuery.append(", ");
            addSortColumn(currentCreateTableSQLQuery, tableName);
        }
        
        addColumnForContainer(modelItemContainer, currentCreateTableSQLQuery, tableName, "");
        
        if (modelItemContainer instanceof ContentType)
        {
            addAdditionalData((ContentType) modelItemContainer, currentCreateTableSQLQuery, tableName);
        }
        
        currentCreateTableSQLQuery.append(") ");
        currentCreateTableSQLQuery.append(createEngineQuery());
        currentCreateTableSQLQuery.append(createCommentQuery(tableName, null, comment));
        
        tableInfo.addCreateQuery(currentCreateTableSQLQuery.toString());
    }  
    
    /**
     * Create table for images in rich text
     */
    protected void createRichTextDataTable()
    {
        String dataTableName = _sqlTablePrefix + ExportManager.RICH_TEXT_DATA_TABLE_NAME;
        ExportTableInfo tableInfo = new ExportTableInfo(dataTableName);
        tableInfo.incrementNbColumns(8);

        String dateTableNameNormalized = _normalizeNameComponent.normalizedTableName(_sqlTablePrefix, "FULL", dataTableName, _connection);
        _tablesInfos.put(dataTableName, tableInfo);
        
        StringBuilder currentCreateTableSQLQuery = new StringBuilder();

        currentCreateTableSQLQuery.append("CREATE TABLE ");
        currentCreateTableSQLQuery.append(dateTableNameNormalized);
        currentCreateTableSQLQuery.append(" (id_data VARCHAR(250)");
        currentCreateTableSQLQuery.append(createPrimaryKeyQuery());
        currentCreateTableSQLQuery.append(createCommentQuery(dataTableName, "id_data", "Data ID"));
        currentCreateTableSQLQuery.append(", id_content VARCHAR(255)");
        currentCreateTableSQLQuery.append(createCommentQuery(dataTableName, "id_content", "Parent ID"));
        
        currentCreateTableSQLQuery.append(", attribute_name ");
        currentCreateTableSQLQuery.append(convertTypeToSql("string"));
        currentCreateTableSQLQuery.append(createCommentQuery(dataTableName, "attribute_name", "Richtext attribute name"));
        currentCreateTableSQLQuery.append(", data_name ");
        currentCreateTableSQLQuery.append(convertTypeToSql("string"));
        currentCreateTableSQLQuery.append(createCommentQuery(dataTableName, "data_name", "Name"));
        
        currentCreateTableSQLQuery.append(", data ");
        currentCreateTableSQLQuery.append(convertTypeToSql("file"));
        currentCreateTableSQLQuery.append(createCommentQuery(dataTableName, "data", "Data"));
        currentCreateTableSQLQuery.append(", data_mimetype VARCHAR(255)");
        currentCreateTableSQLQuery.append(createCommentQuery(dataTableName, "data_mimetype", "Mime type"));
        currentCreateTableSQLQuery.append(", data_size INT");
        currentCreateTableSQLQuery.append(createCommentQuery(dataTableName, "data_size", "Size"));
        currentCreateTableSQLQuery.append(", data_lastmodified ");
        currentCreateTableSQLQuery.append(convertTypeToSql("datetime"));
        currentCreateTableSQLQuery.append(createCommentQuery(dataTableName, "data_lastmodified", "Last modification date"));
        currentCreateTableSQLQuery.append(", ");
        
        addSortColumn(currentCreateTableSQLQuery, dataTableName);
        
        currentCreateTableSQLQuery.append(") ");
        currentCreateTableSQLQuery.append(createEngineQuery());

        String comment = "Data table of all rich text";
        currentCreateTableSQLQuery.append(createCommentQuery(dataTableName, null, comment));
        
        tableInfo.addCreateQuery(currentCreateTableSQLQuery.toString());
    }
    
    /**
     * Create table for content
     */
    protected void createContentTable()
    {
        String dataTableName = _sqlPrefixConf + ExportManager.CONTENT_TABLE_NAME;
        ExportTableInfo tableInfo = new ExportTableInfo(dataTableName);
        tableInfo.incrementNbColumns(2);

        String dataTableNameNormalized = _normalizeNameComponent.normalizedTableName(_sqlTablePrefix, "FULL", dataTableName, _connection);
        _tablesInfos.put(dataTableName, tableInfo);
        
        StringBuilder currentCreateTableSQLQuery = new StringBuilder();

        currentCreateTableSQLQuery.append("CREATE TABLE ");
        currentCreateTableSQLQuery.append(dataTableNameNormalized);
        currentCreateTableSQLQuery.append(" (id_content VARCHAR(80)");
        currentCreateTableSQLQuery.append(createCommentQuery(dataTableName, "id_content", "Content ID"));
        currentCreateTableSQLQuery.append(", table_name VARCHAR(170)");
        currentCreateTableSQLQuery.append(createCommentQuery(dataTableName, "table_name", "Table name"));
        currentCreateTableSQLQuery.append(", CONSTRAINT pk_content PRIMARY KEY (id_content, table_name) ");
        currentCreateTableSQLQuery.append(") ");
        currentCreateTableSQLQuery.append(createEngineQuery());

        String comment = "Link table of content with it's own table";
        currentCreateTableSQLQuery.append(createCommentQuery(dataTableName, null, comment));
        
        tableInfo.addCreateQuery(currentCreateTableSQLQuery.toString());
    }
    
    /**
     * Create the table for the enumerator (key, value)
     * @param definition the attribute definition
     * @param tableParentName the table parent name
     * @param tableName the table name
     * @param columnName the column name
     * @param currentCreateTableSQLQuery the current SQL create table query 
     */
    protected void createTableForEnumerator(ElementDefinition definition, String tableParentName, String tableName, String columnName, StringBuilder currentCreateTableSQLQuery)
    {
        ExportTableInfo tableInfo = new ExportTableInfo(tableName);
        tableInfo.incrementNbColumns();
        
        String tableNameNormalized = _normalizeNameComponent.normalizedTableName(_sqlTablePrefix, _mappingPolicy, tableName, _connection);
        _tablesInfos.put(tableName, tableInfo);
        
        StringBuilder sql = new StringBuilder();
        
        sql.append("CREATE TABLE ");
        sql.append(tableNameNormalized);
        sql.append(" (key_enum ");
        sql.append(definition.getType().getId().equals(org.ametys.runtime.model.type.ModelItemTypeConstants.LONG_TYPE_ID) ? "INT" : "VARCHAR(250)");
        sql.append(createPrimaryKeyQuery());
        sql.append(createCommentQuery(tableName, "key_enum_" + definition.getName(), "Enumerator key"));
        
        addColumn(sql, "value_enum_" + definition.getName(), tableName, null, "string");
        
        sql.append(") ");
        sql.append(createEngineQuery());
        sql.append(createCommentQuery(tableName, null, "Enumerator table " + definition.getName() + " linked to the table " + tableParentName));
        
        tableInfo.addCreateQuery(sql.toString());
        
//        String fkName = _normaliseNameComponent.normalizedColumnName(_mappingPolicy, columnName, tableParentName, _connection);
//        String fkTableName = _normaliseNameComponent.normalizedTableName(_sqlTablePrefix, _mappingPolicy, tableName, _connection);
//        String fkColumnName = "key_enum";
//        currentCreateTableSQLQuery.append(createForeignKeyQuery(fkName, fkTableName, fkColumnName));
        //TODO foreign key ??
        
        fillTableForEnumerator(definition, tableName);
    }
    
    /**
     * Fill values for the enumerator table
     * @param definition the attribute definition
     * @param tableName the table name
     */
    protected void fillTableForEnumerator(ElementDefinition definition, String tableName)
    {   
        try
        {
            for (Entry<Object, I18nizableText> entry : ((Map<Object, I18nizableText>) definition.getEnumerator().getEntries()).entrySet()) 
            {
                String enumValue = _i18nTranslator.translate(entry.getValue(), DEFAULT_LANGUAGE_CODE_FOR_COMMENTS);
                String enumKey = entry.getKey().toString();

                StringBuilder sql = new StringBuilder();
                sql.append("INSERT INTO ");
                sql.append(_normalizeNameComponent.normalizedTableName(_sqlTablePrefix, _mappingPolicy, tableName, _connection));
                sql.append(" VALUES ('");
                sql.append(_normalizeNameComponent.escapeValue(enumKey, _connection));
                sql.append("', '");
                sql.append(_normalizeNameComponent.escapeValue(enumValue, _connection));
                sql.append("')");

                _tablesInfos.get(tableName).addInsertQuery(sql.toString());
            }
        }
        catch (Exception e)
        {
            getLogger().warn(e.getMessage(), e);
        }
    }
    
    /**
     * Create the two table mapping (for table name and column name)
     */
    protected void createMappingTables()
    {
        PreparedStatement stmt = null;
        String mappingTableName = _sqlTablePrefix + ExportManager.MAPPING_TABLE_NAME;
        String mappingColumnName = _sqlTablePrefix + ExportManager.MAPPING_COLUMN_NAME;
        
        ExportTableInfo mappingTableInfo = new ExportTableInfo(mappingTableName);
        mappingTableInfo.incrementNbColumns(3);

        ExportTableInfo mappingColumnInfo = new ExportTableInfo(mappingColumnName);
        mappingColumnInfo.incrementNbColumns(3);

        String mappingTableNameNormalized = _normalizeNameComponent.normalizedTableName(_sqlTablePrefix, "FULL", mappingTableName, _connection);
        String mappingColumnNameNormalized = _normalizeNameComponent.normalizedTableName(_sqlTablePrefix, "FULL", mappingColumnName, _connection);
        
        _tablesInfos.put(mappingTableName, mappingTableInfo);
        _tablesInfos.put(mappingColumnName, mappingColumnInfo);
        
        StringBuilder createMappingTableSQLQuery = new StringBuilder();

        createMappingTableSQLQuery.append("CREATE TABLE ");
        createMappingTableSQLQuery.append(mappingTableNameNormalized);
        createMappingTableSQLQuery.append(" (id_table INT");
        createMappingTableSQLQuery.append(createPrimaryKeyQuery());
        createMappingTableSQLQuery.append(createCommentQuery(mappingTableName, "id_table", "Table name ID"));
        createMappingTableSQLQuery.append(", real_name VARCHAR(512)");
        createMappingTableSQLQuery.append(createCommentQuery(mappingTableName, "real_name", "Real name"));
        createMappingTableSQLQuery.append(", modified_name VARCHAR(512)");
        createMappingTableSQLQuery.append(createCommentQuery(mappingTableName, "modified_name", "Normalized name"));
        
        createMappingTableSQLQuery.append(") ");
        createMappingTableSQLQuery.append(createEngineQuery());

        String comment = "Mapping table between real name and normalized name.";
        createMappingTableSQLQuery.append(createCommentQuery(mappingTableName, null, comment));
        
        try
        {
            _mappingTablesQueries.add(createMappingTableSQLQuery.toString());
            stmt = _connection.prepareStatement(createMappingTableSQLQuery.toString());
            stmt.execute();
        }
        catch (SQLException e)
        {
            getLogger().error(_i18nTranslator.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_EXPORT_LOG_ANALYZE_TABLE_CREATE_MAPPING_TABLE_ERROR")), e);
        }
        finally
        {
            ConnectionHelper.cleanup(stmt);
        }
        
        StringBuilder createMappingColumnSQLQuery = new StringBuilder();

        createMappingColumnSQLQuery.append("CREATE TABLE ");
        createMappingColumnSQLQuery.append(mappingColumnNameNormalized);
        createMappingColumnSQLQuery.append(" (id_table INT");
        createMappingColumnSQLQuery.append(createCommentQuery(mappingColumnName, "id_table", "Table name id"));
        createMappingColumnSQLQuery.append(createForeignKeyQuery("id_table", _normalizeNameComponent.normalizedTableName(_sqlTablePrefix, _mappingPolicy, mappingTableName, _connection), "id_table"));
        createMappingColumnSQLQuery.append(", real_name VARCHAR(512)");
        createMappingColumnSQLQuery.append(createCommentQuery(mappingColumnName, "real_name", "Real column name"));
        createMappingColumnSQLQuery.append(", modified_name VARCHAR(512)");
        createMappingColumnSQLQuery.append(createCommentQuery(mappingColumnName, "modified_name", "Normalized column name"));
        
        createMappingColumnSQLQuery.append(") ");
        createMappingColumnSQLQuery.append(createEngineQuery());

        String commentColum = "Mapping table between real column name and normalized column name.";
        createMappingColumnSQLQuery.append(createCommentQuery(mappingColumnName, null, commentColum));
        
        try
        {
            _mappingTablesQueries.add(createMappingColumnSQLQuery.toString());
            stmt = _connection.prepareStatement(createMappingColumnSQLQuery.toString());
            stmt.execute();
        }
        catch (SQLException e)
        {
            getLogger().error(_i18nTranslator.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_EXPORT_LOG_ANALYZE_TABLE_CREATE_MAPPING_COLUMN_ERROR")), e);
        }
        finally
        {
            ConnectionHelper.cleanup(stmt);
        }
        
        fillMappingTables(mappingTableName, mappingColumnName);
    }
    
    /**
     * Fill the two mapping table (for table name and column name)
     * @param mappingTableName the table mapping table name
     * @param mappingColumnName the column mapping table name
     */
    protected void fillMappingTables(String mappingTableName, String mappingColumnName)
    {
        HashMap<String, String> mappingTable = (HashMap<String, String>) _normalizeNameComponent.getMappingTableNameFromCache();
        HashMap<String, HashMap<String, String>> mappingColumn = (HashMap<String, HashMap<String, String>>) _normalizeNameComponent.getMappingTableColumnNameFromCache();
        
        PreparedStatement stmtTableName = null;
        PreparedStatement stmtColumnName = null;
        try
        {
            stmtTableName = getInsertPreparedStatementFromTableName(mappingTableName);
            stmtColumnName = getInsertPreparedStatementFromTableName(mappingColumnName);
            int i = 0;
            for (Entry<String, String> entry : mappingTable.entrySet()) 
            {
                String realName = entry.getKey();
                String modifiedName = entry.getValue();
                
                if (getLogger().isDebugEnabled())
                {
                    String tableName = _normalizeNameComponent.normalizedTableName(_sqlTablePrefix, _mappingPolicy, mappingTableName, _connection);
                    getLogger().debug("INSERT INTO {} VALUES ('{}', '{}', '{}')", tableName, i, realName, modifiedName);
                }
                
                stmtTableName.setInt(1, i);
                stmtTableName.setString(2, realName);
                stmtTableName.setString(3, modifiedName);
                
                stmtTableName.addBatch();
                
                if (mappingColumn.containsKey(realName))
                {
                    for (Entry<String, String> entryCol : mappingColumn.get(realName).entrySet())
                    {
                        String realNameCol = entryCol.getKey();
                        String modifiedNameCol = entryCol.getValue();
                        
                        if (getLogger().isDebugEnabled())
                        {
                            String tableName = _normalizeNameComponent.normalizedTableName(_sqlTablePrefix, _mappingPolicy, mappingTableName, _connection);
                            getLogger().debug("INSERT INTO {} VALUES ('{}', '{}', '{}')", tableName, i, realNameCol, modifiedNameCol);
                        }

                        stmtColumnName.setInt(1, i);
                        stmtColumnName.setString(2, realNameCol);
                        stmtColumnName.setString(3, modifiedNameCol);
                        
                        stmtColumnName.addBatch();
                        
                    }
                }
                i++;
            }
            
            stmtTableName.executeBatch();
            stmtColumnName.executeBatch();
        }
        catch (SQLException e)
        {
            getLogger().error(_i18nTranslator.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_EXPORT_LOG_ANALYZE_TABLE_FILL_MAPPING_ERROR")), e);
        }
        finally
        {
            ConnectionHelper.cleanup(stmtTableName);
            ConnectionHelper.cleanup(stmtColumnName);
        }
    }
    
    /**
     * Add all the column for each attribute in the {@link ModelItemContainer}.
     * @param modelItemContainer the attribute definitions
     * @param currentCreateTableSQLQuery the current SQL create table query
     * @param tableName the table name
     * @param columnNamePrefix the column namePrefix
     */
    protected void addColumnForContainer(ModelItemContainer modelItemContainer, StringBuilder currentCreateTableSQLQuery, String tableName, String columnNamePrefix)
    {
        for (ModelItem modelItem : modelItemContainer.getModelItems())
        {
            String name = modelItem.getName();
            String columnName = columnNamePrefix + name;
            if (modelItem instanceof ElementDefinition definition)
            {
                // simple element
                if (definition.isMultiple())
                {
                    if (_exportNoMultiValuedTable)
                    {
                        String comment = _i18nTranslator.translate(definition.getLabel(), DEFAULT_LANGUAGE_CODE_FOR_COMMENTS) + ": " + _i18nTranslator.translate(definition.getDescription(), DEFAULT_LANGUAGE_CODE_FOR_COMMENTS);
                        addColumn(currentCreateTableSQLQuery, columnName, tableName, comment, "string");
                    }
                    else
                    {
                        createQueryForMultipleAttribute(definition, tableName, tableName + "_" + columnName); 
                    }
                }
                else
                {
                    addColumnForSingleAttribute(definition, currentCreateTableSQLQuery, tableName, columnName);
                    
                    if (definition.getEnumerator() != null)
                    {        
                        createTableForEnumerator(definition, tableName, tableName + "_" + name, columnName, currentCreateTableSQLQuery);
                    }
                }
            }
            else if (modelItem instanceof CompositeDefinition)
            {
                // composite
                addColumnForContainer((CompositeDefinition) modelItem, currentCreateTableSQLQuery, tableName, columnName + "_");
            }
            else if (modelItem instanceof RepeaterDefinition)
            {
                // repeater
                String commentTable = "Repeater " + _normalizeNameComponent.normalizedColumnName(_mappingPolicy, columnName, tableName, _reservedWords, _connection) + " linked to the table " + _normalizeNameComponent.normalizedTableName(_sqlTablePrefix, _mappingPolicy, tableName, _connection);
                createQueriesForTableCreation((RepeaterDefinition) modelItem, tableName + "_" + columnName, tableName, commentTable, true);
            }
        }
    }
    
    /**
     * Add column for single attribute
     * @param definition the attribute definition
     * @param currentCreateTableSQLQuery the current SQL create table query
     * @param tableName the table name
     * @param columnName the column name
     */
    protected void addColumnForSingleAttribute(ElementDefinition definition, StringBuilder currentCreateTableSQLQuery, String tableName, String columnName)
    {
        String comment = _i18nTranslator.translate(definition.getLabel(), DEFAULT_LANGUAGE_CODE_FOR_COMMENTS) + ": " + _i18nTranslator.translate(definition.getDescription(), DEFAULT_LANGUAGE_CODE_FOR_COMMENTS);

        String type = definition.getType().getId();
        if (type.equals(ModelItemTypeConstants.USER_ELEMENT_TYPE_ID))
        {
            addColumn(currentCreateTableSQLQuery, columnName + "_login", tableName, "User login", "string");
            addColumn(currentCreateTableSQLQuery, columnName + "_population", tableName, "User population", "string");
        }
        else if (type.equals(ModelItemTypeConstants.GEOCODE_ELEMENT_TYPE_ID))
        {
            addColumn(currentCreateTableSQLQuery, columnName + "_longitude", tableName, "Longitude of " + columnName, "double");
            addColumn(currentCreateTableSQLQuery, columnName + "_latitude", tableName, "Latitude type of " + columnName, "double");
        }
        else if (type.equals(ModelItemTypeConstants.BINARY_ELEMENT_TYPE_ID) || type.equals(ModelItemTypeConstants.FILE_ELEMENT_TYPE_ID))
        {
            String actualColumnName = addColumn(currentCreateTableSQLQuery, columnName, tableName, comment, "string");
            addColumn(currentCreateTableSQLQuery, actualColumnName + "_data", tableName, "Data of " + columnName, "binary");
            addColumn(currentCreateTableSQLQuery, actualColumnName + "_mimetype", tableName, "Mime type of " + columnName, "short-string");
            addColumn(currentCreateTableSQLQuery, actualColumnName + "_size", tableName, "Size of " + columnName, "long");
            addColumn(currentCreateTableSQLQuery, actualColumnName + "_lastmodified", tableName, "Last modification date of " + columnName, "datetime");
        }
        else if (type.equals(ModelItemTypeConstants.MULTILINGUAL_STRING_ELEMENT_TYPE_ID))
        {
            for (String lang : _languageManager.getAvailableLanguages().keySet())
            {
                addColumn(currentCreateTableSQLQuery, columnName + "_" + lang, tableName, "Value of " + columnName + " for lang " + lang, "string");
            }
        }
        else
        {
            addColumn(currentCreateTableSQLQuery, columnName, tableName, comment, type);
        }
    }
    
    /**
     * Add a column
     * @param currentCreateTableSQLQuery the current SQL create table query
     * @param tableName the table name
     * @param columnName the desired column name
     * @param comment the column comment
     * @param type the column type
     * @return the actual column name, after normalization of the desired one
     */
    protected String addColumn(StringBuilder currentCreateTableSQLQuery, String columnName, String tableName, String comment, String type)
    {
        String actualColumnName = _normalizeNameComponent.normalizedColumnName(_mappingPolicy, columnName, tableName, _reservedWords, _connection);
        
        currentCreateTableSQLQuery.append(", ").append(actualColumnName);
        currentCreateTableSQLQuery.append(" ").append(convertTypeToSql(type));
        
        if (comment != null)
        {
            currentCreateTableSQLQuery.append(createCommentQuery(tableName, columnName, comment));
        }
        
        ExportTableInfo tableInfo = _tablesInfos.get(tableName);
        tableInfo.incrementNbColumns();
        
        return actualColumnName;
    }
    
    /**
     * Create table for multiple attribute
     * @param definition the attribute definition
     * @param tableParentName the name of the table for the attribute's container
     * @param tableName the table name
     */
    protected void createQueryForMultipleAttribute(ElementDefinition definition, String tableParentName, String tableName)
    {
        ExportTableInfo tableInfo = new ExportTableInfo(tableName);
        tableInfo.incrementNbColumns();
        
        String normalizeTableName = _normalizeNameComponent.normalizedTableName(_sqlTablePrefix, _mappingPolicy, tableName, _connection);
        _tablesInfos.put(tableName, tableInfo);
        
        String normalizedColumnName = _normalizeNameComponent.normalizedColumnName(_mappingPolicy, COLUMN_PARENT_TABLE_PREFIX + tableParentName, tableName, _reservedWords, _connection);
        
        StringBuilder currentCreateTableSQLQuery = new StringBuilder();
        currentCreateTableSQLQuery.append("CREATE TABLE ");
        currentCreateTableSQLQuery.append(normalizeTableName);
        currentCreateTableSQLQuery.append(" (");
        currentCreateTableSQLQuery.append(normalizedColumnName);
        currentCreateTableSQLQuery.append(" VARCHAR(245)");
        currentCreateTableSQLQuery.append(createCommentQuery(tableName, COLUMN_PARENT_TABLE_PREFIX + tableParentName, "Parent ID of the multiple attribute"));
        
        String fkName = normalizedColumnName;
        String fkTableName = _normalizeNameComponent.normalizedTableName(_sqlTablePrefix, _mappingPolicy, tableParentName, _connection);
        String fkColumnName = _normalizeNameComponent.normalizedColumnName(_mappingPolicy, "id_" + tableParentName, tableParentName, _reservedWords, _connection);
        currentCreateTableSQLQuery.append(createForeignKeyQuery(fkName, fkTableName, fkColumnName));
        
        addColumnForSingleAttribute(definition, currentCreateTableSQLQuery, tableName, definition.getName());
        
        currentCreateTableSQLQuery.append(", ");
        addSortColumn(currentCreateTableSQLQuery, tableName);
        
        String primaryKey = "pk_" + _pkIndice;
        _pkIndice++;
        
        currentCreateTableSQLQuery.append(", CONSTRAINT ");
        currentCreateTableSQLQuery.append(primaryKey);
        currentCreateTableSQLQuery.append(" PRIMARY KEY (");
        currentCreateTableSQLQuery.append(normalizedColumnName);
        currentCreateTableSQLQuery.append(", position)");
        currentCreateTableSQLQuery.append(") ");
        currentCreateTableSQLQuery.append(createEngineQuery());
        
        String comment = "Multiple attribute " + definition.getName() + " linked to the table" + _normalizeNameComponent.normalizedTableName(_sqlTablePrefix, _mappingPolicy, tableParentName, _connection);
        currentCreateTableSQLQuery.append(createCommentQuery(tableName, null, comment));
        
        tableInfo.addCreateQuery(currentCreateTableSQLQuery.toString());
    }
    
    /**
     * Add additionnal data for content (title, type, language, creator, creationDate, ....)
     * @param contentType the model
     * @param currentCreateTableSQLQuery the current SQL create table query
     * @param tableName the table name
     */
    protected void addAdditionalData(ContentType contentType, StringBuilder currentCreateTableSQLQuery, String tableName)
    {
        addColumn(currentCreateTableSQLQuery, "content_title", tableName, "Content title", "string");
        addColumn(currentCreateTableSQLQuery, "content_type", tableName, "Content type", "string");
        addColumn(currentCreateTableSQLQuery, "content_language", tableName, "Content lang", "string");
        addColumn(currentCreateTableSQLQuery, "content_creator", tableName, "Content author", "string");
        addColumn(currentCreateTableSQLQuery, "content_creationDate", tableName, "Content creation date", "datetime");
        addColumn(currentCreateTableSQLQuery, "content_lastContributor", tableName, "Content last contributor", "string");
        addColumn(currentCreateTableSQLQuery, "content_lastModificationDate", tableName, "Content last modification date", "datetime");
        addColumn(currentCreateTableSQLQuery, "content_lastValidationDate", tableName, "Content last validation date", "datetime");
        addColumn(currentCreateTableSQLQuery, "content_lastMajorValidationDate", tableName, "Content last major validation date", "datetime");
    }
    
    /**
     * Add a column to sort
     * @param currentCreateTableSQLQuery the current SQL create table query
     * @param tableName the table name
     */
    protected void addSortColumn(StringBuilder currentCreateTableSQLQuery, String tableName)
    {
        currentCreateTableSQLQuery.append("position INT");
        currentCreateTableSQLQuery.append(createCommentQuery(tableName, "position", "Order of the row"));
        
        ExportTableInfo tableInfo = _tablesInfos.get(tableName);
        tableInfo.incrementNbColumns();
    }
    
    /**
     * Add an id column which refer to another table
     * @param currentCreateTableSQLQuery the current SQL create table query
     * @param tableParentName the table parent name
     * @param tableName the table name
     */
    protected void addColumnParentId(StringBuilder currentCreateTableSQLQuery, String tableParentName, String tableName)
    {
        currentCreateTableSQLQuery.append(_normalizeNameComponent.normalizedColumnName(_mappingPolicy, COLUMN_PARENT_TABLE_PREFIX + tableParentName, tableName, _reservedWords, _connection));
        currentCreateTableSQLQuery.append(" VARCHAR(250)"); //TODO NOT NULL
        currentCreateTableSQLQuery.append(createCommentQuery(tableName, COLUMN_PARENT_TABLE_PREFIX + tableParentName, "Parent table ID " + tableParentName));

        String fkName = _normalizeNameComponent.normalizedColumnName(_mappingPolicy, COLUMN_PARENT_TABLE_PREFIX + tableParentName, tableName, _reservedWords, _connection);
        String fkTableName = _normalizeNameComponent.normalizedTableName(_sqlTablePrefix, _mappingPolicy, tableParentName, _connection);
        String fkColumnName = _normalizeNameComponent.normalizedColumnName(_mappingPolicy, "id_" + tableParentName, tableParentName, _reservedWords, _connection);
        currentCreateTableSQLQuery.append(createForeignKeyQuery(fkName, fkTableName, fkColumnName));
        
        ExportTableInfo tableInfo = _tablesInfos.get(tableName);
        tableInfo.incrementNbColumns();
    }
    
    /**
     * Create the query to add comment to a table or a column
     * @param table the table name
     * @param column the column name
     * @param comments the comments
     * @return the query to add comment to a table or a column
     */
    protected String createCommentQuery(String table, String column, String comments)
    {
        String normalizedComment = _normalizeNameComponent.normalizedComment(comments, StringUtils.isEmpty(column) ? _commentTableMaxLength : _commentColumnMaxLength, _connection);
        
        String commentSql = "";
        if (_databaseType.equals(ConnectionHelper.DATABASE_MYSQL))
        {
            commentSql = " COMMENT '" + normalizedComment + "'";
        }
        else if (_databaseType.equals(ConnectionHelper.DATABASE_ORACLE)) //TODO
        {
            /*if (StringUtils.isNotEmpty(table)) 
            {
                if (StringUtils.isEmpty(column)) 
                {
//                    _tablesInfos.get(table).addCommentQuery("COMMENT ON TABLE " + _normaliseNameComponent.normalizedTableName(_sqlTablePrefix, _mappingPolicy, table, _connection) + " IS '" + normalizedComment + "'");
                }
                else
                {
//                    _tablesInfos.get(table).addCommentQuery("COMMENT ON COLUMN " + _normaliseNameComponent.normalizedTableName(_sqlTablePrefix, _mappingPolicy, table, _connection) + "." + _normaliseNameComponent.normalizedColumnName(_mappingPolicy, column, table, _connection) + " IS '" + normalizedComment + "'");
                }
            }*/
        }
        
        return commentSql;
    }
    
    /**
     * Create the query to add foreign key
     * @param fkName the foreign key name
     * @param tableName the table name 
     * @param columnName the column name 
     * @return the query to add comment to a table or a column
     */
    protected String createForeignKeyQuery(String fkName, String tableName, String columnName)
    {
        StringBuilder foreignKeySQLQuery = new StringBuilder();
        foreignKeySQLQuery.append(", CONSTRAINT fk_");
        foreignKeySQLQuery.append(_fkIndice);
        _fkIndice++;

        foreignKeySQLQuery.append(" FOREIGN KEY (");
        foreignKeySQLQuery.append(fkName);
        foreignKeySQLQuery.append(")");
        foreignKeySQLQuery.append(" REFERENCES ");
        foreignKeySQLQuery.append(tableName);
        foreignKeySQLQuery.append(" (");
        foreignKeySQLQuery.append(columnName);
        foreignKeySQLQuery.append(")");
        
        return foreignKeySQLQuery.toString();
    }
    
    /**
     * Create the query to add engine
     * @return the query to add engine
     */
    protected String createEngineQuery()
    {
        if (_databaseType.equals(ConnectionHelper.DATABASE_MYSQL))
        {
            return "ENGINE=" + MYSQL_CONTENT_EXPORT_ENGINE + " ROW_FORMAT=COMPRESSED DEFAULT CHARACTER SET " + MYSQL_CONTENT_EXPORT_CHARSET + " ";
        }
        else if (_databaseType.equals(ConnectionHelper.DATABASE_ORACLE))
        {
            return "STORAGE (INITIAL 8K NEXT 8K)";
        }

        return "";
    }
    
    /**
     * Create the query to add primary key
     * @return the query to add primary key
     */
    protected String createPrimaryKeyQuery()
    {
        return " PRIMARY KEY NOT NULL";
    }

    /**
     * Return the sql type corresponding to the attribute and the database type.<br>
     * Default to the sql type corresponding to "string" in case of unknown type.
     * @param type the attribute's type
     * @return the sql type
     */
    protected String convertTypeToSql(String type)
    {
        Map<String, String> mapping = null;
        if (_databaseType.equals(ConnectionHelper.DATABASE_MYSQL))
        {
            mapping = _mapping.get("mysql");
        }
        else if (_databaseType.equals(ConnectionHelper.DATABASE_ORACLE))
        {
            mapping = _mapping.get("oracle");
        }
        
        return mapping != null ? mapping.getOrDefault(type, mapping.get("string")) : "";
    }
    
    /**
     * Execute SQL queries
     * @throws SQLException if a sql error occurred
     * @throws IOException if an IO error occurred
     */
    protected void executeQueries() throws SQLException, IOException
    {
        int nbTotalTable = _getNbTable();

        boolean isInfoEnabled = getLogger().isInfoEnabled();
        if (isInfoEnabled)
        {
            List<String> i18nParams = new ArrayList<>();
            i18nParams.add(String.valueOf(nbTotalTable));
            getLogger().info(_i18nTranslator.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_EXPORT_LOG_ANALYZE_TABLE_CREATE_BEGINNING", i18nParams)));
        }

        int nbTableCreated = 0;
        int limitPourcentReport = 10;
        try
        {
            for (Entry<String, ExportTableInfo> entry : _tablesInfos.entrySet()) 
            {
                ExportTableInfo tableInfo = entry.getValue();
                
                List<String> queries = tableInfo.getCreateQueries();
                _executeQueries(queries);
                nbTableCreated += queries.size();
                
                int pourcent = nbTableCreated * 100 / nbTotalTable;
                if (pourcent >= limitPourcentReport)
                {
                    if (isInfoEnabled)
                    {
                        List<String> i18nParams = new ArrayList<>();
                        i18nParams.add(String.valueOf(limitPourcentReport));
                        getLogger().info(_i18nTranslator.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_EXPORT_LOG_ANALYZE_TABLE_CREATE_ADVANCE", i18nParams)));
                    }
                    limitPourcentReport += 10;
                }
                
                _executeQueries(tableInfo.getCommentQueries());
                _executeQueries(tableInfo.getInsertQueries());
            }
        }
        catch (IOException e)
        {
            getLogger().error(_i18nTranslator.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_EXPORT_LOG_ANALYZE_TABLE_GET_FILE_ERROR")), e);
        }
        finally
        {
            if (isInfoEnabled)
            {
                List<String> i18nParams = new ArrayList<>();
                i18nParams.add(String.valueOf(nbTableCreated));
                getLogger().info(_i18nTranslator.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_EXPORT_LOG_ANALYZE_TABLE_CREATE_FINISH", i18nParams)));
            }
        }
    }
    
    private int _getNbTable()
    {
        int nbTable = 0;
        for (Entry<String, ExportTableInfo> entry : _tablesInfos.entrySet())
        {
            ExportTableInfo tableInfo = entry.getValue();
            
            List<String> listQuery = tableInfo.getCreateQueries();
            nbTable += listQuery.size();
        }
        
        return nbTable;
    }
    
    /**
     * Execute query list
     * @param listQuery the list of query to execute
     * @throws SQLException if a sql error occurred
     * @throws IOException if an IO error occurred
     */
    protected void _executeQueries(List<String> listQuery) throws SQLException, IOException
    {
        for (String query : listQuery)
        {
            getLogger().debug(query);

            PreparedStatement stmt = null;
            try
            {
                stmt = _connection.prepareStatement(query);
                stmt.execute();
            }
            catch (SQLException e)
            {
                throw new SQLException("The SQL query failed : " + query, e);
            }
            finally
            {
                // Close the connection resources
                ConnectionHelper.cleanup(stmt);
            }
        }
    }
    
    /**
     * Prepare INSERT statement
     * @param tableName the table name
     * @return the INSERT preparedStatement
     * @throws SQLException if a sql error occurred
     */
    protected PreparedStatement getInsertPreparedStatementFromTableName(String tableName) throws SQLException
    {
        StringBuilder sql = new StringBuilder();
        sql.append("INSERT INTO ");
        sql.append(_normalizeNameComponent.normalizedTableName(_sqlTablePrefix, _mappingPolicy, tableName, _connection));
        sql.append(" VALUES ( ?");
        
        ExportTableInfo tableInfo = _tablesInfos.get(tableName);

        for (int i = 1; i < tableInfo.getNbColumns(); i++)
        {
            sql.append(", ?");
        }
        sql.append(")");
        
        PreparedStatement stmt = _connection.prepareStatement(sql.toString());
        return stmt;
    }
}
