/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.export.sql;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.datasource.ConnectionHelper;
import org.ametys.core.util.I18nUtils;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Delete existing table for export
 */
public class DeleteSqlTableComponent extends AbstractLogEnabled implements Component, Serviceable
{
    /** The component role */
    public static final String ROLE = DeleteSqlTableComponent.class.getName();

    /** The i18n translator. */
    protected I18nUtils _i18nTranslator;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _i18nTranslator = (I18nUtils) manager.lookup(I18nUtils.ROLE);
    }
    
    /**
     * Delete all tables of a preceding content export
     * @param exportConfiguration the content export configuration
     * @throws SQLException if an error occurred while deleting tables
     */
    public void deleteExistingSqlTables(ExportConfiguration exportConfiguration) throws SQLException
    {
        String tablePrefix = exportConfiguration.getTablePrefix();
        
        Connection connection = null;
        ResultSet rs = null;
        try
        {
            String datasourceId = Config.getInstance().getValue("org.ametys.plugins.contentio.content.export.datasource");
            connection = ConnectionHelper.getConnection(datasourceId);
            String datatype = ConnectionHelper.getDatabaseType(connection);
            
            if (datatype.equals(ConnectionHelper.DATABASE_ORACLE))
            {
                tablePrefix = tablePrefix.toUpperCase();
            }
            
            // Request to select all table whose name begins by '_sqlTablePrefix'
            String[] tableType = {"TABLE"};
            rs = connection.getMetaData().getTables(connection.getCatalog(), connection.getSchema(), tablePrefix + "%", tableType);
            
            if (datatype.equals(ConnectionHelper.DATABASE_MYSQL))
            {
                _deleteMysqlTable(rs, connection);
            }
            else if (datatype.equals(ConnectionHelper.DATABASE_ORACLE))
            {
                _deleteOracleTable(rs, connection);
            }
        }
        catch (Exception e)
        {
            getLogger().error(_i18nTranslator.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_EXPORT_LOG_DELETE_ERROR_SQL")), e);
        }
        finally
        {
            // Close the connection resources
            ConnectionHelper.cleanup(rs);
            ConnectionHelper.cleanup(connection);
        }
    }
    
    private void _deleteMysqlTable(ResultSet rs, Connection connection) throws SQLException
    {
        StringBuilder dropRequest = new StringBuilder();
        int nbTable = 0;
        if (rs.next())
        {
            dropRequest.append("DROP TABLE ");
            dropRequest.append(rs.getString("table_name")); 
            nbTable++;
            while (rs.next()) 
            {
                dropRequest.append(", ");
                dropRequest.append(rs.getString("table_name")); 
                nbTable++;
            }
        }
        
        if (getLogger().isInfoEnabled())
        {
            List<String> i18nParams = new ArrayList<>();
            i18nParams.add(String.valueOf(nbTable));
            getLogger().info(_i18nTranslator.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_EXPORT_LOG_DELETE_NB_TABLE", i18nParams)));
        }
        
        PreparedStatement stmt = null;
        try 
        {
            if (StringUtils.isNotEmpty(dropRequest.toString()))
            {
                stmt = connection.prepareStatement("SET FOREIGN_KEY_CHECKS = 0");
                stmt.execute();
                ConnectionHelper.cleanup(stmt);
                
                stmt = connection.prepareStatement(dropRequest.toString());
                stmt.execute();
                ConnectionHelper.cleanup(stmt);
                
                stmt = connection.prepareStatement("SET FOREIGN_KEY_CHECKS = 1");
                stmt.execute();
            }
        }
        finally
        {
            ConnectionHelper.cleanup(stmt);
        }

        if (getLogger().isInfoEnabled())
        {
            List<String> i18nParams = new ArrayList<>();
            i18nParams.add(String.valueOf(nbTable));
            getLogger().info(_i18nTranslator.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_EXPORT_LOG_DELETE_FINISH", i18nParams)));
        }
    }
    
    private void _deleteOracleTable(ResultSet rs, Connection connection) throws SQLException
    {
        PreparedStatement stmt = null;
        try
        {
            stmt = connection.prepareStatement("ALTER SESSION SET RECYCLEBIN=OFF");
            stmt.execute();
        }
        finally
        {
            ConnectionHelper.cleanup(stmt);
        }
        
        StringBuilder dropRequest = new StringBuilder();
        List<String> listRequest = new ArrayList<>();

        if (rs.next())
        {
            dropRequest.append("DROP TABLE ");
            dropRequest.append(rs.getString("table_name"));
            dropRequest.append(" CASCADE CONSTRAINTS");
            listRequest.add(dropRequest.toString());
            
            while (rs.next()) 
            {
                dropRequest = new StringBuilder();
                
                dropRequest.append("DROP TABLE ");
                dropRequest.append(rs.getString("table_name"));
                dropRequest.append(" CASCADE CONSTRAINTS");
                listRequest.add(dropRequest.toString());
            }
            
        }
        
        int nbTable = listRequest.size();
        
        boolean isInfoEnabled = getLogger().isInfoEnabled();
        boolean isWarnEnabled = getLogger().isWarnEnabled();
        
        if (isInfoEnabled)
        {
            List<String> i18nParams = new ArrayList<>();
            i18nParams.add(String.valueOf(nbTable));
            getLogger().info(_i18nTranslator.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_EXPORT_LOG_DELETE_NB_TABLE_ORACLE", i18nParams)));
        }
        
        int nbTableDeleted = 0;
        int pourcentLimit = 10;
        for (String request : listRequest)
        {
            getLogger().debug(request);
            
            try 
            {
                stmt = connection.prepareStatement(request);
                stmt.execute();
                nbTableDeleted++;
            }
            catch (Exception e)
            {
                if (isWarnEnabled)
                {
                    List<String> i18nParams = new ArrayList<>();
                    i18nParams.add(String.valueOf(request));
                    getLogger().warn(_i18nTranslator.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_EXPORT_LOG_DELETE_ERROR_TABLE_ORACLE", i18nParams)), e);
                }
            }
            finally
            {
                ConnectionHelper.cleanup(stmt);
            }

            int pourcent = nbTableDeleted * 100 / nbTable;
            if (pourcent >= pourcentLimit)
            {
                if (isInfoEnabled)
                {
                    List<String> i18nParams = new ArrayList<>();
                    i18nParams.add(String.valueOf(pourcentLimit));
                    getLogger().info(_i18nTranslator.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_EXPORT_LOG_DELETE_ADVANCE_ORACLE", i18nParams)));
                }
                pourcentLimit += 10;
            }
        }
        
        if (isInfoEnabled)
        {
            List<String> i18nParams = new ArrayList<>();
            i18nParams.add(String.valueOf(nbTableDeleted));
            getLogger().info(_i18nTranslator.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_EXPORT_LOG_DELETE_FINISH", i18nParams)));
        }
    }

}
