/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.export.sql;

import java.io.IOException;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * This component handles content export in database
 */
public class ExportManager extends AbstractLogEnabled implements Configurable, Component, Serviceable
{
    /** The component role */
    public static final String ROLE = ExportManager.class.getName();

    /** The default prefix */
    public static final String DEFAULT_TABLE_PREFIX = "AmetysExport";
    
    /** Connection pool name */
    public static final String CONTENT_EXPORT_POOL_NAME = "content.export.jdbc.pool";
    
    /** The name of the table for datas from rich text */
    public static final String RICH_TEXT_DATA_TABLE_NAME = "Ametys_RichTextImages";
    
    /** The name of the table for mapping table name */
    public static final String MAPPING_TABLE_NAME = "Ametys_tableMapping";
    
    /** The name of the table for mapping column name */
    public static final String MAPPING_COLUMN_NAME = "Ametys_columnMapping";
    
    /** The name of the table for mapping column name */
    public static final String CONTENT_TABLE_NAME = "Ametys_AllContents";
    
    /** The default separator */
    public static final String DEFAULT_SEPARATOR = ", ";
    
    /** Content type extension point. */
    protected ContentTypeExtensionPoint _contentTypeExtensionPoint;
  
    /** The i18n translator. */
    protected I18nUtils _i18nTranslator;
    
    /** The delete sql table component. */
    protected DeleteSqlTableComponent _deleteSqlTableComponent;
    
    /** The create sql table component. */
    protected CreateSqlTableComponent _createSqlTableComponent;
    
    /** The fill sql table component. */
    protected FillSqlTableComponent _fillSqlTableComponent;
    
    private ExportConfiguration _exportConfiguration;
    
    @Override
    public void service(ServiceManager sManager) throws ServiceException
    {
        _contentTypeExtensionPoint = (ContentTypeExtensionPoint) sManager.lookup(ContentTypeExtensionPoint.ROLE);
        _i18nTranslator = (I18nUtils) sManager.lookup(I18nUtils.ROLE);
        _deleteSqlTableComponent = (DeleteSqlTableComponent) sManager.lookup(DeleteSqlTableComponent.ROLE);
        _createSqlTableComponent = (CreateSqlTableComponent) sManager.lookup(CreateSqlTableComponent.ROLE);
        _fillSqlTableComponent = (FillSqlTableComponent) sManager.lookup(FillSqlTableComponent.ROLE);
    }
    
    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        _exportConfiguration = new ExportConfiguration();
        
        // Set the table prefix
        String tablePrefix = configuration.getAttribute("prefix", DEFAULT_TABLE_PREFIX);
        if (StringUtils.isNotEmpty(tablePrefix) && !StringUtils.endsWith(tablePrefix, "_"))
        {
            tablePrefix += "_";
        }
        _exportConfiguration.setTablePrefix(tablePrefix);
        
        String exportOnlyValidatedContentAsString = configuration.getAttribute("exportOnlyValidatedContent", "false");
        _exportConfiguration.setExportOnlyValidatedContent("true".equals(exportOnlyValidatedContentAsString));
        
        String exportNoMultiValuedTableAsString = configuration.getAttribute("exportNoMultiValuedTable", "false");
        _exportConfiguration.setExportNoMultiValuedTable("true".equals(exportNoMultiValuedTableAsString));
        
        String separator = configuration.getAttribute("separator", DEFAULT_SEPARATOR);
        _exportConfiguration.setSeparator(separator);
        
        // Set the content type to export
        _configureContentTypeToExport(configuration, tablePrefix);
        
        // Set the sql mapping
        _configureSQLMapping(configuration);
        
        // Set the reserved words
        _configureReservedWords(configuration);
        
        // Set the mapping policy
        String mappingPolicy = configuration.getChild("mapping-policy").getAttribute("value", "FULL");
        _exportConfiguration.setMappingPolicy(mappingPolicy);
    }

    /**
     * Configure the content type to export
     * @param configuration the configuration
     * @param tablePrefix the table prefix
     * @throws ConfigurationException if an error occurred
     */
    protected void _configureContentTypeToExport(Configuration configuration, String tablePrefix) throws ConfigurationException
    {
        Map<String, String> contentTypesToExport = new HashMap<>();
        for (Configuration contentConf : configuration.getChild("contents").getChildren("content"))
        {
            String cTypeId = contentConf.getAttribute("id");
            String name = contentConf.getAttribute("name", null);
            if (StringUtils.isNotBlank(name))
            {
                contentTypesToExport.put(cTypeId, tablePrefix + name);
            }
            else
            {
                String defaultTableName = cTypeId.substring(cTypeId.lastIndexOf(".") + 1);
                contentTypesToExport.put(cTypeId, tablePrefix + defaultTableName);
            }
        }
        
        if (contentTypesToExport.isEmpty())
        {
            for (String cTypeId : _contentTypeExtensionPoint.getExtensionsIds())
            {
                String defaultTableName = cTypeId.substring(cTypeId.lastIndexOf(".") + 1);
                contentTypesToExport.put(cTypeId, tablePrefix + defaultTableName);
            }
        }
        
        _exportConfiguration.setContentTypesToExport(contentTypesToExport);
    }

    /**
     * Configure the reserved words
     * @param configuration the configuration
     * @throws ConfigurationException if an error occurred
     */
    protected void _configureReservedWords(Configuration configuration) throws ConfigurationException
    {
        boolean override = "true".equals(configuration.getChild("reserved-words").getAttribute("override", "false"));
        
        Map<String, Map<String, String>> reservedWords = new HashMap<>();

        Configuration mysqlReservedWords = configuration.getChild("reserved-words").getChild("mysql");
        Map<String, String> mysqlReservedWordsMap = new HashMap<>();
        if (!override)
        {
            mysqlReservedWordsMap = ReservedWordsUtils.RESERVED_WORDS_MYSQL;
        }
        for (Configuration wordConf : mysqlReservedWords.getChildren("word"))
        {
            mysqlReservedWordsMap.put(wordConf.getAttribute("name"), wordConf.getAttribute("alias"));
        }
        if (mysqlReservedWordsMap.isEmpty())
        {
            mysqlReservedWordsMap = ReservedWordsUtils.RESERVED_WORDS_MYSQL;
        }
        reservedWords.put(ReservedWordsUtils.MYSQL_KEY, mysqlReservedWordsMap);
        
        Configuration oracleReservedWords = configuration.getChild("reserved-words").getChild("oracle");
        Map<String, String> oracleReservedWordsMap = new HashMap<>();
        if (!override)
        {
            oracleReservedWordsMap = ReservedWordsUtils.RESERVED_WORDS_ORACLE;
        }
        for (Configuration wordConf : oracleReservedWords.getChildren("word"))
        {
            oracleReservedWordsMap.put(wordConf.getAttribute("name"), wordConf.getAttribute("alias"));
        }
        if (oracleReservedWordsMap.isEmpty())
        {
            oracleReservedWordsMap = ReservedWordsUtils.RESERVED_WORDS_ORACLE;
        }
        reservedWords.put(ReservedWordsUtils.ORACLE_KEY, oracleReservedWordsMap);
        
        _exportConfiguration.setReservedWords(reservedWords);
    }

    /**
     * Configure the sql mapping
     * @param configuration the configuration
     */
    protected void _configureSQLMapping(Configuration configuration)
    {
        Map<String, Map<String, String>> mappingSql = new HashMap<>();
        
        Map<String, String> mappingSqlMetadata = new HashMap<>();
        mappingSqlMetadata.put("short-string", configuration.getChild("sql").getChild("mysql").getChild("short-string").getValue("VARCHAR(255)"));
        mappingSqlMetadata.put("string", configuration.getChild("sql").getChild("mysql").getChild("string").getValue("TEXT"));
        mappingSqlMetadata.put("long", configuration.getChild("sql").getChild("mysql").getChild("long").getValue("BIGINT"));
        mappingSqlMetadata.put("boolean", configuration.getChild("sql").getChild("mysql").getChild("boolean").getValue("TINYINT"));
        mappingSqlMetadata.put("date", configuration.getChild("sql").getChild("mysql").getChild("date").getValue("DATE"));
        mappingSqlMetadata.put("datetime", configuration.getChild("sql").getChild("mysql").getChild("datetime").getValue("DATETIME"));
        mappingSqlMetadata.put("double", configuration.getChild("sql").getChild("mysql").getChild("double").getValue("DOUBLE"));
        mappingSqlMetadata.put("rich-text", configuration.getChild("sql").getChild("mysql").getChild("rich-text").getValue("MEDIUMTEXT"));
        mappingSqlMetadata.put("content", configuration.getChild("sql").getChild("mysql").getChild("content").getValue("VARCHAR(512)"));
        mappingSqlMetadata.put("file", configuration.getChild("sql").getChild("mysql").getChild("file").getValue("LONGBLOB"));
        mappingSqlMetadata.put("binary", configuration.getChild("sql").getChild("mysql").getChild("binary").getValue("LONGBLOB"));
        mappingSql.put("mysql", mappingSqlMetadata);
        
        Map<String, String> mappingOracleMetadata = new HashMap<>();
        mappingOracleMetadata.put("short-string", configuration.getChild("sql").getChild("mysql").getChild("short-string").getValue("VARCHAR(255)"));
        mappingOracleMetadata.put("string", configuration.getChild("sql").getChild("oracle").getChild("string").getValue("VARCHAR(4000)"));
        mappingOracleMetadata.put("long", configuration.getChild("sql").getChild("oracle").getChild("long").getValue("NUMBER"));
        mappingOracleMetadata.put("boolean", configuration.getChild("sql").getChild("oracle").getChild("boolean").getValue("CHAR(1)"));
        mappingOracleMetadata.put("date", configuration.getChild("sql").getChild("oracle").getChild("date").getValue("DATE"));
        mappingOracleMetadata.put("datetime", configuration.getChild("sql").getChild("oracle").getChild("datetime").getValue("DATE"));
        mappingOracleMetadata.put("double", configuration.getChild("sql").getChild("oracle").getChild("double").getValue("NUMBER"));
        mappingOracleMetadata.put("rich-text", configuration.getChild("sql").getChild("oracle").getChild("rich-text").getValue("VARCHAR(4000)"));
        mappingOracleMetadata.put("content", configuration.getChild("sql").getChild("oracle").getChild("content").getValue("VARCHAR(512)"));
        mappingOracleMetadata.put("file", configuration.getChild("sql").getChild("oracle").getChild("file").getValue("BLOB"));
        mappingOracleMetadata.put("binary", configuration.getChild("sql").getChild("oracle").getChild("binary").getValue("BLOB"));
        mappingSql.put("oracle", mappingOracleMetadata);
        
        _exportConfiguration.setMappingSql(mappingSql);
    }

    /**
     * Export the configured content types to the CONTENT_EXPORT_POOL_NAME db pool
     * @throws SQLException If a sql error occurred
     * @throws ContextException  if a context error occurred
     * @throws AmetysRepositoryException if a ametys repository error occurred
     * @throws IOException if an IO error occurred
     */
    public void export() throws SQLException, AmetysRepositoryException, ContextException, IOException
    {
        boolean isInfoEnabled = getLogger().isInfoEnabled();
        if (isInfoEnabled)
        {
            getLogger().info(_i18nTranslator.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_EXPORT_LOG_BEGIN")));
            getLogger().info(_i18nTranslator.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_EXPORT_LOG_INIT_BEGIN")));
        }
        
        // Initialization
        initialize();
        
        if (isInfoEnabled)
        {
            getLogger().info(_i18nTranslator.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_EXPORT_LOG_INIT_END")));
            getLogger().info(_i18nTranslator.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_EXPORT_LOG_DELETE_BEGIN")));
        }
        
        // First delete all tables
        _deleteSqlTableComponent.deleteExistingSqlTables(_exportConfiguration);

        if (isInfoEnabled)
        {
            getLogger().info(_i18nTranslator.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_EXPORT_LOG_CREATE_BEGIN")));
        }
        
        // Then create all tables
        Map<String, ExportTableInfo> tableInfo = _createSqlTableComponent.createTables(_exportConfiguration);
        
        if (isInfoEnabled)
        {
            getLogger().info(_i18nTranslator.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_EXPORT_LOG_FILL_BEGIN")));
        }
        
        // Finally insert values
        _fillSqlTableComponent.fillTable(_exportConfiguration, tableInfo);
    }
    
    /**
     * Initialization
     */
    protected void initialize()
    {
        Map<String, String> contents = _exportConfiguration.getContentTypesToExport();

        if (contents.isEmpty())
        {
            String prefix = _exportConfiguration.getTablePrefix();
            for (String contentTypeId : _contentTypeExtensionPoint.getExtensionsIds())
            {
                String defaultTableName = prefix + contentTypeId.substring(contentTypeId.lastIndexOf(".") + 1);
                contents.put(contentTypeId, defaultTableName);
            }
            
            _exportConfiguration.setContentTypesToExport(contents);
        }
    }
    
    /**
     * Get export configuration
     * @return the export configuration
     */
    public ExportConfiguration getExportConfiguration()
    {
        return _exportConfiguration;
    }
    
    /**
     * Set export configuration
     * @param exportConfiguration the export configuration
     */
    public void setExportConfiguration(ExportConfiguration exportConfiguration)
    {
        this._exportConfiguration = exportConfiguration;
    }
}
