/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.export.sql;

import java.io.IOException;
import java.io.InputStream;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.sql.Types;
import java.time.LocalDate;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.input.AutoCloseInputStream;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.LocaleUtils;

import org.ametys.cms.CmsConstants;
import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.data.File;
import org.ametys.cms.data.Geocode;
import org.ametys.cms.data.NamedResource;
import org.ametys.cms.data.RichText;
import org.ametys.cms.data.type.ModelItemTypeConstants;
import org.ametys.cms.languages.LanguagesManager;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.core.datasource.ConnectionHelper;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.plugins.repository.data.holder.group.ModelAwareComposite;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeaterEntry;
import org.ametys.plugins.repository.metadata.MultilingualString;
import org.ametys.plugins.repository.model.CompositeDefinition;
import org.ametys.plugins.repository.model.RepeaterDefinition;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.version.VersionableAmetysObject;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ModelItemContainer;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Fill sql table component
 */
public class FillSqlTableComponent extends AbstractLogEnabled implements Component, Serviceable
{
    /** The component role */
    public static final String ROLE = FillSqlTableComponent.class.getName();
    
    private static final String _EXCLUDE_XML_TAGS = "<(.*?)>";
    
    /** Content type extension point. */
    protected ContentTypeExtensionPoint _contentTypeExtensionPoint;
  
    /** The ametys object resolver. */
    protected AmetysObjectResolver _resolver;
  
    /** The normalise name component. */
    protected NormalizeNameComponent _normalizeNameComponent;
    
    /** The configured list of content types to export associated to the SQL table name to use */
    protected Map<String, String> _contentTypesToExport;
    
    /** The i18n translator. */
    protected I18nUtils _i18nTranslator;
    
    /** The language manager */
    protected LanguagesManager _languageManager;
    
    /** The user manager */
    protected UserManager _userManager;

    private Connection _connection;
    private String _sqlTablePrefix;
    private String _sqlPrefixConf;
    private Map<String, ExportTableInfo> _tablesInfos;
    private String _mappingPolicy;
    private LinkedList<PreparedStatement> _stmtList;
    private boolean _exportOnlyValidatedContent;
    private boolean _exportNoMultiValuedTable;
    private String _separator;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _contentTypeExtensionPoint = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _normalizeNameComponent = (NormalizeNameComponent) manager.lookup(NormalizeNameComponent.ROLE);
        _i18nTranslator = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _languageManager = (LanguagesManager) manager.lookup(LanguagesManager.ROLE);
        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
    }
    
    /**
     * Fill table with contents
     * @param exportConfiguration the content export configuration
     * @param tableInfo the map of table information
     * @throws SQLException if a sql error occurred
     * @throws AmetysRepositoryException if an ametys repository error occurred
     * @throws IOException if an IO error occurred
     */
    public synchronized void fillTable(ExportConfiguration exportConfiguration, Map<String, ExportTableInfo> tableInfo) throws SQLException, AmetysRepositoryException, IOException
    {
        // Get from configuration
        _sqlTablePrefix = exportConfiguration.getTablePrefix();
        _sqlPrefixConf = exportConfiguration.getTablePrefix();
        _mappingPolicy = exportConfiguration.getMappingPolicy();
        _contentTypesToExport = exportConfiguration.getContentTypesToExport();
        _exportOnlyValidatedContent = exportConfiguration.exportOnlyValidatedContent();
        _exportNoMultiValuedTable = exportConfiguration.exportNoMultiValuedTable();
        _separator = exportConfiguration.getSeparator();

        // Initialization
        _tablesInfos = tableInfo;
        _stmtList = new LinkedList<>();
        
        try
        {
            String datasourceId = Config.getInstance().getValue("org.ametys.plugins.contentio.content.export.datasource");
            _connection = ConnectionHelper.getConnection(datasourceId);
            insertValues();
        }
        finally
        {
            ConnectionHelper.cleanup(_connection);
            _sqlTablePrefix = null;
            _sqlPrefixConf = null;
            _mappingPolicy = null;
            _contentTypesToExport = null;
            _separator = null;
            _stmtList = null;
        }
    }
    
    private void executeInsert() throws SQLException
    {
        for (PreparedStatement stmt : _stmtList)
        {
            try
            {
                if (getLogger().isDebugEnabled())
                {
                    getLogger().debug("Query : {}", stmt.toString());
                }
                stmt.executeBatch();
            }
            catch (SQLException e)
            {
                throw new SQLException(_i18nTranslator.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_EXPORT_LOG_FILL_ERROR_SQL")), e);
            }
            finally
            {
                ConnectionHelper.cleanup(stmt);
            }
        }
        
        _stmtList.clear();
    }

    /**
     * Fill the SQL tables with values from JCR
     * @throws SQLException if a sql error occurred
     * @throws AmetysRepositoryException if an ametys repository error occurred
     * @throws IOException if an IO error occurred
     */
    protected void insertValues() throws SQLException, AmetysRepositoryException, IOException
    {
        boolean isInfoEnabled = getLogger().isInfoEnabled();
        
        int nbTotalTypeContenu = _contentTypesToExport.entrySet().size();
        if (isInfoEnabled && nbTotalTypeContenu != 0)
        {
            List<String> i18nParams = new ArrayList<>();
            i18nParams.add(String.valueOf(nbTotalTypeContenu));
            getLogger().info(_i18nTranslator.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_EXPORT_LOG_FILL_ANALYZE_BEGIN", i18nParams)));
        }
        
        int nbTypeContenu = 0;
        int pourcentMax = 10;
        for (Entry<String, String> entry : _contentTypesToExport.entrySet())
        {
            String contentTypeId = entry.getKey();
            ContentType contentType = _contentTypeExtensionPoint.getExtension(contentTypeId);
            
            if (!contentType.isAbstract())
            {
                _sqlTablePrefix = entry.getValue();
                
                if (isInfoEnabled)
                {
                    List<String> i18nParams = new ArrayList<>();
                    i18nParams.add(String.valueOf(entry.getValue()));
                    getLogger().info(_i18nTranslator.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_EXPORT_LOG_FILL_CONTENT_TYPE", i18nParams)));
                }
                
                fillTableForContentType(contentTypeId);
            }
            
            int pourcent = nbTypeContenu * 100 / nbTotalTypeContenu;
            if (pourcent >= pourcentMax)
            {
                if (isInfoEnabled)
                {
                    List<String> i18nParams = new ArrayList<>();
                    i18nParams.add(String.valueOf(pourcentMax));
                    getLogger().info(_i18nTranslator.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_EXPORT_LOG_FILL_ANALYZE", i18nParams)));
                }
                pourcentMax += 10;
            }
            
            nbTypeContenu++;
            _sqlTablePrefix = _sqlPrefixConf;
        }
        
        if (isInfoEnabled)
        {
            getLogger().info(_i18nTranslator.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_EXPORT_LOG_FILL_END")));
        }
    }
    
    /**
     * Fill values from JCR request with specific content type
     * @param contentTypeId the content type id
     * @throws SQLException if a sql error occurred
     * @throws AmetysRepositoryException if an ametys repository error occurred
     * @throws IOException if an IO error occurred
     */
    protected void fillTableForContentType(String contentTypeId) throws SQLException, AmetysRepositoryException, IOException
    {
        String tableName = getTableName(contentTypeId);
        
        ContentTypeExpression typeE = new ContentTypeExpression(Operator.EQ, contentTypeId);
        String xpath = ContentQueryHelper.getContentXPathQuery(typeE);
        AmetysObjectIterable<Content> contents = _resolver.query(xpath);
        
        ContentType cType = _contentTypeExtensionPoint.getExtension(contentTypeId);

        boolean isInfoEnabled = getLogger().isInfoEnabled();
        
        int nbTotalContent = (int) contents.getSize();
        if (isInfoEnabled && nbTotalContent != 0)
        {
            List<String> i18nParams = new ArrayList<>();
            i18nParams.add(String.valueOf(nbTotalContent));
            getLogger().info(_i18nTranslator.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_EXPORT_LOG_FILL_NB_CONTENT_MAX", i18nParams)));
        }
        
        int nbContent = 0;
        int pourcentMax = 10;
        for (Content content : contents)
        {
            List<String> labels = Arrays.asList(((VersionableAmetysObject) content).getAllLabels());
            boolean exportContent = true;
            if (_exportOnlyValidatedContent)
            {
                if (labels.contains(CmsConstants.LIVE_LABEL))
                {
                    ((VersionableAmetysObject) content).switchToLabel(CmsConstants.LIVE_LABEL);
                }
                else
                {
                    exportContent = false;
                }
            }
            
            if (exportContent)
            {
                String id = content.getId();
                PreparedStatement stmt = null;
                try
                {
                    stmt = getInsertPreparedStatementFromTableName(_sqlTablePrefix, tableName, _mappingPolicy);
                    stmt.setString(1, id);
                    
                    getLogger().debug(" with id: {}", id);
                    
                    ExportCounter fillIndex = new ExportCounter(2);
                    fillValues(fillIndex, cType, content.getDataHolder(), tableName, "", id, stmt);
                    fillAdditionalData(fillIndex, content, stmt);
                    stmt.execute();
                }
                finally
                {
                    ConnectionHelper.cleanup(stmt);
                }
                
                _fillContentTable(id, tableName);
                executeInsert();
                
                int pourcent = nbContent * 100 / nbTotalContent;
                if (pourcent >= pourcentMax)
                {
                    if (isInfoEnabled)
                    {
                        List<String> i18nParams = new ArrayList<>();
                        i18nParams.add(String.valueOf(pourcentMax));
                        getLogger().info(_i18nTranslator.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_EXPORT_LOG_FILL_NB_CONTENT_FILL", i18nParams)));
                    }
                    pourcentMax += 10;
                }
                
                nbContent++;
            }
        }
        
        if (isInfoEnabled && nbTotalContent != 0)
        {
            getLogger().info(_i18nTranslator.translate(new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_EXPORT_LOG_FILL_NB_CONTENT_FILL_MAX")));
        }
    }
    
    private void _fillContentTable(String idContent, String tableName) throws SQLException
    {
        String dataTableName = _sqlPrefixConf + ExportManager.CONTENT_TABLE_NAME;
        PreparedStatement stmt = getInsertPreparedStatementFromTableName(_sqlPrefixConf, dataTableName, "FULL");
        _stmtList.add(stmt);
        
        stmt.setString(1, idContent);
        stmt.setString(2, tableName);

        stmt.addBatch();
        
        if (getLogger().isDebugEnabled())
        {
            getLogger().debug(stmt.toString());
        }
    }
    
    /**
     * Fill values from a composite metadata
     * @param fillIndex the position indicator in the insert statement
     * @param modelItemContainer the attribute definitions
     * @param dataHolder the {@link ModelAwareDataHolder}
     * @param tableName the table name
     * @param columnNamePrefix the column name prefix
     * @param id the content id
     * @param stmt the jdbc statement
     * @throws SQLException if a sql error occurred
     * @throws AmetysRepositoryException if an ametys repository error occurred
     * @throws IOException if an IO error occurred
     */
    protected void fillValues(ExportCounter fillIndex, ModelItemContainer modelItemContainer, ModelAwareDataHolder dataHolder, String tableName, String columnNamePrefix, String id, PreparedStatement stmt) throws SQLException, AmetysRepositoryException, IOException
    {
        for (ModelItem modelItem : modelItemContainer.getModelItems())
        {
            String name = modelItem.getName();
            String columnName = columnNamePrefix + name;
            if (modelItem instanceof ElementDefinition definition)
            {
                // simple element
                if (definition.isMultiple() && !_exportNoMultiValuedTable)
                {
                    if (dataHolder != null && dataHolder.hasValue(name))
                    {
                        fillTableForMultipleAttribute(definition, dataHolder, name, tableName + "_" + columnName, id);
                    }
                }
                else
                {
                    fillValue(fillIndex, definition, dataHolder, name, id, stmt);
                }
            }
            else if (modelItem instanceof CompositeDefinition)
            {
                // composite
                ModelAwareComposite composite = dataHolder != null ? dataHolder.getValue(name) : null;
                fillValues(fillIndex, (CompositeDefinition) modelItem, composite, tableName, columnName + "_", id, stmt);
            }
            else if (modelItem instanceof RepeaterDefinition)
            {
                // repeater
                if (dataHolder != null)
                {
                    ModelAwareRepeater repeater = dataHolder.getRepeater(name);
                    if (repeater != null)
                    {
                        fillTableForRepeater((RepeaterDefinition) modelItem, dataHolder.getRepeater(name), tableName + "_" + columnName, id);
                    }
                }
            }
        }
    }

    /**
     * Fill values from a repeater
     * @param definition the repeater definition
     * @param repeater the repeater
     * @param tableName the table name
     * @param id the content id
     * @throws SQLException if a sql error occurred
     * @throws AmetysRepositoryException if an ametys repository error occurred
     * @throws IOException if an IO error occurred
     */
    protected void fillTableForRepeater(RepeaterDefinition definition, ModelAwareRepeater repeater, String tableName, String id) throws SQLException, AmetysRepositoryException, IOException
    {
        PreparedStatement stmt = getInsertPreparedStatementFromTableName(_sqlTablePrefix, tableName, _mappingPolicy);
        _stmtList.add(stmt);
        
        for (ModelAwareRepeaterEntry entry : repeater.getEntries())
        {
            int position = entry.getPosition();
            String idRepeater = id + "@" + position;

            stmt.setString(1, idRepeater);
            stmt.setString(2, id);
            stmt.setInt(3, position);
            
            ExportCounter fillIndex = new ExportCounter(4);

            fillValues(fillIndex, definition, entry, tableName, "", idRepeater, stmt);
            
            if (getLogger().isDebugEnabled())
            {
                getLogger().debug(stmt.toString());
            }
            
            stmt.addBatch();
        }
    }
    
    /**
     * Fill values from multiple attribute
     * @param definition the attribute definition
     * @param dataHolder the {@link ModelAwareDataHolder}
     * @param name the attribute name
     * @param tableName the table name
     * @param id the content id
     * @throws SQLException if a sql error occurred
     * @throws IOException if an IO error occurred
     * @throws AmetysRepositoryException if an ametys repository error occurred
     */
    protected void fillTableForMultipleAttribute(ElementDefinition definition, ModelAwareDataHolder dataHolder, String name, String tableName, String id) throws SQLException, AmetysRepositoryException, IOException
    {
        PreparedStatement stmt = getInsertPreparedStatementFromTableName(_sqlTablePrefix, tableName, _mappingPolicy);
        _stmtList.add(stmt);
        
        stmt.setString(1, id);
        
        ExportCounter fillIndex = new ExportCounter(2);
        
        fillValue(fillIndex, definition, dataHolder, name, id, stmt);
    }
    
    /**
     * Fill values from an attribute
     * @param fillIndex the position indicator in the insert statement
     * @param definition the metadata definition model
     * @param dataHolder the {@link ModelAwareDataHolder}
     * @param name the attribute name
     * @param id the content id
     * @param stmt the jdbc statement
     * @throws SQLException if a sql error occurred
     * @throws AmetysRepositoryException if an ametys repository error occurred
     * @throws IOException if an IO error occurred
     */
    protected void fillValue(ExportCounter fillIndex, ElementDefinition definition, ModelAwareDataHolder dataHolder, String name, String id, PreparedStatement stmt) throws SQLException, AmetysRepositoryException, IOException
    {
        String type = definition.getType().getId();
        Object value = dataHolder != null ? dataHolder.getValue(name) : null;
        if (definition.isMultiple())
        {
            if (_exportNoMultiValuedTable)
            {
                String multipleValue = "";
                if (value != null)
                {
                    Object[] values = (Object[]) value;
                    for (int i = 0; i < values.length; i++)
                    {
                        if (i != 0)
                        {
                            multipleValue += _separator;
                        }
                         
                        multipleValue += _getValueForNoMultiValuedTable(values[i], type);
                    }
                }

                if (StringUtils.isNotBlank(multipleValue))
                {
                    stmt.setString(fillIndex.getCount(), multipleValue);
                }
                else
                {
                    stmt.setNull(fillIndex.getCount(), java.sql.Types.VARCHAR);
                }
                
                fillIndex.incrementCount();
            }
            else
            {
                int position = 1;
                if (value != null)
                {
                    Object[] values = (Object[]) value;
                    for (Object o : values)
                    {
                        int nbPos = _setValue(o, 2, stmt, type, name, id);
                        stmt.setInt(2 + nbPos, position);
                        position++;
                        
                        if (getLogger().isDebugEnabled())
                        {
                            getLogger().debug(stmt.toString());
                        }
                        
                        stmt.addBatch();
                    }
                }
            }
        }
        else
        {
            if (value != null)
            {
                int nbPos = _setValue(value, fillIndex.getCount(), stmt, type, name, id);
                fillIndex.incrementCount(nbPos);
            }
            else
            {
                int nbPos = _setNull(fillIndex.getCount(), stmt, type);
                fillIndex.incrementCount(nbPos);
            }
        }
    }
    
    private int _setValue(Object value, int position, PreparedStatement stmt, String type, String name, String contentId) throws SQLException, IOException
    {
        switch (type)
        {
            case ModelItemTypeConstants.CONTENT_ELEMENT_TYPE_ID:
                stmt.setString(position, ((ContentValue) value).getContentId());
                return 1;
            case org.ametys.runtime.model.type.ModelItemTypeConstants.LONG_TYPE_ID:
                stmt.setLong(position, (long) value);
                return 1;
            case org.ametys.runtime.model.type.ModelItemTypeConstants.BOOLEAN_TYPE_ID:
                stmt.setBoolean(position, (boolean) value);
                return 1;
            case org.ametys.runtime.model.type.ModelItemTypeConstants.DATE_TYPE_ID:
                stmt.setDate(position, java.sql.Date.valueOf((LocalDate) value));
                return 1;
            case org.ametys.runtime.model.type.ModelItemTypeConstants.DATETIME_TYPE_ID:
                stmt.setTimestamp(position, Timestamp.from(((ZonedDateTime) value).toInstant()));
                return 1;
            case org.ametys.runtime.model.type.ModelItemTypeConstants.DOUBLE_TYPE_ID:
                stmt.setDouble(position, (double) value);
                return 1;
            case ModelItemTypeConstants.USER_ELEMENT_TYPE_ID:
                UserIdentity user = (UserIdentity) value;
                stmt.setString(position, user.getLogin());
                stmt.setString(position + 1, user.getPopulationId());
                return 2;
            case ModelItemTypeConstants.GEOCODE_ELEMENT_TYPE_ID:
                Geocode geocode = (Geocode) value;
                stmt.setDouble(position, geocode.getLongitude());
                stmt.setDouble(position + 1, geocode.getLatitude());
                return 2;
            case ModelItemTypeConstants.MULTILINGUAL_STRING_ELEMENT_TYPE_ID:
                MultilingualString multilingualString = (MultilingualString) value;
                int i = 0;
                for (String lang : _languageManager.getAvailableLanguages().keySet())
                {
                    Locale locale = LocaleUtils.toLocale(lang);
                    if (multilingualString.hasLocale(locale))
                    {
                        stmt.setString(position + i, multilingualString.getValue(locale));
                    }
                    else
                    {
                        stmt.setNull(position + i, Types.VARCHAR);
                    }
                    
                    i++;
                }
                return i;
            case ModelItemTypeConstants.BINARY_ELEMENT_TYPE_ID:
            case ModelItemTypeConstants.FILE_ELEMENT_TYPE_ID:
                File file = (File) value;
                stmt.setString(position, file.getName());
                stmt.setBlob(position + 1, AutoCloseInputStream.builder().setInputStream(file.getInputStream()).get());
                stmt.setString(position + 2, file.getMimeType());
                stmt.setLong(position + 3, file.getLength());
                stmt.setTimestamp(position + 4, Timestamp.from(file.getLastModificationDate().toInstant()));
                return 5;
            case ModelItemTypeConstants.RICH_TEXT_ELEMENT_TYPE_ID:
                RichText richText = (RichText) value;
                String richTextValue;
                try (InputStream is = richText.getInputStream())
                {
                    richTextValue = IOUtils.toString(is, "UTF-8").replaceAll(_EXCLUDE_XML_TAGS, "");
                    stmt.setString(position, richTextValue);
                }
                catch (IOException e)
                {
                    getLogger().warn("Error with richText of attribute '{}' of content '{}'.", name, contentId);
                    stmt.setNull(position, java.sql.Types.BLOB);
                }
                
                fillColumnForRichTextData(richText, name, contentId);

                return 1;
            case org.ametys.runtime.model.type.ModelItemTypeConstants.STRING_TYPE_ID:
            default:
                stmt.setString(position, value.toString());
                return 1;
        }
    }
    
    private int _setNull(int position, PreparedStatement stmt, String type) throws SQLException
    {
        switch (type)
        {
            case org.ametys.runtime.model.type.ModelItemTypeConstants.LONG_TYPE_ID:
                stmt.setNull(position, Types.INTEGER);
                return 1;
            case org.ametys.runtime.model.type.ModelItemTypeConstants.BOOLEAN_TYPE_ID:
                stmt.setNull(position, Types.NULL);
                return 1;
            case org.ametys.runtime.model.type.ModelItemTypeConstants.DATE_TYPE_ID:
            case org.ametys.runtime.model.type.ModelItemTypeConstants.DATETIME_TYPE_ID:
                stmt.setNull(position, Types.DATE);
                return 1;
            case org.ametys.runtime.model.type.ModelItemTypeConstants.DOUBLE_TYPE_ID:
                stmt.setNull(position, Types.DOUBLE);
                return 1;
            case ModelItemTypeConstants.USER_ELEMENT_TYPE_ID:
                stmt.setNull(position, Types.VARCHAR);
                stmt.setNull(position + 1, Types.VARCHAR);
                return 2;
            case ModelItemTypeConstants.GEOCODE_ELEMENT_TYPE_ID:
                stmt.setNull(position, Types.DOUBLE);
                stmt.setNull(position + 1, Types.DOUBLE);
                return 2;
            case ModelItemTypeConstants.MULTILINGUAL_STRING_ELEMENT_TYPE_ID:
                Set<String> availableLanguages = _languageManager.getAvailableLanguages().keySet();
                for (int i = 0; i < availableLanguages.size(); i++)
                {
                    stmt.setNull(position + i, Types.VARCHAR);
                }
                return availableLanguages.size();
            case ModelItemTypeConstants.BINARY_ELEMENT_TYPE_ID:
            case ModelItemTypeConstants.FILE_ELEMENT_TYPE_ID:
                stmt.setNull(position, java.sql.Types.VARCHAR);
                stmt.setNull(position + 1, java.sql.Types.BLOB);
                stmt.setNull(position + 2, java.sql.Types.VARCHAR);
                stmt.setNull(position + 3, java.sql.Types.INTEGER);
                stmt.setNull(position + 4, java.sql.Types.DATE);
                return 5;
            case ModelItemTypeConstants.RICH_TEXT_ELEMENT_TYPE_ID:
                stmt.setNull(position, java.sql.Types.BLOB);
                return 1;
            case org.ametys.runtime.model.type.ModelItemTypeConstants.STRING_TYPE_ID:
            case ModelItemTypeConstants.CONTENT_ELEMENT_TYPE_ID:
            default:
                stmt.setNull(position, Types.VARCHAR);
                return 1;
        }
    }
    
    private String _getValueForNoMultiValuedTable(Object value, String type)
    {
        if (ModelItemTypeConstants.USER_ELEMENT_TYPE_ID.equals(type))
        {
            User user = _userManager.getUser((UserIdentity) value);
            if (user != null)
            {
                return user.getFullName();
            }
            else
            {
                return UserIdentity.userIdentityToString((UserIdentity) value);
            }
        }
        else
        {
            return value.toString();
        }
    }

    /**
     * Add additional values for content (Title, type, language, creator, creationDate, ...)
     * @param fillIndex the position indicator in the insert statement
     * @param content the content
     * @param stmt the jdbc statement
     * @throws SQLException if a sql error occurred
     * @throws AmetysRepositoryException if an ametys repository error occurred
     */
    protected void fillAdditionalData(ExportCounter fillIndex, Content content, PreparedStatement stmt) throws AmetysRepositoryException, SQLException
    {
        if (content != null)
        {
            stmt.setString(fillIndex.getCount(), content.getTitle());
            fillIndex.incrementCount();
            
            stmt.setString(fillIndex.getCount(), content.getTypes()[0]);
            fillIndex.incrementCount();
            
            //TODO site for webContent
            
            stmt.setString(fillIndex.getCount(), content.getLanguage());
            fillIndex.incrementCount();
            
            stmt.setString(fillIndex.getCount(), content.getCreator().getLogin());
            fillIndex.incrementCount();
            
            if (content.getCreationDate() != null)
            {
                java.sql.Date sqlCreationDate = new java.sql.Date(content.getCreationDate().toInstant().toEpochMilli());
                stmt.setDate(fillIndex.getCount(), sqlCreationDate);
            }
            else
            {
                stmt.setNull(fillIndex.getCount(), java.sql.Types.DATE);
            }
            fillIndex.incrementCount();
            
            stmt.setString(fillIndex.getCount(), content.getLastContributor().getLogin());
            fillIndex.incrementCount();
            
            if (content.getLastModified() != null)
            {
                java.sql.Date sqlLastModificationDate = new java.sql.Date(content.getLastModified().toInstant().toEpochMilli());
                stmt.setDate(fillIndex.getCount(), sqlLastModificationDate);
            }
            else
            {
                stmt.setNull(fillIndex.getCount(), java.sql.Types.DATE);
            }
            fillIndex.incrementCount();
            
            if (content.getLastValidationDate() != null)
            {
                java.sql.Date sqlLastValidationDate = new java.sql.Date(content.getLastValidationDate().toInstant().toEpochMilli());
                stmt.setDate(fillIndex.getCount(), sqlLastValidationDate);
            }
            else
            {
                stmt.setNull(fillIndex.getCount(), java.sql.Types.DATE);
            }
            fillIndex.incrementCount();
            
            if (content.getLastMajorValidationDate() != null)
            {
                java.sql.Date sqlLastMajorValidationDate = new java.sql.Date(content.getLastMajorValidationDate().toInstant().toEpochMilli());
                stmt.setDate(fillIndex.getCount(), sqlLastMajorValidationDate);
            }
            else
            {
                stmt.setNull(fillIndex.getCount(), java.sql.Types.DATE);
            }
            fillIndex.incrementCount();
        }
    }

    /**
     * Fill column for data in rich text
     * @param richText the rich text
     * @param attributeName the metadata name
     * @param contentId the content id
     * @throws AmetysRepositoryException if an error occurred
     * @throws SQLException if a sql error occurred
     * @throws IOException if an IO error occurred
     */
    protected void fillColumnForRichTextData(RichText richText, String attributeName, String contentId) throws AmetysRepositoryException, SQLException, IOException
    {
        String dataTableName = _sqlPrefixConf + ExportManager.RICH_TEXT_DATA_TABLE_NAME;
        PreparedStatement stmt = getInsertPreparedStatementFromTableName(_sqlPrefixConf, dataTableName, "FULL");
        _stmtList.add(stmt);
        
        int position = 1;
        for (NamedResource resource : richText.getAttachments())
        {
            String id = contentId + "@" + attributeName + ";" + resource.getFilename();
            stmt.setString(1, id);
            stmt.setString(2, contentId);
            stmt.setString(3, attributeName);
            stmt.setString(4, resource.getFilename());
            stmt.setBlob(5, AutoCloseInputStream.builder().setInputStream(resource.getInputStream()).get());
            stmt.setString(6, resource.getMimeType());
            stmt.setLong(7, resource.getLength());
            stmt.setTimestamp(8, Timestamp.from(resource.getLastModificationDate().toInstant()));
            stmt.setInt(9, position);
            position++;
            
            if (getLogger().isDebugEnabled())
            {
                getLogger().debug(stmt.toString());
            }
            
            stmt.addBatch();
        }
    }

    /**
     * Prepare INSERT statement
     * @param prefix the table prefix
     * @param tableName the table name
     * @param mappingPolicy the mapping policy
     * @return the INSERT preparedStatement
     * @throws SQLException if a sql error occurred
     */
    protected PreparedStatement getInsertPreparedStatementFromTableName(String prefix, String tableName, String mappingPolicy) throws SQLException
    {
        StringBuilder sql = new StringBuilder();
        sql.append("INSERT INTO ");
        sql.append(_normalizeNameComponent.normalizedTableName(prefix, mappingPolicy, tableName, _connection));
        sql.append(" VALUES ( ?");
        
        ExportTableInfo tableInfo = _tablesInfos.get(tableName);

        for (int i = 1; i < tableInfo.getNbColumns(); i++)
        {
            sql.append(", ?");
        }
        sql.append(")");
        
        if (getLogger().isDebugEnabled())
        {
            getLogger().debug("Executing: {}", sql.toString());
        }
        
        PreparedStatement stmt = _connection.prepareStatement(sql.toString());
        return stmt;
    }

    /**
     * Get the name of SQL table for given content type
     * @param cTypeId The id of content type
     * @return the name of SQL table or null if not found
     */
    protected String getTableName (String cTypeId)
    {
        return _contentTypesToExport.get(cTypeId);
    }
}
