/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.archive;

import java.io.ByteArrayOutputStream;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import org.apache.commons.collections4.CollectionUtils;

/**
 * A report of an import ,whether it is a global, partial or unitary one.
 */
public final class ImportReport
{
    private List<ImportError> _errors;
    
    /**
     * Creates a new empty report.
     */
    public ImportReport()
    {
        _errors = new ArrayList<>();
    }
    
    private ImportReport(Collection<ImportError> errors)
    {
        _errors = new ArrayList<>(errors);
    }
    
    /**
     * Returns a new {@link ImportReport} which is the union of the given ones
     * @param reports The reports on which to build the union
     * @return a new {@link ImportReport} which is the union of the given ones
     */
    public static ImportReport union(Collection<ImportReport> reports)
    {
        return union(reports.toArray(ImportReport[]::new));
    }
    
    /**
     * Returns a new {@link ImportReport} which is the union of the given ones
     * @param reports The reports on which to build the union
     * @return a new {@link ImportReport} which is the union of the given ones
     */
    public static ImportReport union(ImportReport... reports)
    {
        if (reports.length == 0)
        {
            return new ImportReport();
        }
        else if (reports.length == 1)
        {
            return new ImportReport(reports[0]._errors);
        }
        
        Collection<ImportError> mergedErrors = Collections.EMPTY_LIST;
        for (ImportReport current : reports)
        {
            mergedErrors = CollectionUtils.union(mergedErrors, current._errors);
        }
        
        ImportReport merged = new ImportReport(mergedErrors);
        return merged;
    }
    
    /**
     * Gets an unmodifiable view of the errors of this report
     * @return an unmodifiable view of the errors of this report
     */
    public Collection<ImportError> getErrors()
    {
        return Collections.unmodifiableCollection(_errors);
    }
    
    /**
     * Adds an error to this report
     * @param error The error to add
     */
    public void addError(ImportError error)
    {
        _errors.add(error);
    }
    
    /**
     * Adds all information (errors...) from the other report to this report
     * @param other The other report
     */
    public void addFrom(ImportReport other)
    {
        _errors.addAll(other._errors);
    }
    
    /**
     * An error during an import, which can be {@link ImportReport#addError(ImportError) added} to an {@link ImportReport}
     */
    public static final class ImportError
    {
        private final Throwable _throwable;

        /**
         * Creates a new error.
         * @param t The {@link Throwable}
         */
        public ImportError(Throwable t)
        {
            _throwable = t;
        }
        
        /**
         * Gets the {@link Throwable#getMessage() message} of the {@link Throwable}
         * @return the message
         */
        public String getMessage()
        {
            return _throwable.getMessage();
        }
        
        /**
         * Gets the {@link Throwable#getStackTrace() stack trace} of the {@link Throwable}, as formatted string
         * @return the stacktrace as formatted string
         */
        public String getStackTrace()
        {
            OutputStream os = new ByteArrayOutputStream();
            var printWriter = new PrintWriter(os); 
            _throwable.printStackTrace(printWriter);
            printWriter.flush();
            return os.toString();
        }
    }
}
