/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.csv;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;

import org.apache.avalon.framework.activity.Initializable;

import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.Enumerator;

/**
 * {@link Enumerator} for the memory caches.
 */
public class SynchronizeModeEnumerator implements Enumerator<String>, Initializable
{
    /** Avalon role */
    public static final String ROLE = SynchronizeModeEnumerator.class.getName();

    /**
     * The import mode
     */
    public enum ImportMode
    {
        /** This mode will both create and update contents*/
        CREATE_AND_UPDATE,
        /** This mode will only create contents. If the content already exists, it will fail, 
         *  if the subcontent exists, the subcontconent will not be updated, and the content will be only partially imported */
        CREATE_ONLY,
        /** This mode will only update existing contents. If the content does not exists, it will fail, 
         *  if the subcontent does not exists, the subcontconent will not be created, and the content will be only partially imported */
        UPDATE_ONLY
    }
    
    private Map<String, I18nizableText> _synchronizeModes;

    @Override
    public I18nizableText getEntry(String value)
    {
        return _synchronizeModes.get(value);
    }
    
    @Override
    public Map<String, I18nizableText> getEntries()
    {
        return Collections.unmodifiableMap(_synchronizeModes);
    }

    public void initialize() throws Exception
    {
        String catalogName = "plugin.contentio";
        _synchronizeModes = new LinkedHashMap<>(3);
        _synchronizeModes.put(ImportMode.CREATE_AND_UPDATE.toString(), new I18nizableText(catalogName, "PLUGINS_CONTENTIO_CONTENT_IMPORT_DIALOG_SYNCHRONIZE_MODE_CREATE_AND_UPDATE"));
        _synchronizeModes.put(ImportMode.CREATE_ONLY.toString(), new I18nizableText(catalogName, "PLUGINS_CONTENTIO_CONTENT_IMPORT_DIALOG_SYNCHRONIZE_MODE_CREATE_ONLY"));
        _synchronizeModes.put(ImportMode.UPDATE_ONLY.toString(), new I18nizableText(catalogName, "PLUGINS_CONTENTIO_CONTENT_IMPORT_DIALOG_SYNCHRONIZE_MODE_UPDATE_ONLY"));
    }
    
}
