/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.export.sql;

import java.sql.Connection;
import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.text.WordUtils;

import org.ametys.core.datasource.ConnectionHelper;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Normalize name component
 */
public class NormalizeNameComponent extends AbstractLogEnabled implements Component, Contextualizable
{
    /** The component role */
    public static final String ROLE = NormalizeNameComponent.class.getName();
    
    /** The avalon context. */
    protected Context _context;
        
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    /**
     * Get the normalized table name
     * @param prefix the table name prefix
     * @param mappingPolicy the mapping policy
     * @param initialTableName The initial table name
     * @param connection the connection
     * @return normalized table name
     */
    protected String normalizedTableName (String prefix, String mappingPolicy, String initialTableName, Connection connection)
    {
        Map<String, String> mappingTableName = getMappingTableNameFromCache();
        if (mappingTableName.containsKey(initialTableName))
        {
            return mappingTableName.get(initialTableName);
        }
        else
        {
            String initialTableNameNoPrefix = initialTableName.substring(prefix.length());
            String normalizedTableName = initialTableNameNoPrefix.replace("-", "_");
            StringBuilder changedInitialName = new StringBuilder(); 
            int i = 0;
            
            for (String name : normalizedTableName.split("_"))
            {
                if (i != 0)
                {
                    changedInitialName.append("_");
                }
                changedInitialName.append(normalizeNameFromPolicy(mappingPolicy, StringUtils.capitalize(name)));
                i++;
            }
            
            normalizedTableName = prefix + changedInitialName.toString();

            int maxLength = Integer.MAX_VALUE;
            
            String datatype = ConnectionHelper.getDatabaseType(connection);
            if (datatype.equals(ConnectionHelper.DATABASE_MYSQL))
            {
                maxLength = 64 /* mysql max */;
            }
            else if (datatype.equals(ConnectionHelper.DATABASE_ORACLE))
            {
                maxLength = 30 /* oracle max */;
            }
            
            if (normalizedTableName.length() >= maxLength)
            {
                final int hashSize = 6;
                normalizedTableName = normalizedTableName.substring(0, maxLength - hashSize) + String.valueOf((int) Math.abs(normalizedTableName.hashCode() % Math.pow(10, hashSize)));
            }
            
            if (mappingTableName.containsValue(normalizedTableName))
            {
                normalizedTableName = StringUtils.substring(normalizedTableName, 0, normalizedTableName.length() - 2) + "_" + _getCountSameValue(normalizedTableName, mappingTableName, 1);
            }
            
            mappingTableName.put(initialTableName, normalizedTableName);
            
            return normalizedTableName;
        }
    }
    
    /**
     * Get the normalized column name
     * @param mappingPolicy the mappingPolicy
     * @param initialColumnName The initial column name
     * @param initialTableName The initial table name
     * @param reservedWords the map of reserved words
     * @param connection the connection
     * @return normalized column name
     */
    protected String normalizedColumnName (String mappingPolicy, String initialColumnName, String initialTableName, Map<String, Map<String, String>> reservedWords, Connection connection)
    {
        String datatype = ConnectionHelper.getDatabaseType(connection);
        String columnName = initialColumnName;
        if (datatype.equals(ConnectionHelper.DATABASE_MYSQL))
        {
            Map<String, String> mysqlReservedWords = reservedWords.get("mysql");
            if (mysqlReservedWords.containsKey(StringUtils.upperCase(initialColumnName)))
            {
                columnName = StringUtils.lowerCase(mysqlReservedWords.get(StringUtils.upperCase(initialColumnName)));
            }
        }
        else if (datatype.equals(ConnectionHelper.DATABASE_ORACLE))
        {
            Map<String, String> oracleReservedWords = reservedWords.get("oracle");
            if (oracleReservedWords.containsKey(StringUtils.upperCase(initialColumnName)))
            {
                columnName = StringUtils.lowerCase(oracleReservedWords.get(StringUtils.upperCase(initialColumnName)));
            }
        }
        
        Map<String, HashMap<String, String>> mappingTableColumnName = getMappingTableColumnNameFromCache();
        HashMap<String, String> mappingColumnName = new HashMap<>();
        if (!mappingTableColumnName.containsKey(initialTableName))
        {
            mappingTableColumnName.put(initialTableName, mappingColumnName);
        }
        
        mappingColumnName = mappingTableColumnName.get(initialTableName);
        if (mappingColumnName.containsKey(columnName))
        {
            return mappingColumnName.get(columnName);
        }
        else
        {
            String normalizedColumnName = columnName.replace("-", "_");
            StringBuilder changedInitialName = new StringBuilder(); 
            int i = 0;
            for (String name : normalizedColumnName.split("_"))
            {
                if (i != 0)
                {
                    changedInitialName.append("_");
                }
                
                changedInitialName.append(normalizeNameFromPolicy(mappingPolicy, StringUtils.capitalize(name)));
                i++;
            }
            
            normalizedColumnName = changedInitialName.toString();
            
            int maxLength = Integer.MAX_VALUE;
            
            if (datatype.equals(ConnectionHelper.DATABASE_MYSQL))
            {
                maxLength = 64 /* mysql max */;
            }
            else if (datatype.equals(ConnectionHelper.DATABASE_ORACLE))
            {
                maxLength = 30 /* oracle max */;
            }
            
            if (normalizedColumnName.length() > maxLength)
            {
                final int hashSize = 9;
                normalizedColumnName = normalizedColumnName.substring(0, maxLength - hashSize) + String.valueOf((int) Math.abs(normalizedColumnName.hashCode() % Math.pow(10, hashSize)));
            }
            
            if (mappingColumnName.containsValue(normalizedColumnName))
            {
                normalizedColumnName = StringUtils.substring(normalizedColumnName, 0, normalizedColumnName.length() - 2) + "_" + _getCountSameValue(normalizedColumnName, mappingColumnName, 1);
            }
            
            mappingColumnName.put(columnName, normalizedColumnName);
            
            return normalizedColumnName;
        }
    }
    
    private int _getCountSameValue(String value, Map<String, String> mappingTable, int count)
    {
        String valueAux = value;
        int countAux = count;
        for (String val : mappingTable.values())
        {
            if (val.equals(valueAux))
            {
                countAux++;
                valueAux = StringUtils.substring(valueAux, 0, valueAux.length() - 2) + "_" + countAux;
                return _getCountSameValue(valueAux, mappingTable, countAux);
            }
        }
        
        return countAux;
    }
    
    /**
     * Get the normalized comment
     * @param initialComment the initial comment
     * @param maxLength The maximun number of authorized characters
     * @param connection the connection
     * @return normalized comment
     */
    protected String normalizedComment (String initialComment, int maxLength, Connection connection)
    {
        String normalizedComment = escapeValue(initialComment, connection);
        
        if (maxLength != 0 && normalizedComment.length() > maxLength)
        {
            normalizedComment = normalizedComment.substring(0, maxLength - 3) + "...";
        }
        
        return normalizedComment;
    }

    /**
     * Escape column value or comment
     * @param value The value or SQL comment
     * @param connection the connection
     * @return the escaped value
     */
    protected String escapeValue (String value, Connection connection)
    {
        String newValue = value;
        String datatype = ConnectionHelper.getDatabaseType(connection);
        if (datatype.equals(ConnectionHelper.DATABASE_MYSQL))
        {
            newValue = value.replace("\'", "\\'");
        }
        else if (datatype.equals(ConnectionHelper.DATABASE_ORACLE))
        {
            newValue = value.replace("'", "''");
        }
        
        return newValue;
    }
    
    /**
     * Get the table name from the cache
     * @return the mapping for table name
     */
    @SuppressWarnings("unchecked")
    protected Map<String, String> getMappingTableNameFromCache()
    {
        Request request = ContextHelper.getRequest(_context);
        
        Map<String, String> mappingTableName = new HashMap<>();
        if (request.getAttribute("mappingTableName") != null)
        {
            mappingTableName = (Map<String, String>) request.getAttribute("mappingTableName");
        }
        else
        {
            request.setAttribute("mappingTableName", mappingTableName);
        }
        
        return mappingTableName;
    }
    
    /**
     * Get the column name from the cache
     * @return the mapping for column name
     */
    @SuppressWarnings("unchecked")
    protected Map<String, HashMap<String, String>> getMappingTableColumnNameFromCache()
    {
        Request request = ContextHelper.getRequest(_context);
        
        Map<String, HashMap<String, String>> mappingTableColumnName = new HashMap<>();
        if (request.getAttribute("mappingTableColumnName") != null)
        {
            mappingTableColumnName = (Map<String, HashMap<String, String>>) request.getAttribute("mappingTableColumnName");
        }
        else
        {
            request.setAttribute("mappingTableColumnName", mappingTableColumnName);
        }
        
        return mappingTableColumnName;
    }
    
    /**
     * Normalize the initialName into a name depends on the mapping Policy
     * @param mappingPolicy (FULL, CAMELCASE, FIRSTCHAR)
     * @param initialName the initial table name
     * @return the name changed depends on the mapping Policy
     */
    protected String normalizeNameFromPolicy(String mappingPolicy, String initialName)
    {
        String modifiedName = "";
        if (mappingPolicy.equals("FULL"))
        {
            modifiedName = initialName;
        }
        else if (mappingPolicy.equals("CAMELCASE"))
        {
            modifiedName = initialName.replaceAll("(\\p{Ll})(\\p{Lu})", "$1 $2");
            modifiedName = WordUtils.initials(modifiedName);
        }
        else if (mappingPolicy.equals("FIRSTCHAR"))
        {
            modifiedName = initialName.replaceAll("(\\p{Ll})(\\p{Lu})", "$1 $2");
            String[] words = modifiedName.split(" ");
            
            StringBuilder finalName = new StringBuilder();
            for (String word : words)
            {
                if (word.length() > 3)
                {
                    for (int i = 3; i < word.length(); i++)
                    {
                        char charWord = word.charAt(i);
                        if (Character.isUpperCase(charWord)
                            || charWord == 'a' || charWord == 'e' || charWord == 'i' 
                            || charWord == 'o' || charWord == 'u' || charWord == 'y')
                        {
                            finalName.append(word.substring(0, i));
                            break;
                        }
                        else if (i == word.length() - 1)
                        {
                            finalName.append(word);
                        }
                    }
                }
                else
                {
                    finalName.append(word);
                }
            }
            modifiedName = finalName.toString();
        }
        
        return modifiedName;
    }
}
