/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.in.xml;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import javax.xml.transform.OutputKeys;
import javax.xml.transform.TransformerException;
import javax.xml.transform.dom.DOMResult;
import javax.xml.transform.sax.TransformerHandler;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.xml.dom.DOMParser;
import org.apache.excalibur.xml.sax.SAXParser;
import org.apache.excalibur.xml.xpath.PrefixResolver;
import org.apache.excalibur.xml.xpath.XPathProcessor;
import org.apache.excalibur.xml.xslt.XSLTProcessor;
import org.apache.excalibur.xml.xslt.XSLTProcessorException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import org.ametys.plugins.contentio.ContentImporterHelper;
import org.ametys.plugins.contentio.in.AbstractContentImporter;

/**
 * Abstract {@link XmlContentImporter} class which provides base XML importer configuration and logic.
 */
public abstract class AbstractXmlContentImporter extends AbstractContentImporter implements XmlContentImporter
{
    /** The service manager. */
    protected ServiceManager _manager;
    
    /** The source resolver. */
    protected SourceResolver _srcResolver;
    
    /** A DOM parser. */
    protected DOMParser _domParser;
    
    /** The XPath processor. */
    protected XPathProcessor _xPathProcessor;
    
    /** The runtime XSLT processor. */
    protected XSLTProcessor _xsltProcessor;
    
    /** The prefix resolver. */
    protected PrefixResolver _prefixResolver;
    
    /** The XSL transformer handler. */
    protected TransformerHandler _xslTransformerHandler;
    
    /** The configured XML transformation stylesheet. */
    protected String _xsl;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _manager = serviceManager;
        _srcResolver = (SourceResolver) serviceManager.lookup(SourceResolver.ROLE);
        _domParser = (DOMParser) serviceManager.lookup(DOMParser.ROLE);
        _xPathProcessor = (XPathProcessor) serviceManager.lookup(XPathProcessor.ROLE);
        _xsltProcessor = (XSLTProcessor) serviceManager.lookup(XSLTProcessor.ROLE + "/xalan");
    }
    
    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        super.configure(configuration);
        configureXml(configuration.getChild("xml"));
    }
    
    /**
     * Configure XML-specific properties.
     * @param configuration the XML configuration.
     * @throws ConfigurationException if an error occurs.
     */
    protected void configureXml(Configuration configuration) throws ConfigurationException
    {
        _xsl = configuration.getChild("xsl").getAttribute("src", null);
        configureNamespaces(configuration.getChild("namespaces"));
    }
    
    @Override
    protected void configureContentCreation(Configuration configuration) throws ConfigurationException
    {
        // Override default configuration to be more permissive.
        String typesStr = configuration.getChild("content-types").getValue("");
        _contentTypes = StringUtils.split(typesStr, ", ");
        
        String mixins = configuration.getChild("mixins").getValue("");
        _mixins = StringUtils.split(mixins, ", ");
        
        _language = configuration.getChild("language").getValue("");
        
        configureWorkflow(configuration);
    }
    
    /**
     * Configure the namespace to use.
     * @param configuration the namespaces configuration, can be null.
     * @throws ConfigurationException if an error occurs.
     */
    protected void configureNamespaces(Configuration configuration) throws ConfigurationException
    {
        Map<String, String> namespaces = new HashMap<>();
        
        for (Configuration nsConf : configuration.getChildren("namespace"))
        {
            String prefix = nsConf.getAttribute("prefix", "");
            String namespace = nsConf.getAttribute("uri");
            
            namespaces.put(prefix, namespace);
        }
        
        _prefixResolver = new DefaultPrefixResolver(namespaces);
    }
    
    /**
     * Get the prefix resolver.
     * @return the prefix resolver.
     */
    protected PrefixResolver getPrefixResolver()
    {
        return _prefixResolver;
    }
    
    @Override
    public boolean supports(InputStream is, String name) throws IOException
    {
        try
        {
            Document doc = _domParser.parseDocument(new InputSource(is));
            
            return supports(doc);
        }
        catch (SAXException e)
        {
            throw new IOException("Error parsing the document.", e);
        }
    }
    
    @Override
    public Set<String> importContents(InputStream is, Map<String, Object> params) throws IOException
    {
        Set<String> contentIds = new HashSet<>();
        SAXParser saxParser = null;
        
        try
        {
            Document document = null;
            
            // Either parse the document (no XSL) or transform 
            if (_xsl == null)
            {
                document = _domParser.parseDocument(new InputSource(is));
            }
            else
            {
                // Initialize the XSL transformer.
                initializeXslTransformerHandler();
                
                // Transform the XML doc with the configured XSL.
                DOMResult result = new DOMResult();
                _xslTransformerHandler.setResult(result);
                saxParser = (SAXParser) _manager.lookup(SAXParser.ROLE);
                saxParser.parse(new InputSource(is), _xslTransformerHandler);
                Node node = result.getNode();
                
                if (node instanceof Document)
                {
                    document = (Document) node;
                }
            }
            
            if (document != null)
            {
                if (getLogger().isDebugEnabled())
                {
                    getLogger().debug("Importing contents from document:\n {}", ContentImporterHelper.serializeNode(document, true));
                }
                
                contentIds = importContents(document, params);
            }
        }
        catch (ServiceException e)
        {
            getLogger().error("Unable to get a SAX parser.", e);
            throw new IOException("Unable to get a SAX parser.", e);
        }
        catch (SAXException | TransformerException | XSLTProcessorException e)
        {
            getLogger().error("Error parsing the XML document.", e);
            throw new IOException("Error parsing the XML document.", e);
        }
        finally
        {
            _manager.release(saxParser);
        }
        
        return contentIds;
    }
    
    /**
     * Import the contents from the XML DOM {@link Document}.
     * @param document the XML Document.
     * @param params the import parameters.
     * @return a Set of the imported content IDs.
     * @throws IOException if an error occurs importing the contents.
     */
    protected abstract Set<String> importContents(Document document, Map<String, Object> params) throws IOException;
    
    /**
     * Initialize the transformer from the configured XSL.
     * @throws IOException if an errors occurs reading the XSL.
     * @throws XSLTProcessorException of an error occurs during the XSL transformer manipulation
     */
    protected void initializeXslTransformerHandler() throws IOException, XSLTProcessorException
    {
        if (_xslTransformerHandler == null && StringUtils.isNotEmpty(_xsl))
        {
            Source xslSource = null;
            
            try
            {
                xslSource = _srcResolver.resolveURI(_xsl);
                
                _xslTransformerHandler = _xsltProcessor.getTransformerHandler(xslSource);
                
                Properties format = new Properties();
                format.put(OutputKeys.METHOD, "xml");
                format.put(OutputKeys.INDENT, "no");
                format.put(OutputKeys.ENCODING, "UTF-8");
                
                _xslTransformerHandler.getTransformer().setOutputProperties(format);
            }
            finally
            {
                _srcResolver.release(xslSource);
            }
        }
    }
    
    /**
     * Get a node's text content, without trimming it.
     * @param node the node, can be null.
     * @param defaultValue the default value.
     * @return the node's text content, or the default value if the given node is null.
     */
    protected String getTextContent(Node node, String defaultValue)
    {
        return getTextContent(node, defaultValue, false);
    }
    
    /**
     * Get a node's text content, optionally trimmed.
     * @param node the node, can be null.
     * @param defaultValue the default value.
     * @param trim true to trim the text content, false otherwise.
     * @return the node's text content, or the default value if the given node is null.
     */
    protected String getTextContent(Node node, String defaultValue, boolean trim)
    {
        String value = defaultValue;
        if (node != null)
        {
            value = trim ? node.getTextContent().trim() : node.getTextContent();
        }
        
        return value;
    }
    
    /**
     * Get an element attribute value (trimmed).
     * @param element the {@link Element}, can be null.
     * @param name the attribute name.
     * @param defaultValue the default value.
     * @return the node's attribute value, or the default value if the given node is null
     * or the attribute doesn't exist.
     */
    protected String getAttributeValue(Element element, String name, String defaultValue)
    {
        return getAttributeValue(element, name, defaultValue, true);
    }
    
    /**
     * Get a node's attribute value, optionally trimmed.
     * @param element the {@link Element}, can be null.
     * @param name the attribute name.
     * @param defaultValue the default value.
     * @param trim true
     * @return the node's attribute value, or the default value if the given node is null or the attribute doesn't exist.
     */
    protected String getAttributeValue(Element element, String name, String defaultValue, boolean trim)
    {
        String value = defaultValue;
        if (element != null)
        {
            value = element.getAttribute(name);
        }
        
        return value != null ? trim ? value.trim() : value : null;
    }
    
    /**
     * Configurable XML prefix resolver.
     */
    protected static class DefaultPrefixResolver implements PrefixResolver
    {
        /** Map of namespace URIs, indexed by prefix. */
        private Map<String, String> _namespaces;
        
        /**
         * Constructor.
         * @param namespaces the namespaces to resolve, indexed by prefix.
         */
        public DefaultPrefixResolver(Map<String, String> namespaces)
        {
            _namespaces = new HashMap<>(namespaces);
        }
        
        @Override
        public String prefixToNamespace(String prefix)
        {
            return _namespaces.get(prefix);
        }
    }
}
