/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.synchronize;

import java.time.ZonedDateTime;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.LocaleUtils;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.data.holder.ModelLessDataHolder;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.ExpressionContext;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.model.type.ModelItemTypeConstants;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;

/**
 * Helper for Synchronizable Contents Collections.
 */
public class SynchronizableContentsCollectionHelper extends AbstractLogEnabled implements Serviceable, Component
{
    /** The Avalon Role */
    public static final String ROLE = SynchronizableContentsCollectionHelper.class.getName();
    
    /** SCC DAO */
    protected SynchronizableContentsCollectionDAO _sccDAO;
    /** The content type extension point */
    protected ContentTypeExtensionPoint _contentTypeEP;
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    /** The Ametys resolver */
    protected AmetysObjectResolver _resolver;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _sccDAO = (SynchronizableContentsCollectionDAO) smanager.lookup(SynchronizableContentsCollectionDAO.ROLE);
        _contentTypeEP = (ContentTypeExtensionPoint) smanager.lookup(ContentTypeExtensionPoint.ROLE);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
    }
    
    /**
     * Get the first {@link SynchronizableContentsCollection} found for the given SCC model id.
     * @param modelId Id of the SCC model
     * @return The first SCC found or null
     */
    public SynchronizableContentsCollection getSCCFromModelId(String modelId)
    {
        SynchronizableContentsCollection collection = null;

        // Get the first collection corresponding to the SCC model
        for (SynchronizableContentsCollection scc : _sccDAO.getSynchronizableContentsCollections())
        {
            if (scc.getSynchronizeCollectionModelId().equals(modelId))
            {
                collection = scc;
                break;
            }
        }
        
        return collection;
    }

    /**
     * Transform results to be organized by content attribute, and remove the null values.
     * @param searchResult Remote values from a search by content and column or attribute
     * @param mapping Mapping between content attribute and columns/attributes
     * @return A {@link Map} of possible attribute values organized by content synchronization key and attribute name
     */
    public Map<String, Map<String, List<Object>>> organizeRemoteValuesByAttribute(Map<String, Map<String, Object>> searchResult, Map<String, List<String>> mapping)
    {
        Map<String, Map<String, List<Object>>> result = new LinkedHashMap<>();
        
        // For each searchResult line (1 line = 1 content)
        for (String resultKey : searchResult.keySet())
        {
            Map<String, Object> searchItem = searchResult.get(resultKey);
            Map<String, List<Object>> contentResult = new HashMap<>();
            
            // For each attribute in the mapping
            for (String attributeName : mapping.keySet())
            {
                List<String> columns = mapping.get(attributeName); // Get the columns for the current attribute
                List<Object> values = columns.stream() // For each column corresponding to the attribute
                        .map(searchItem::get) // Map the values
                        .flatMap(o ->
                        {
                            if (o instanceof Collection<?>)
                            {
                                return ((Collection<?>) o).stream();
                            }
                            return Stream.of(o);
                        }) // If it's a list of objects, get a flat stream
                        .filter(Objects::nonNull) // Remove null values
                        .collect(Collectors.toList()); // Collect it into a List
                
                contentResult.put(attributeName, values); // Add the retrieved attribute values list to the contentResult
            }
            
            result.put(resultKey, contentResult);
        }
        
        return result;
    }
    
    /**
     * Add the given synchronizable collection id to the existing ones
     * @param content The synchronized content
     * @param collectionId The ID of the collection to add
     */
    public void updateSCCProperty(Content content, String collectionId)
    {
        Set<String> collectionIds = getSynchronizableCollectionIds(content);
        if (collectionIds.add(collectionId))
        {
            content.getInternalDataHolder().setValue(SynchronizableContentsCollection.COLLECTION_ID_DATA_NAME, collectionIds.toArray(new String[collectionIds.size()]));
        }
    }
    
    /**
     * Remove the synchronizable collection id from the SCC property
     * @param content The synchronized content
     * @param collectionId The ID of the collection to remove
     */
    public void removeSCCProperty(Content content, String collectionId)
    {
        Set<String> collectionIds = getSynchronizableCollectionIds(content);
        if (collectionIds.remove(collectionId))
        {
            content.getInternalDataHolder().setValue(SynchronizableContentsCollection.COLLECTION_ID_DATA_NAME, collectionIds.toArray(new String[collectionIds.size()]));
        }
    }
    
    /**
     * Update the given content's synchronization properties
     * @param content the synchronized content
     */
    public void updateLastSynchronizationProperties(Content content)
    {
        content.getInternalDataHolder().setValue(SynchronizableContentsCollection.LAST_SYNCHRONIZATION_DATA_NAME, ZonedDateTime.now(), ModelItemTypeConstants.DATETIME_TYPE_ID);
        content.getInternalDataHolder().setValue(SynchronizableContentsCollection.LAST_SYNCHRONIZATION_USER_DATA_NAME, _currentUserProvider.getUser(), org.ametys.cms.data.type.ModelItemTypeConstants.USER_ELEMENT_TYPE_ID);
    }
    
    /**
     * Retrieves the synchronizable collection identifiers
     * @param content the content
     * @return the synchronizable collection identifiers
     * @throws AmetysRepositoryException if an error occurs while reading SCC info on the given content
     */
    public Set<String> getSynchronizableCollectionIds(Content content) throws AmetysRepositoryException
    {
        ModelLessDataHolder internalDataHolder = content.getInternalDataHolder();
        
        Set<String> collectionIds = new HashSet<>();
        if (internalDataHolder.hasValue(SynchronizableContentsCollection.COLLECTION_ID_DATA_NAME))
        {
            String[] existingCollectionIds = internalDataHolder.getValue(SynchronizableContentsCollection.COLLECTION_ID_DATA_NAME);
            collectionIds = Arrays.stream(existingCollectionIds)
                                  .collect(Collectors.toSet());
        }
        
        return collectionIds;
    }
    
    /**
     * Retrieves a query expression testing the collection
     * @param collectionId the identifier of the collection to test
     * @return the query expression
     */
    public Expression getCollectionExpression(String collectionId)
    {
        ExpressionContext context = ExpressionContext.newInstance()
                                                     .withInternal(true);
        
        return new StringExpression(SynchronizableContentsCollection.COLLECTION_ID_DATA_NAME, Operator.EQ, collectionId, context);
    }
    
    /**
     * Import the content specified by the id in the specified collection.
     * @param collectionId Collection ID
     * @param id Synchronization ID of the content
     * @param additionalParameters Additional parameters
     * @return Imported contents
     */
    public Map<String, Object> importContent(String collectionId, String id, Map<String, Object> additionalParameters)
    {
        if (StringUtils.isBlank(id))
        {
            getLogger().warn("The synchronization code cannot be empty.");
            return Map.of("error", "noSyncCode");
        }
        
        Map<String, Object> result = new HashMap<>();
        
        try
        {
            Locale defaultLocale = additionalParameters.containsKey("language") ? LocaleUtils.toLocale((String) additionalParameters.get("language")) : null;
            Set<Map<String, String>> contentsList = new HashSet<>();
            
            SynchronizableContentsCollection collection = _sccDAO.getSynchronizableContentsCollection(collectionId);
            Content existingContent = collection.getContent(null, id, true);
            if (existingContent == null)
            {
                List<ModifiableContent> contents = collection.importContent(id, additionalParameters, getLogger());
                for (ModifiableContent content : contents)
                {
                    Map<String, String> contentMap = new HashMap<>();
                    contentMap.put("id", content.getId());
                    contentMap.put("title", content.getTitle(defaultLocale));
                    contentMap.put("lang", content.getLanguage());
                    contentsList.add(contentMap);
                }
                result.put("contents", contentsList);
                result.put("total", contents.size());
            }
            else
            {
                result.put("contents", ImmutableList.of(ImmutableMap.of("id", existingContent.getId(), "title", existingContent.getTitle(defaultLocale), "lang", existingContent.getLanguage())));
                result.put("error", "alreadyImported");
            }
        }
        catch (Exception e)
        {
            String errorMessage = "An exception occured during import of the content '" + id + "' on SCC '" + collectionId + "'";
            getLogger().error(errorMessage, e);
            throw new IllegalStateException(errorMessage);
        }
        
        return result;
    }
    
    /**
     * Synchronize the content on the given collection with the given synchronization code.
     * @param collectionId Collection ID
     * @param contentId Content ID
     * @param syncCode Synchronization code
     * @return true if an error occurred
     */
    public boolean synchronizeContent(String collectionId, String contentId, String syncCode)
    {
        ModifiableContent content = _resolver.resolveById(contentId);
        boolean hasErrors = false;
        
        try
        {
            SynchronizableContentsCollection collection = _sccDAO.getSynchronizableContentsCollection(collectionId);
            
            // First, add, update or remove synchronization informations
            collection.updateSyncInformations(content, syncCode, getLogger());
            
            // If the synchronization code is empty, the process ends here
            if (StringUtils.isBlank(syncCode))
            {
                return false;
            }
            
            Map<String, Object> searchParameters = new HashMap<>();
            searchParameters.put(collection.getIdField(), syncCode);
            
            if (collection.getTotalCount(searchParameters, getLogger()) > 0)
            {
                collection.synchronizeContent(content, getLogger());
            }
            else
            {
                getLogger().warn("In the collection '{}', there is not content matching with the synchronization code '{}'.", collectionId, syncCode);
                hasErrors = true;
            }
        }
        catch (Exception e)
        {
            getLogger().error("An error occured while synchronizing the content '{}' with the synchronization code '{}' from the '{}' collection.", contentId, syncCode, collectionId, e);
            hasErrors = true;
        }
        
        return hasErrors;
    }
    
    /**
     * Get the value of the synchronization field.
     * @param collectionId Collection ID
     * @param contentId Content ID
     * @return The value of the synchronization field
     */
    public String getSyncCode(String contentId, String collectionId)
    {
        SynchronizableContentsCollection collection = _sccDAO.getSynchronizableContentsCollection(collectionId);
        Content content = _resolver.resolveById(contentId);
        
        String syncCode = null;
        if (content.hasValue(collection.getIdField()))
        {
            syncCode = content.getValue(collection.getIdField());
        }
        return syncCode;
    }
}
