/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.synchronize.impl;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.datasource.DataSourceConsumer;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollection;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollectionDAO;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollectionModel;
import org.ametys.plugins.contentio.synchronize.SynchronizeContentsCollectionModelExtensionPoint;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.type.ModelItemTypeConstants;

/**
 * Implementation of {@link DataSourceConsumer} allowing to know whether a data source is used by the collections of synchronizable contents or not.
 * It also allows to retrieve the data source ids that are currently in use.
 */
public class CollectionDataSourceConsumer implements DataSourceConsumer, Component, Serviceable
{
    /** Avalon Role */
    public static final String ROLE = CollectionDataSourceConsumer.class.getName();
    
    /** The DAO for synchronizable contents collections */
    protected SynchronizableContentsCollectionDAO _synchronizableContentsCollectionDAO;

    /** The extension point for synchronizable contents collection models */
    protected SynchronizeContentsCollectionModelExtensionPoint _synchronizeContentsCollectionModelEP;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _synchronizableContentsCollectionDAO = (SynchronizableContentsCollectionDAO) manager.lookup(SynchronizableContentsCollectionDAO.ROLE);
        _synchronizeContentsCollectionModelEP = (SynchronizeContentsCollectionModelExtensionPoint) manager.lookup(SynchronizeContentsCollectionModelExtensionPoint.ROLE);
    }
    
    @Override
    public TypeOfUse isInUse(String id)
    {
        for (String modelId : _synchronizeContentsCollectionModelEP.getExtensionsIds())
        {
            SynchronizableContentsCollectionModel model = _synchronizeContentsCollectionModelEP.getExtension(modelId);
            
            // for this model, which parameters are of type "datasource"
            List<String> datasourceParameters = model.getModelItems()
                                                     .stream()
                                                     .filter(parameter -> ModelItemTypeConstants.DATASOURCE_ELEMENT_TYPE_ID.equals(parameter.getType().getId()))
                                                     .map(ModelItem::getPath)
                                                     .toList();
            
            // search the collections of this model
            for (SynchronizableContentsCollection collection : _synchronizableContentsCollectionDAO.getSynchronizableContentsCollections())
            {
                if (modelId.equals(collection.getSynchronizeCollectionModelId()))
                {
                    for (String datasourceParameter : datasourceParameters)
                    {
                        if (id.equals(collection.getParameterValues().get(datasourceParameter)))
                        {
                            return TypeOfUse.NON_BLOCKING; // A data source use by a collection is always non-blocking
                        }
                    }
                }
            }
        }
        
        return TypeOfUse.NOT_USED;
    }

    @Override
    public Map<String, TypeOfUse> getUsedDataSourceIds()
    {
        Map<String, TypeOfUse> result = new HashMap<>();
        
        for (String modelId : _synchronizeContentsCollectionModelEP.getExtensionsIds())
        {
            SynchronizableContentsCollectionModel model = _synchronizeContentsCollectionModelEP.getExtension(modelId);
            
            // for this model, which parameters are of type "datasource"
            List<String> datasourceParameters = model.getModelItems()
                                                     .stream()
                                                     .filter(parameter -> ModelItemTypeConstants.DATASOURCE_ELEMENT_TYPE_ID.equals(parameter.getType().getId()))
                                                     .map(ModelItem::getPath)
                                                     .toList();
            
            // search the collections of this model
            for (SynchronizableContentsCollection collection : _synchronizableContentsCollectionDAO.getSynchronizableContentsCollections())
            {
                for (String datasourceParameter : datasourceParameters)
                {
                    // this datasource value is used
                    result.put((String) collection.getParameterValues().get(datasourceParameter), TypeOfUse.NON_BLOCKING); // A data source use by a collection is always non-blocking
                }
            }
        }
        
        return result;
    }
}
