/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.synchronize.systemprop;

import java.util.Map;
import java.util.Optional;
import java.util.stream.Stream;

import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.repository.Content;
import org.ametys.cms.search.model.SystemProperty;
import org.ametys.cms.search.query.Query;
import org.ametys.cms.search.query.Query.Operator;
import org.ametys.cms.search.systemprop.AbstractIndexableSystemProperty;
import org.ametys.core.util.LambdaUtils;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollection;
import org.ametys.plugins.contentio.synchronize.search.query.CollectionsQuery;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.type.DataContext;
import org.ametys.runtime.model.type.ModelItemTypeConstants;

/**
 * {@link SystemProperty} which represents the synchronizable contents collections of a content.
 */
public class CollectionsSystemProperty extends AbstractIndexableSystemProperty<String, String, Content>
{
    /** Solr field name. */
    public static final String SOLR_FIELD_NAME = "scc";
    
    @Override
    public Query getQuery(Object value, Operator operator, String language, Map<String, Object> contextualParameters)
    {
        String[] synchronizableContentsCollectionIds = parseStringArray(value);
        Operator allowedOperator = _allowedOperator(operator);
        return new CollectionsQuery(allowedOperator, synchronizableContentsCollectionIds);
    }
    
    // Just to avoid an exception if a not-allowed operator is passed (for instance, SEARCH, etc.)
    // The problem is that currently, we cannot restrain the proposed operator to the UI,
    // and the operator for STRING type are plentiful, whereas CollectionsQuery only supports EQ and NE
    private Operator _allowedOperator(Operator requestedOperator)
    {
        return requestedOperator == Operator.NE
                ? requestedOperator
                : Operator.EQ;
    }
    
    @Override
    public String getSolrFieldName()
    {
        return SOLR_FIELD_NAME;
    }
    
    @Override
    public String getSolrSortFieldName()
    {
        return SOLR_FIELD_NAME;
    }
    
    @Override
    public String getSolrFacetFieldName()
    {
        return SOLR_FIELD_NAME + "_dv";
    }
    
    @Override
    public Object getValue(Content content)
    {
        return content.getInternalDataHolder()
                      .getValue(SynchronizableContentsCollection.COLLECTION_ID_DATA_NAME, new String[0]);
    }
    
    public Object valueToJSON(Content content, DataContext context)
    {
        return Stream.of((String[]) getValue(content))
            .map(sccId -> Map.of(
                "value", sccId,
                "label", _getLabel(sccId)
            ))
            .toList();
    }
    
    public void valueToSAX(ContentHandler contentHandler, Content content, DataContext context) throws SAXException
    {
        String[] sccIds = (String[]) getValue(content);
        for (String sccId : sccIds)
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("value", sccId);
            XMLUtils.startElement(contentHandler, "scc", attrs);
            _getLabel(sccId).toSAX(contentHandler);
            XMLUtils.endElement(contentHandler, "scc");
        }
    }
    
    private I18nizableText _getLabel(String sccId)
    {
        return Optional.of(sccId)
            // Exception should not happen because the CollectionEnumerator does not return any exception
            .map(LambdaUtils.wrap(getEnumerator()::getEntry))
            // Return the ID it self it no entry
            .orElseGet(() -> new I18nizableText(sccId));
    }
    
    @Override
    protected String getTypeId()
    {
        return ModelItemTypeConstants.STRING_TYPE_ID;
    }
}
