/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.archive;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.authentication.AuthenticateAction;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.mail.SendMailHelper;
import org.ametys.plugins.core.impl.schedule.AbstractStaticSchedulable;
import org.ametys.runtime.config.Config;

/**
 * Common code for {@link ExportArchiveSchedulable} and {@link ImportArchiveSchedulable}
 */
abstract class AbstractArchiveSchedulable extends AbstractStaticSchedulable
{
    protected I18nUtils _i18nUtils;
    protected ArchiveHandler _archiveHandler;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _archiveHandler = (ArchiveHandler) manager.lookup(ArchiveHandler.ROLE);
    }
    
    protected final User _getUser()
    {
        Request request = ContextHelper.getRequest(_context);
        UserIdentity userIdentity = AuthenticateAction.getUserIdentityFromSession(request);
        User user = userIdentity != null ? _userManager.getUser(userIdentity) : null;
        
        return user;
    }
    
    protected final void _sendMail(String subject, String body, String userEmail)
    {
        String adminEmail = Config.getInstance().getValue("smtp.mail.sysadminto");
        String email = StringUtils.isNotEmpty(userEmail) ? userEmail : StringUtils.isNotEmpty(adminEmail) ? adminEmail : null;
        
        if (email != null)
        {
            try
            {
                SendMailHelper.newMail()
                              .withSubject(subject)
                              .withHTMLBody(body)
                              .withRecipient(email)
                              .sendMail();
            }
            catch (Exception e)
            {
                getLogger().warn("Unable to send archiving/import result by email to " + email, e);
            }
        }
        else
        {
            getLogger().warn("An email should have been sent after archiving/import process but the current user has no email and the admin email is not set in the configuration.");
        }
    }
}

