/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.archive;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Path;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.nodetype.NodeType;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.sax.SAXTransformerFactory;
import javax.xml.transform.sax.TransformerHandler;
import javax.xml.transform.stream.StreamResult;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.RegExUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.jackrabbit.core.NodeImpl;
import org.apache.xml.serializer.OutputPropertiesFactory;
import org.slf4j.Logger;
import org.xml.sax.SAXException;

import org.ametys.cms.data.Binary;
import org.ametys.cms.data.NamedResource;
import org.ametys.cms.data.RichText;
import org.ametys.cms.data.type.ModelItemTypeConstants;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.plugins.repository.data.holder.impl.DataHolderHelper;

/**
 * Convenient methods for {@link Archiver} API implementations
 */
public final class Archivers
{
    /** The warning message for a root of {@link Archiver} which stil has pending change while we do a unitary save on every imported object. */
    public static final String WARN_MESSAGE_ROOT_HAS_PENDING_CHANGES = "{} still has pending changes while we should have saved unitarily every object. Another save will be done, but it is not normal.";
    
    static final String __BINARY_ATTRIBUTES_FOLDER_NAME = "_binaryAttributes";
    static final String __FILE_ATTRIBUTES_FOLDER_NAME = "_fileAttributes";
    static final String __RICH_TEXT_ATTACHMENTS_FOLDER_NAME = "_richTextAttachments";
    
    private static final Properties __OUTPUT_FORMAT_PROPERTIES = new Properties();
    static
    {
        __OUTPUT_FORMAT_PROPERTIES.put(OutputKeys.ENCODING, "UTF-8");
        __OUTPUT_FORMAT_PROPERTIES.put(OutputKeys.METHOD, "xml");
        __OUTPUT_FORMAT_PROPERTIES.put(OutputKeys.INDENT, "yes");
        __OUTPUT_FORMAT_PROPERTIES.put(OutputPropertiesFactory.S_KEY_INDENT_AMOUNT, "4");
    }
    
    private static SAXTransformerFactory __transformerFactory;
    
    private Archivers()
    {
        // Nothing
    }
    
    /**
     * Gets a {@link SAXTransformerFactory}
     * @return a {@link SAXTransformerFactory}
     */
    public static SAXTransformerFactory getSaxTransformerFactory()
    {
        if (__transformerFactory == null)
        {
            __transformerFactory = (SAXTransformerFactory) TransformerFactory.newInstance();
        }
        return __transformerFactory;
    }
    
    /**
     * Get a TransformerHandler object that can process SAXContentHandler events into a Result
     * @return A non-null reference to a TransformerHandler, that maybe used as a ContentHandler for SAX parse events.
     * @throws TransformerConfigurationException If for some reason theTransformerHandler cannot be created.
     */
    public static TransformerHandler newTransformerHandler() throws TransformerConfigurationException
    {
        TransformerHandler transformerHandler = getSaxTransformerFactory().newTransformerHandler();
        setStandardOutputProperties(transformerHandler);
        return transformerHandler;
    }
    
    /**
     * Sets standard output properties to the transformer of the given handler, such as encoding and indentation.
     * @param transformerHandler The transformer handler
     */
    public static void setStandardOutputProperties(TransformerHandler transformerHandler)
    {
        transformerHandler.getTransformer().setOutputProperties(__OUTPUT_FORMAT_PROPERTIES);
    }
    
    /**
     * Export ACL sub-node
     * @param node The JCR node
     * @param zos the ZIP OutputStream.
     * @param path the zip entry path
     * @throws RepositoryException if an error occurs
     * @throws IOException if an I/O error occurs
     */
    public static void exportAcl(Node node, ZipOutputStream zos, String path) throws RepositoryException, IOException
    {
        try
        {
            if (node.hasNode("ametys-internal:acl"))
            {
                ZipEntry aclEntry = new ZipEntry(path);
                zos.putNextEntry(aclEntry);
                
                TransformerHandler aclHandler = Archivers.newTransformerHandler();
                aclHandler.setResult(new StreamResult(zos));
                
                node.getSession().exportSystemView(node.getNode("ametys-internal:acl").getPath(), aclHandler, true, false);
            }
        }
        catch (SAXException | TransformerConfigurationException e)
        {
            throw new RuntimeException("Unable to SAX ACL for node '" + node.getPath() + "' for archiving", e);
        }
    }
    
    /**
     * Import ACL sub-node
     * @param node The JCR node
     * @param zipPath the input zip path
     * @param merger The {@link Merger}
     * @param zipEntryPath the zip entry path
     * @param logger The logger
     * @throws RepositoryException if an error occurs
     * @throws IOException if an I/O error occurs
     */
    public static void importAcl(Node node, Path zipPath, Merger merger, String zipEntryPath, Logger logger) throws RepositoryException, IOException
    {
        if (ZipEntryHelper.zipEntryFileExists(zipPath, zipEntryPath))
        {
            if (node.hasNode(RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":acl"))
            {
                Node existingAclNode = node.getNode(RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":acl");
                logger.info("Existing ACL node '{}' will be removed", existingAclNode);
                existingAclNode.remove();
            }
            Session session = node.getSession();
            String parentAbsPath = node.getPath();
            int uuidBehavior = merger.getImportUuidBehavior();
            try (InputStream in = ZipEntryHelper.zipEntryFileInputStream(zipPath, zipEntryPath))
            {
                session.importXML(parentAbsPath, in, uuidBehavior);
                logger.info("XML from '{}!{}' imported to '{}' with uuidBehavior '{}'", zipPath, zipEntryPath, parentAbsPath, uuidBehavior);
            }
        }
    }
    
    /**
     * Evaluates a non-empty XPath query. If the result is empty, an {@link AmetysObjectNotImportedException} is thrown.
     * @param xPath The XPath query
     * @param domNode The DOM node
     * @return The evaluation as String
     * @throws AmetysObjectNotImportedException If the result is empty
     * @throws TransformerException if an evaluation error occured
     */
    public static String xpathEvalNonEmpty(String xPath, org.w3c.dom.Node domNode) throws AmetysObjectNotImportedException, TransformerException
    {
        String val = DomNodeHelper.nullableStringValue(domNode, xPath);
        if (StringUtils.isEmpty(val))
        {
            throw new AmetysObjectNotImportedException(String.format("'%s' xPath value is empty while it is required. The Ametys Object could not be imported.", xPath));
        }
        return val;
    }
    
    /**
     * Exception indicating an Ametys Object cannot be imported
     */
    public static final class AmetysObjectNotImportedException extends Exception
    {
        /**
         * Constructor with detail message
         * @param message The detail message
         */
        public AmetysObjectNotImportedException(String message)
        {
            super(message);
        }
        
        /**
         * Constructor with cause
         * @param cause The cause
         */
        public AmetysObjectNotImportedException(Throwable cause)
        {
            super(cause);
        }
    }
    
    /**
     * Replace the given JCR Node by a copy of it with the given UUID. The source JCR Node is removed.
     * @param srcNode The source JCR Node
     * @param uuid The desired UUID
     * @return The JCR Node with the desired UUID
     * @throws RepositoryException if an error occurs
     */
    public static Node replaceNodeWithDesiredUuid(Node srcNode, String uuid) throws RepositoryException
    {
        // The passed 'srcNode' was created just to create its node hierarchy and retrieve its mixin node types
        // But immediatly after that, remove it because the uuid was not chosen
        NodeImpl parentNode = (NodeImpl) srcNode.getParent();
        String name = srcNode.getName();
        String type = srcNode.getPrimaryNodeType().getName();
        NodeType[] mixinNodeTypes = srcNode.getMixinNodeTypes();
        srcNode.remove();
        
        // Add a node with the desired uuid at the same place than the first one (which was just removed)
        // And set its mixin node types
        Node nodeWithDesiredUuid = parentNode.addNodeWithUuid(name, type, uuid);
        for (NodeType mixinNodeType : mixinNodeTypes)
        {
            nodeWithDesiredUuid.addMixin(mixinNodeType.getName());
        }
        
        return nodeWithDesiredUuid;
    }
    
    /**
     * Save the pending changes brought to this node associated to an {@link AmetysObject}
     * <br>If the save failed, it is logged in ERROR level and the changes are discarded.
     * @param ametysObjectNode The node
     * @param logger The logger
     * @throws AmetysObjectNotImportedException If the save failed
     * @throws ImportGlobalFailException If a severe error occured and the global import process must be stopped
     */
    public static void unitarySave(Node ametysObjectNode, Logger logger) throws AmetysObjectNotImportedException, ImportGlobalFailException
    {
        Session session;
        try
        {
            session = ametysObjectNode.getSession();
        }
        catch (RepositoryException e)
        {
            // Cannot even retrieve a session...
            throw new ImportGlobalFailException(e);
        }
        
        try
        {
            logger.info("Saving '{}'...", ametysObjectNode);
            session.save();
        }
        catch (RepositoryException saveFailedException)
        {
            logger.error("Save did not succeed, changes on current object '{}' will be discarded...", ametysObjectNode, saveFailedException);
            try
            {
                session.refresh(false);
            }
            catch (RepositoryException refreshFailedException)
            {
                // rollback did not succeed, global fail is inevitable...
                throw new ImportGlobalFailException(refreshFailedException);
            }
            
            // rollback succeeded, throw AmetysObjectNotImportedException to indicate the save was a failure and thus, the object was not imported
            throw new AmetysObjectNotImportedException(saveFailedException);
        }
    }
    
    /**
     * Export the attachments of the given data holder's rich texts
     * @param dataHolder The data holder
     * @param zos The {@link ZipOutputStream} where to export the attachments
     * @param path The path of the folder used to export the attachments
     * @throws IOException if an error occurs while exporting the attachments
     */
    public static void exportRichTexts(ModelAwareDataHolder dataHolder, ZipOutputStream zos, String path) throws IOException
    {
        String prefix = path + __RICH_TEXT_ATTACHMENTS_FOLDER_NAME + "/";
        Map<String, Object> richTexts = DataHolderHelper.findItemsByType(dataHolder, ModelItemTypeConstants.RICH_TEXT_ELEMENT_TYPE_ID);
        for (Entry<String, Object> entry : richTexts.entrySet())
        {
            String richTextDataPath = entry.getKey();
            Object value = entry.getValue();
            if (value instanceof RichText)
            {
                _exportRichText(richTextDataPath, (RichText) value, zos, prefix);
            }
            else if (value instanceof RichText[])
            {
                for (RichText richText : (RichText[]) value)
                {
                    _exportRichText(richTextDataPath, richText, zos, prefix);
                }
            }
        }
    }
    
    private static void _exportRichText(String richTextDataPath, RichText richText, ZipOutputStream zos, String prefix) throws IOException
    {
        for (NamedResource resource : richText.getAttachments())
        {
            _exportResource(richTextDataPath, resource, zos, prefix);
        }
    }
    
    /**
     * Export the given data holder's binaries
     * @param dataHolder The data holder
     * @param zos The {@link ZipOutputStream} where to export the binaries
     * @param path The path of the folder used to export the binaries
     * @throws IOException if an error occurs while exporting the binaries
     */
    public static void exportBinaries(ModelAwareDataHolder dataHolder, ZipOutputStream zos, String path) throws IOException
    {
        String prefix = path + __BINARY_ATTRIBUTES_FOLDER_NAME + "/";
        Map<String, Object> binaries = DataHolderHelper.findItemsByType(dataHolder, ModelItemTypeConstants.BINARY_ELEMENT_TYPE_ID);
        for (Entry<String, Object> entry : binaries.entrySet())
        {
            String binaryDataPath = entry.getKey();
            Object value = entry.getValue();
            if (value instanceof Binary)
            {
                _exportResource(binaryDataPath, (Binary) value, zos, prefix);
            }
            else if (value instanceof Binary[])
            {
                for (Binary binary : (Binary[]) value)
                {
                    _exportResource(binaryDataPath, binary, zos, prefix);
                }
            }
        }
    }
    
    /**
     * Export the given data holder's files
     * @param dataHolder The data holder
     * @param zos The {@link ZipOutputStream} where to export the files
     * @param path The path of the folder used to export the files
     * @throws IOException if an error occurs while exporting the files
     */
    public static void exportFiles(ModelAwareDataHolder dataHolder, ZipOutputStream zos, String path) throws IOException
    {
        String prefix = path + __FILE_ATTRIBUTES_FOLDER_NAME + "/";
        Map<String, Object> files = DataHolderHelper.findItemsByType(dataHolder, ModelItemTypeConstants.FILE_ELEMENT_TYPE_ID);
        for (Entry<String, Object> entry : files.entrySet())
        {
            String fileDataPath = entry.getKey();
            Object value = entry.getValue();
            if (value instanceof Binary)
            {
                _exportResource(fileDataPath, (Binary) value, zos, prefix);
            }
            else if (value instanceof Binary[])
            {
                for (Binary binary : (Binary[]) value)
                {
                    _exportResource(fileDataPath, binary, zos, prefix);
                }
            }
        }
    }
    
    private static void _exportResource(String dataPath, NamedResource resource, ZipOutputStream zos, String prefix) throws IOException
    {
        String resourcePath = getFolderPathFromDataPath(dataPath);
        ZipEntry newEntry = new ZipEntry(prefix + resourcePath + "/" + resource.getFilename());
        zos.putNextEntry(newEntry);
        
        try (InputStream is = resource.getInputStream())
        {
            IOUtils.copy(is, zos);
        }
    }
    
    /**
     * Retrieves a folder path from a data path.
     * Replaces all repeater entries like '[x]', to a folder with the position ('/x')
     * @param dataPath the data path
     * @return the folder path
     */
    public static String getFolderPathFromDataPath(String dataPath)
    {
        return RegExUtils.replaceAll(dataPath, "\\[([0-9]+)\\]", "/$1");
    }
}
