/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.archive;

import java.io.IOException;
import java.net.URI;
import java.nio.file.Path;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import javax.jcr.AccessDeniedException;
import javax.jcr.ItemNotFoundException;
import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.sax.TransformerHandler;
import javax.xml.transform.stream.StreamResult;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.w3c.dom.Document;
import org.xml.sax.SAXException;

import org.ametys.cms.CmsConstants;
import org.ametys.cms.content.references.OutgoingReferences;
import org.ametys.cms.content.references.OutgoingReferencesExtractor;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.DefaultContent;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.cms.repository.ModifiableContentHelper;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.cms.repository.WorkflowAwareContentHelper;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.contentio.archive.Archivers.AmetysObjectNotImportedException;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.TraversableAmetysObject;
import org.ametys.plugins.repository.collection.AmetysObjectCollection;
import org.ametys.plugins.repository.data.extractor.xml.XMLValuesExtractorAdditionalDataGetter;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.plugins.repository.jcr.JCRTraversableAmetysObject;
import org.ametys.plugins.repository.jcr.NodeHelper;
import org.ametys.plugins.repository.version.VersionableAmetysObject;
import org.ametys.plugins.workflow.support.WorkflowProvider;
import org.ametys.plugins.workflow.support.WorkflowProvider.AmetysObjectWorkflow;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import com.opensymphony.workflow.WorkflowException;
import com.opensymphony.workflow.spi.Step;

/**
 * Export a contents collection as individual XML files.
 */
public class ContentsArchiverHelper extends AbstractLogEnabled implements Component, Serviceable
{
    /** Avalon role. */
    public static final String ROLE = ContentsArchiverHelper.class.getName();
    
    private static final String __CONTENT_ZIP_ENTRY_FILENAME = "content.xml";
    private static final String __ACL_ZIP_ENTRY_FILENAME = "_acl.xml";
    
    private AmetysObjectResolver _resolver;
    private ResourcesArchiverHelper _resourcesArchiverHelper;
    private WorkflowProvider _workflowProvider;
    private ModifiableContentHelper _modifiableContentHelper;
    private OutgoingReferencesExtractor _outgoingReferencesExtractor;
    private ContentTypeExtensionPoint _contentTypeEP;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _resourcesArchiverHelper = (ResourcesArchiverHelper) manager.lookup(ResourcesArchiverHelper.ROLE);
        _workflowProvider = (WorkflowProvider) manager.lookup(WorkflowProvider.ROLE);
        _modifiableContentHelper = (ModifiableContentHelper) manager.lookup(ModifiableContentHelper.ROLE);
        _outgoingReferencesExtractor = (OutgoingReferencesExtractor) manager.lookup(OutgoingReferencesExtractor.ROLE);
        _contentTypeEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
    }

    /**
     * Exports contents from a root Node.
     * @param prefix the prefix for the ZIP archive.
     * @param rootNode the root JCR Node holding the contents collection.
     * @param zos the ZIP OutputStream.
     * @throws RepositoryException if an error occurs while resolving Node.
     * @throws IOException if an error occurs while archiving
     */
    public void exportContents(String prefix, Node rootNode, ZipOutputStream zos) throws RepositoryException, IOException
    {
        TraversableAmetysObject rootContents = _resolver.resolve(rootNode, false);
        exportContents(prefix, rootContents, zos);
    }
    
    /**
     * Exports contents from a root AmetysObject.
     * @param prefix the prefix for the ZIP archive.
     * @param rootContents the root JCR Node holding the contents collection.
     * @param zos the ZIP OutputStream.
     * @throws IOException if an error occurs while archiving
     */
    public void exportContents(String prefix, TraversableAmetysObject rootContents, ZipOutputStream zos) throws IOException
    {
        zos.putNextEntry(new ZipEntry(StringUtils.appendIfMissing(prefix, "/"))); // even if there is no child, at least export the root of contents
        
        AmetysObjectIterable<Content> contents = rootContents.getChildren();
        for (Content content : contents)
        {
            try
            {
                _exportContent(prefix, content, zos);
            }
            catch (Exception e)
            {
                throw new RuntimeException("An error occured while exporting content '" + content.getId() + "'.", e);
            }
        }
        
        // finally process ACL for the contents root
        try
        {
            Node contentNode = ((JCRAmetysObject) rootContents).getNode();
            Archivers.exportAcl(contentNode, zos, prefix + __ACL_ZIP_ENTRY_FILENAME);
        }
        catch (RepositoryException e)
        {
            throw new RuntimeException("Unable to SAX ACL for root contents at '" + rootContents.getPath() + "' for archiving", e);
        }
    }
    
    private void _exportContent(String prefix, Content content, ZipOutputStream zos) throws IOException
    {
        List<String> unexistingContentTypesAndMixins = Stream.of(content.getTypes(), content.getMixinTypes())
                .flatMap(Stream::of)
                .filter(Predicate.not(_contentTypeEP::hasExtension))
                .collect(Collectors.toList());
        if (!unexistingContentTypesAndMixins.isEmpty())
        {
            getLogger().error("Content \"{}\" will not be exported as at least one of its types or mixins does not exist: {}", content, unexistingContentTypesAndMixins);
            return;
        }
        
        // for each content, first an XML file with attributes, comments, tags, ...
        String name = content.getName();
        String path = prefix + NodeHelper.getFullHashPath(name) + "/";
        ZipEntry contentEntry = new ZipEntry(path + __CONTENT_ZIP_ENTRY_FILENAME);
        zos.putNextEntry(contentEntry);
        
        try
        {
            TransformerHandler contentHandler = Archivers.newTransformerHandler();
            contentHandler.setResult(new StreamResult(zos));
            
            contentHandler.startDocument();
            content.toSAX(contentHandler, null, null, true);
            contentHandler.endDocument();
        }
        catch (SAXException | TransformerConfigurationException e)
        {
            throw new RuntimeException("Unable to SAX content '" + content.getPath() + "' for archiving", e);
        }
        
        // then all attachments
        _resourcesArchiverHelper.exportCollection(content.getRootAttachments(), zos, path + "_attachments/");
        
        // then all files local to rich texts (images)
        Archivers.exportRichTexts(content, zos, path);
        
        // then all binary attributes
        Archivers.exportBinaries(content, zos, path);
        
        // then all file attributes
        Archivers.exportFiles(content, zos, path);
        
        // then ACL
        try
        {
            Node contentNode = ((JCRAmetysObject) content).getNode();
            Archivers.exportAcl(contentNode, zos, path + __ACL_ZIP_ENTRY_FILENAME);
        }
        catch (RepositoryException e)
        {
            throw new RuntimeException("Unable to SAX ACL for content '" + content.getPath() + "' for archiving", e);
        }
    }
    
    /**
     * Imports contents from the given ZIP archive and path, under the given root of contents
     * @param commonPrefix The common prefix in the ZIP archive
     * @param rootContents the root {@link JCRTraversableAmetysObject} holding the contents collection.
     * @param zipPath the input zip path
     * @param merger The {@link Merger}
     * @param contentFillers The fillers in order to fill additional attributes on imported contents
     * @return The {@link ImportReport}
     * @throws IOException if an error occurs while importing archive
     */
    public ImportReport importContents(String commonPrefix, AmetysObjectCollection rootContents, Path zipPath, Merger merger, Collection<ContentFiller> contentFillers) throws IOException
    {
        Importer importer;
        List<DefaultContent> createdContents;
        try
        {
            importer = new Importer(commonPrefix, rootContents, zipPath, merger, contentFillers, getLogger());
            createdContents = importer.importRoot();
        }
        catch (ParserConfigurationException e)
        {
            throw new IOException(e);
        }
        _saveContents(rootContents);
        _checkoutContents(createdContents);
        return importer._report;
    }
    
    private void _saveContents(AmetysObjectCollection rootContents)
    {
        if (rootContents.needsSave())
        {
            getLogger().warn(Archivers.WARN_MESSAGE_ROOT_HAS_PENDING_CHANGES, rootContents);
            rootContents.saveChanges();
        }
    }

    private void _checkoutContents(List<DefaultContent> createdContents)
    {
        for (DefaultContent createdContent : createdContents)
        {
            createdContent.checkpoint();
        }
    }
    
    /**
     * A filler in order to fill additional attributes on imported contents
     */
    @FunctionalInterface
    public static interface ContentFiller
    {
        /**
         * Fill the content with additional attributes
         * @param content The imported content
         */
        void fillContent(DefaultContent content);
    }
    
    private class Importer
    {
        final ImportReport _report = new ImportReport();
        private final String _commonPrefix;
        private final AmetysObjectCollection _root;
        private final Path _zipArchivePath;
        private final Merger _merger;
        private final Collection<ContentFiller> _contentFillers;
        private final Logger _logger;
        private final DocumentBuilder _builder;
        private final UnitaryContentImporter _unitaryImporter = new UnitaryContentImporter();
        
        Importer(String commonPrefix, AmetysObjectCollection root, Path zipArchivePath, Merger merger, Collection<ContentFiller> contentFillers, Logger logger) throws ParserConfigurationException
        {
            _commonPrefix = commonPrefix;
            _root = root;
            _zipArchivePath = zipArchivePath;
            _merger = merger;
            _contentFillers = contentFillers;
            _logger = logger;
            _builder = DocumentBuilderFactory.newInstance()
                    .newDocumentBuilder();
        }
        
        List<DefaultContent> importRoot() throws IOException
        {
            _fillRoot();
            
            try (Stream<Path> zippedFiles = _matchingZippedFiles())
            {
                // no stream pipeline here because exception flow is important
                List<DefaultContent> createdContents = new ArrayList<>();
                for (Path zipEntryPath : zippedFiles.toArray(Path[]::new))
                {
                    Optional<DefaultContent> createdContent = _importContent(zipEntryPath);
                    createdContent.ifPresent(createdContents::add);
                }
                return createdContents;
            }
        }
        
        private void _fillRoot() throws IOException
        {
            _createRootContentAcl();
            try
            {
                Archivers.unitarySave(_root.getNode(), _logger);
            }
            catch (AmetysObjectNotImportedException e)
            {
                // ACL were not exported => it was already logged in error level, and it does not affect the future import of contents => continue
            }
        }
        
        private void _createRootContentAcl() throws IOException
        {
            Node rootNode = _root.getNode();
            String zipEntryPath = new StringBuilder()
                    .append(StringUtils.strip(_commonPrefix, "/"))
                    .append("/")
                    .append(__ACL_ZIP_ENTRY_FILENAME)
                    .toString();
            _createAcl(rootNode, zipEntryPath);
        }
        
        private void _createContentAcl(Node contentNode, Path contentZipEntryPath) throws IOException
        {
            String zipEntryPath = contentZipEntryPath
                    .getParent()
                    .resolve(__ACL_ZIP_ENTRY_FILENAME)
                    .toString();
            _createAcl(contentNode, zipEntryPath);
        }
        
        private void _createAcl(Node node, String zipAclEntryPath) throws IOException
        {
            try
            {
                _logger.debug("Trying to import ACL node for Content (or root of contents) '{}', from ACL XML file '{}', if it exists", node, zipAclEntryPath);
                Archivers.importAcl(node, _zipArchivePath, _merger, zipAclEntryPath, _logger);
            }
            catch (RepositoryException e)
            {
                throw new IOException(e);
            }
        }
        
        private Stream<Path> _matchingZippedFiles() throws IOException
        {
            return ZipEntryHelper.zipFileTree(
                _zipArchivePath,
                Optional.of(_commonPrefix),
                (Path p, BasicFileAttributes attrs) ->
                        !attrs.isDirectory()
                        && __CONTENT_ZIP_ENTRY_FILENAME.equals(p.getFileName().toString()));
        }
        
        private Optional<DefaultContent> _importContent(Path zipEntryPath) throws ImportGlobalFailException
        {
            return _unitaryImporter.unitaryImport(_zipArchivePath, zipEntryPath, _merger, _logger);
        }
        
        private Document _getContentPropertiesXml(Path zipEntryPath) throws SAXException, IOException
        {
            URI zipEntryUri = zipEntryPath.toUri();
            return _builder.parse(zipEntryUri.toString());
        }
        
        private DefaultContent _createContent(Path contentZipEntry, Document propertiesXml) throws AmetysObjectNotImportedException, Exception
        {
            // At first, check the content types and mixins exist in the current application, otherwise do not import the content ASAP
            String[] contentTypes = _retrieveContentTypes(contentZipEntry, propertiesXml, "content/contentTypes/contentType");
            String[] mixins = _retrieveContentTypes(contentZipEntry, propertiesXml, "content/mixins/mixin");
            
            // Create the JCR Node
            String uuid = Archivers.xpathEvalNonEmpty("content/@uuid", propertiesXml);
            String contentDesiredName = Archivers.xpathEvalNonEmpty("content/@name", propertiesXml);
            String type =  Archivers.xpathEvalNonEmpty("content/@primaryType", propertiesXml);
            _logger.info("Creating a Content object for '{}' file (uuid={}, type={}, desiredName={})", contentZipEntry, uuid, type, contentDesiredName);
            
            DefaultContent createdContent = _createChild(uuid, contentDesiredName, type);
            
            // Set mandatory properties
            _setContentMandatoryProperties(createdContent, contentTypes, mixins, propertiesXml);
            // Set other properties
            _fillContentNode(createdContent, propertiesXml, contentZipEntry);
            // Set content attachments
            ImportReport importAttachmentReport = _fillContentAttachments(createdContent, contentZipEntry);
            _report.addFrom(importAttachmentReport);
            // Fill other attributes
            _fillAdditionalContentAttributes(createdContent);
            // Outgoing references
            _setOutgoingReferences(createdContent);
            
            // Initialize workflow
            if (createdContent instanceof WorkflowAwareContent)
            {
                String workflowName = Archivers.xpathEvalNonEmpty("content/workflow-step/@workflowName", propertiesXml);
                _handleWorkflow(workflowName, (WorkflowAwareContent) createdContent);
            }
            
            return createdContent;
        }
        
        private String[] _retrieveContentTypes(Path contentZipEntry, Document propertiesXml, String xPath) throws TransformerException, AmetysObjectNotImportedException
        {
            String[] contentTypes = DomNodeHelper.stringValues(propertiesXml, xPath);
            List<String> unexistingTypes = Stream.of(contentTypes)
                    .filter(Predicate.not(_contentTypeEP::hasExtension))
                    .collect(Collectors.toList());
            if (!unexistingTypes.isEmpty())
            {
                String message = String.format("Content defined in '%s' has at least one of its types or mixins which does not exist: %s", contentZipEntry, unexistingTypes);
                throw new AmetysObjectNotImportedException(message);
            }
            return contentTypes;
        }
        
        private DefaultContent _createChild(String uuid, String contentDesiredName, String type) throws AccessDeniedException, ItemNotFoundException, RepositoryException
        {
            // Create a content with AmetysObjectCollection.createChild
            String unusedContentName = _getUnusedContentName(contentDesiredName);
            JCRAmetysObject srcContent = (JCRAmetysObject) _root.createChild(unusedContentName, type);
            Node srcNode = srcContent.getNode();
            // But then call 'replaceNodeWithDesiredUuid' to have it with the desired UUID (srcNode will be removed)
            Node nodeWithDesiredUuid = Archivers.replaceNodeWithDesiredUuid(srcNode, uuid);
            
            // Then resolve and return a Content
            String parentPath = _root.getPath();
            DefaultContent createdContent = _resolver.resolve(parentPath, nodeWithDesiredUuid, null, false);
            return createdContent;
        }
        
        // ~ same algorithm than org.ametys.cms.workflow.CreateContentFunction._createContent
        // no use of org.ametys.cms.FilterNameHelper.filterName because it was already filtered during the export (taken from the existing content name)
        private String _getUnusedContentName(String desiredName)
        {
            String contentName = desiredName;
            for (int errorCount = 0; true; errorCount++)
            {
                if (errorCount != 0)
                {
                    _logger.debug("Name '{}' from Content is already used. Trying another one...", contentName);
                    contentName = desiredName + "-" + (errorCount + 1);
                }
                if (!_root.hasChild(contentName))
                {
                    _logger.debug("Content will be created with unused name '{}'. The desired name was '{}'", contentName, desiredName);
                    return contentName;
                }
            }
        }
        
        private void _setContentMandatoryProperties(DefaultContent content, String[] contentTypes, String[] mixins, Document propertiesXml) throws TransformerException, AmetysObjectNotImportedException
        {
            content.setTypes(contentTypes);
            content.setMixinTypes(mixins);
            
            Date creationDate = Objects.requireNonNull(DomNodeHelper.nullableDatetimeValue(propertiesXml, "content/@createdAt"));
            _modifiableContentHelper.setCreationDate(content, DateUtils.asZonedDateTime(creationDate));
            
            String creator = Archivers.xpathEvalNonEmpty("content/@creator", propertiesXml);
            _modifiableContentHelper.setCreator(content, UserIdentity.stringToUserIdentity(creator));
            
            Date lastModifiedAt = Objects.requireNonNull(DomNodeHelper.nullableDatetimeValue(propertiesXml, "content/@lastModifiedAt"));
            _modifiableContentHelper.setLastModified(content, DateUtils.asZonedDateTime(lastModifiedAt));
            
            String lastContributor = Archivers.xpathEvalNonEmpty("content/@lastContributor", propertiesXml);
            _modifiableContentHelper.setLastContributor(content, UserIdentity.stringToUserIdentity(lastContributor));
        }
        
        private void _fillContentNode(DefaultContent content, Document propertiesXml, Path contentZipEntry) throws TransformerException, Exception
        {
            String language = DomNodeHelper.nullableStringValue(propertiesXml, "content/@language");
            if (language != null)
            {
                content.setLanguage(language);
            }
            
            Date lastValidatedAt = DomNodeHelper.nullableDatetimeValue(propertiesXml, "content/@lastValidatedAt");
            if (lastValidatedAt != null)
            {
                _modifiableContentHelper.setLastValidationDate(content, DateUtils.asZonedDateTime(lastValidatedAt));
            }
            
            if (content instanceof ModifiableContent)
            {
                Path contentPath = contentZipEntry.getParent();
                _fillContent((ModifiableContent) content, propertiesXml, contentPath);
            }
        }
        
        private void _fillContent(ModifiableContent content, Document propertiesXml, Path contentZipEntry) throws Exception
        {
            XMLValuesExtractorAdditionalDataGetter additionalDataGetter = new ResourcesAdditionalDataGetter(_zipArchivePath, contentZipEntry);
            content.fillContent(propertiesXml, additionalDataGetter);
        }
        
        private ImportReport _fillContentAttachments(DefaultContent createdContent, Path contentZipEntry) throws IOException, RepositoryException
        {
            Node contentNode = createdContent.getNode();
            
            // ametys-internal:attachments is created automatically
            if (contentNode.hasNode(DefaultContent.ATTACHMENTS_NODE_NAME))
            {
                contentNode.getNode(DefaultContent.ATTACHMENTS_NODE_NAME).remove();
            }
            
            Path contentAttachmentsZipEntryFolder = contentZipEntry.resolveSibling("_attachments/");
            String commonPrefix = StringUtils.appendIfMissing(contentAttachmentsZipEntryFolder.toString(), "/");
            return _resourcesArchiverHelper.importCollection(commonPrefix, contentNode, _zipArchivePath, _merger);
        }
        
        private void _fillAdditionalContentAttributes(DefaultContent content)
        {
            for (ContentFiller contentFiller : _contentFillers)
            {
                contentFiller.fillContent(content);
            }
        }
        
        private void _setOutgoingReferences(DefaultContent content)
        {
            if (content instanceof ModifiableContent)
            {
                Map<String, OutgoingReferences> outgoingReferencesByPath = _outgoingReferencesExtractor.getOutgoingReferences(content);
                ((ModifiableContent) content).setOutgoingReferences(outgoingReferencesByPath);
            }
        }
        
        private void _handleWorkflow(String workflowName, WorkflowAwareContent createdContent) throws WorkflowException
        {
            // In the case of import on the original instance, the version history is still present.
            // As we are about to reset the workflow and potentially change the workflowId
            // We must unpublish any prior version to make sure that we won't publish a version
            // With an incoherent workflowId
            if (createdContent instanceof VersionableAmetysObject vao
                && ArrayUtils.contains(vao.getAllLabels(), CmsConstants.LIVE_LABEL))
            {
                vao.removeLabel(CmsConstants.LIVE_LABEL);
            }
            
            // Then initialize the workflow
            AmetysObjectWorkflow workflow = _workflowProvider.getAmetysObjectWorkflow(createdContent);
            
            int initialAction = 0;
            Map<String, Object> inputs = new HashMap<>(Map.of());
            long workflowId = workflow.initialize(workflowName, initialAction, inputs);
            WorkflowAwareContentHelper.setWorkflowId(createdContent, workflowId);
            
            Step currentStep = (Step) workflow.getCurrentSteps(workflowId).iterator().next();
            createdContent.setCurrentStepId(currentStep.getStepId());
        }
        
        private final class UnitaryContentImporter implements UnitaryImporter<DefaultContent>
        {
            @Override
            public String objectNameForLogs()
            {
                return "Content";
            }

            @Override
            public Document getPropertiesXml(Path zipEntryPath) throws Exception
            {
                return _getContentPropertiesXml(zipEntryPath);
            }

            @Override
            public String retrieveId(Document propertiesXml) throws Exception
            {
                return Archivers.xpathEvalNonEmpty("content/@id", propertiesXml);
            }

            @Override
            public DefaultContent create(Path zipEntryPath, String id, Document propertiesXml) throws AmetysObjectNotImportedException, Exception
            {
                DefaultContent createdContent = _createContent(zipEntryPath, propertiesXml);
                Node contentNode = createdContent.getNode();
                _createContentAcl(contentNode, zipEntryPath);
                Archivers.unitarySave(contentNode, _logger);
                return createdContent;
            }
            
            @Override
            public ImportReport getReport()
            {
                return _report;
            }
        }
    }
}
