/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.archive;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Path;
import java.util.Collections;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.xml.transform.sax.TransformerHandler;
import javax.xml.transform.stream.StreamResult;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.xml.sax.ContentHandler;

import org.ametys.plugins.repository.AmetysObjectFactoryExtensionPoint;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.collection.AmetysObjectCollection;
import org.ametys.plugins.repository.collection.AmetysObjectCollectionFactory;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Default implementation of a {@link PluginArchiver}. It uses the JCR system view for all data but contents.
 * For contents, the {@link ContentsArchiverHelper} is used.
 */
public class DefaultPluginArchiver extends AbstractLogEnabled implements PluginArchiver, Serviceable
{
    /** Id for default implementation */
    public static final String EXTENSION_ID = "__default";
    
    private static final String __CONTENT_ROOT_NODE_NAME = RepositoryConstants.NAMESPACE_PREFIX + ":contents";
    
    private ContentsArchiverHelper _contentsArchiverHelper;
    private AmetysObjectCollectionFactory _ametysObjectCollectionFactory;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _contentsArchiverHelper = (ContentsArchiverHelper) manager.lookup(ContentsArchiverHelper.ROLE);
        AmetysObjectFactoryExtensionPoint ametysObjectFactoryEP = (AmetysObjectFactoryExtensionPoint) manager.lookup(AmetysObjectFactoryExtensionPoint.ROLE);
        _ametysObjectCollectionFactory = (AmetysObjectCollectionFactory) ametysObjectFactoryEP.getExtension(AmetysObjectCollectionFactory.class.getName());
    }

    @Override
    public void export(String pluginName, Node pluginNode, ZipOutputStream zos, String prefix) throws IOException
    {
        // first export non-content data with the JCR system view
        ZipEntry pluginEntry = new ZipEntry(prefix + "/plugin.xml");
        zos.putNextEntry(pluginEntry);
        
        try
        {
            TransformerHandler handler = Archivers.newTransformerHandler();
            handler.setResult(new StreamResult(zos));
            
            pluginNode.getSession().exportSystemView(pluginNode.getPath(), getSystemViewHandler(handler), false, false);
            
            // then the contents if any, one by one
            if (pluginNode.hasNode(__CONTENT_ROOT_NODE_NAME))
            {
                _contentsArchiverHelper.exportContents(prefix + "/contents/", pluginNode.getNode("ametys:contents"), zos);
            }
        }
        catch (Exception e)
        {
            throw new RuntimeException("Unable to archive plugin " + pluginName, e);
        }
    }
    
    /**
     * Returns the actual handler receiving the JCR system view. May be used to filter out some parts of the JCR export.
     * @param initialHandler the target {@link ContentHandler}.
     * @return a ContentHandler.
     */
    protected ContentHandler getSystemViewHandler(ContentHandler initialHandler)
    {
        return new SystemViewHandler(initialHandler, name -> "ametys:contents".equals(name), __ -> false);
    }
    
    @Override
    public ImportReport partialImport(String pluginName, Node allPluginsNode, Path zipPath, String zipPluginEntryPath, Merger merger) throws IOException
    {
        _importPluginXml(allPluginsNode, zipPath, zipPluginEntryPath, merger);
        ImportReport importContentReport = importContentsIfAny(pluginName, allPluginsNode, zipPath, zipPluginEntryPath, merger);
        return ImportReport.union(importContentReport);
    }
    
    private void _importPluginXml(Node allPluginsNode, Path zipPath, String zipPluginEntryPath, Merger merger) throws IOException
    {
        String zipPluginXmlEntryPath = zipPluginEntryPath + "/plugin.xml";
        try (InputStream in = ZipEntryHelper.zipEntryFileInputStream(zipPath, zipPluginXmlEntryPath))
        {
            Session session = allPluginsNode.getSession();
            String parentAbsPath = allPluginsNode.getPath();
            getLogger().info("XML from '{}!{}' will be imported to '{}' with implementation of merger '{}'", zipPath, zipPluginXmlEntryPath, parentAbsPath, merger);
            merger.jcrImportXml(session, parentAbsPath, in);
            session.save();
        }
        catch (RepositoryException e)
        {
            throw new IOException(e);
        }
    }
    
    /**
     * Import some contents if there is a folder named 'contents'
     * @param pluginName the plugin name.
     * @param allPluginsNode the {@link Node} for all plugins.
     * @param zipPath The input ZIP file
     * @param zipPluginEntryPath The input ZIP entry
     * @param merger The {@link Merger}
     * @return The {@link ImportReport}
     * @throws IOException if an error occurs while reading the archive.
     */
    protected ImportReport importContentsIfAny(String pluginName, Node allPluginsNode, Path zipPath, String zipPluginEntryPath, Merger merger) throws IOException
    {
        String baseImportContentPath = zipPluginEntryPath + "/contents";
        if (ZipEntryHelper.zipEntryFolderExists(zipPath, baseImportContentPath))
        {
            try
            {
                // it must exist as #_importPluginXml must be called before
                Node pluginNode = allPluginsNode.getNode(pluginName);
                
                String pluginNodePath = pluginNode.getPath();
                AmetysObjectCollection rootContents = pluginNode.hasNode(__CONTENT_ROOT_NODE_NAME)
                        ? _ametysObjectCollectionFactory.getObject(pluginNodePath, pluginNode.getNode(__CONTENT_ROOT_NODE_NAME), pluginNodePath)
                        : _ametysObjectCollectionFactory.createChild(pluginNodePath, pluginNode, __CONTENT_ROOT_NODE_NAME, "ametys:collection");
                return _contentsArchiverHelper.importContents(baseImportContentPath + "/", rootContents, zipPath, merger, Collections.EMPTY_SET);
            }
            catch (RepositoryException e)
            {
                throw new IOException(e);
            }
        }
        else
        {
            getLogger().info("No content to be imported for Plugin '{}', the path '{}!{}' does not exist", pluginName, zipPath, baseImportContentPath);
            return new ImportReport();
        }
    }
}
