/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.archive;

import java.io.IOException;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.quartz.SchedulerException;

import org.ametys.core.ui.Callable;
import org.ametys.core.ui.ClientSideElement;
import org.ametys.core.ui.StaticClientSideElement;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.core.schedule.Scheduler;

import com.google.common.base.Predicates;

/**
 * {@link ClientSideElement} for import archive button.
 */
public class ImportArchiveClientSideElement extends StaticClientSideElement
{
    private ArchiveHandler _archiveHandler;
    private Scheduler _scheduler;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _archiveHandler = (ArchiveHandler) smanager.lookup(ArchiveHandler.ROLE);
        _scheduler = (Scheduler) smanager.lookup(Scheduler.ROLE);
    }
    
    /**
     * Gets as JSON the {@link MergePolicy MergePolicies}
     * @return the merge policies
     */
    @Callable(rights = "Runtime_Rights_Admin_Access", context = "/admin")
    public List<Map<String, Object>> getMergePolicies()
    {
        return MergePolicy._ids
                .values()
                .stream()
                .map(MergePolicy::toJson)
                .collect(Collectors.toList());
    }
    
    /**
     * Enumerates the archive files managed by the archive mechanism
     * @return the archive files
     */
    @Callable(rights = "Runtime_Rights_Admin_Access", context = "/admin")
    public List<Map<String, String>> getArchiveFiles()
    {
        return _archiveHandler.getArchiveFiles()
                .map(label -> Map.of("archiveName", label))
                .collect(Collectors.toList());
    }
    
    /**
     * Enumerates the available partial imports of the given archive
     * @param archiveName The archive name
     * @return the available partial imports
     * @throws IOException if an IO exception occured
     */
    @Callable(rights = "Runtime_Rights_Admin_Access", context = "/admin")
    public List<Map<String, Object>> getAvailablePartialImports(String archiveName) throws IOException
    {
        return _archiveHandler.getPartialImports(archiveName)
                .map(PartialImport::toJson)
                .collect(Collectors.toList());
    }
    
    /**
     * Schedules the import
     * @param values The client side values
     * @return A result map
     */
    @Callable(rights = "Runtime_Rights_Admin_Access", context = "/admin")
    public Map<String, Object> scheduleImport(Map<String, Object> values)
    {
        UserIdentity user = _currentUserProvider.getUser();
        String archiveName = Objects.requireNonNull((String) values.get(ImportArchiveSchedulable.ARCHIVE_KEY));
        Optional<Collection<String>> clientSideElements = Optional.of(ImportArchiveSchedulable.ELEMENTS_KEY)
                .map(values::get)
                .filter(Collection.class::isInstance)
                .map(Collection.class::cast);
        Optional<Collection<String>> elements = clientSideElements
                .filter(Predicates.not(Collection::isEmpty));
        MergePolicy mergePolicy = MergePolicy.forId((String) values.get(ImportArchiveSchedulable.MERGE_POLICY_KEY));
        
        try
        {
            _scheduler.scheduleJob(new ImportArchiveRunnable(archiveName, elements, mergePolicy, user));
        }
        catch (SchedulerException e)
        {
            getLogger().error("An exception occured when trying to schedule archive import", e);
            return Map.of("ok", false);
        }
        return Map.of("ok", true);
    }
}

