/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.archive;

import java.io.IOException;
import java.nio.file.Path;
import java.util.zip.ZipOutputStream;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

/**
 * Component responsible to archive plugin data.
 */
public interface PluginArchiver
{
    /**
     * Export plugin data into the destination archive
     * @param pluginName the plugin name.
     * @param pluginNode the plugin {@link Node}.
     * @param zos the output data stream.
     * @param prefix the prefix for ZIP entries.
     * @throws IOException if an error occurs while writing entries to the archive.
     */
    public void export(String pluginName, Node pluginNode, ZipOutputStream zos, String prefix) throws IOException;
    
    /**
     * When in {@link MergePolicy#DELETE_BEFORE}, the plugin needs to delete its Node before importing (if it exists).
     * @param pluginName the plugin name.
     * @param allPluginsNode the {@link Node} for all plugins.
     * @throws RepositoryException if a Repository error occured
     */
    public default void deleteBeforePartialImport(String pluginName, Node allPluginsNode) throws RepositoryException
    {
        if (allPluginsNode.hasNode(pluginName))
        {
            Session session = allPluginsNode.getSession();
            Node pluginNode = allPluginsNode.getNode(pluginName);
            pluginNode.remove();
            session.save();
        }
    }
    
    /**
     * Import plugin data from the source archive
     * @param pluginName the plugin name.
     * @param allPluginsNode the {@link Node} for all plugins.
     * @param zipPath The input ZIP file
     * @param zipPluginEntryPath The input ZIP entry
     * @param merger The {@link Merger}
     * @return The {@link ImportReport}
     * @throws IOException if an error occurs while reading the archive.
     */
    public ImportReport partialImport(String pluginName, Node allPluginsNode, Path zipPath, String zipPluginEntryPath, Merger merger) throws IOException;
}
