/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.archive;

import java.io.IOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.zip.ZipOutputStream;

import javax.jcr.Node;
import javax.jcr.Repository;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;

import org.ametys.core.util.LambdaUtils;
import org.ametys.plugins.repository.provider.AbstractRepository;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.Plugin;
import org.ametys.runtime.plugin.PluginsManager;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Archives resources in /ametys:plugins
 */
public class PluginsArchiver extends AbstractLogEnabled implements Archiver, Serviceable
{
    /** Archiver id. */
    public static final String ID = "plugins";
    
    private static final String __PARTIAL_IMPORT_PREFIX = ID + "/";
    
    private Repository _repository;
    private PluginArchiverExtensionPoint _pluginArchiverExtPoint;
    
    private ManifestReaderWriter _manifestReaderWriter = new PluginsArchiverManifestReaderWriter(getLogger());
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _repository = (Repository) manager.lookup(AbstractRepository.ROLE);
        _pluginArchiverExtPoint = (PluginArchiverExtensionPoint) manager.lookup(PluginArchiverExtensionPoint.ROLE);
    }
    
    @Override
    public ManifestReaderWriter getManifestReaderWriter()
    {
        return _manifestReaderWriter;
    }
    
    private Map<String, String> _getManifestData()
    {
        Session session = null;
        Map<String, Plugin> pluginDefinitions = PluginsManager.getInstance().getAllPlugins();
        Map<String, String> pluginVersions = new HashMap<>();
        
        try
        {
            session = _repository.login();
            
            Iterator<Node> it = _getPluginNodes(session);
            while (it.hasNext())
            {
                Node pluginNode = it.next();
                String pluginName = pluginNode.getName();
                Plugin pluginDef = pluginDefinitions.get(pluginName);
                if (pluginDef != null)
                {
                    String version = pluginDef.getVersion();
                    pluginVersions.put(pluginName, version != null ? version : "");
                }
                else
                {
                    // case of non-plugins eg. 'web-explorer'
                    pluginVersions.put(pluginName, "");
                }
            }
            
            return pluginVersions;
        }
        catch (RepositoryException e)
        {
            throw new IllegalArgumentException("Unable to get plugin list", e);
        }
        finally
        {
            _closeSession(session);
        }
    }
    
    private static Node _getAllPluginsNode(Session session) throws RepositoryException
    {
        return session.getRootNode()
                .getNode("ametys:root/ametys:plugins");
    }
    
    private static Iterator<Node> _getPluginNodes(Session session) throws RepositoryException
    {
        return _getAllPluginsNode(session)
                .getNodes();
    }
    
    private static void _closeSession(Session session)
    {
        if (session != null)
        {
            session.logout();
        }
    }
    
    @Override
    public void export(ZipOutputStream zos) throws IOException
    {
        Session session = null;
        try
        {
            session = _repository.login();
            
            _getPluginNodes(session)
                    .forEachRemaining(LambdaUtils.wrapConsumer(pluginNode -> _exportPlugin(pluginNode, zos)));
        }
        catch (Exception e)
        {
            throw new IllegalArgumentException("Unable to archive plugins", e);
        }
        finally
        {
            _closeSession(session);
        }
    }
    
    private PluginArchiver _retrievePluginArchiver(String pluginName)
    {
        PluginArchiver pluginArchiver = _pluginArchiverExtPoint.getExtension(pluginName);
        if (pluginArchiver == null)
        {
            // there's no specific exporter for this plugin, let's fallback to the default export
            pluginArchiver = _pluginArchiverExtPoint.getExtension(DefaultPluginArchiver.EXTENSION_ID);
            
            if (pluginArchiver == null)
            {
                throw new IllegalStateException("There should be a '__default' extension to PluginArchiverExtensionPoint. Please check your excluded features.");
            }
        }
        
        return pluginArchiver;
    }
    
    private void _exportPlugin(Node pluginNode, ZipOutputStream zos) throws Exception
    {
        String pluginName = pluginNode.getName();
        PluginArchiver pluginArchiver = _retrievePluginArchiver(pluginName);
        getLogger().info("Processing plugin '{}' for archiving at {} with archiver '{}'", pluginName, pluginNode, pluginArchiver.getClass().getName());
        pluginArchiver.export(pluginName, pluginNode, zos, "plugins/" + pluginName);
    }
    
    @Override
    public Collection<String> managedPartialImports(Collection<String> partialImports)
    {
        return partialImports
                .stream()
                .filter(partialImport -> partialImport.startsWith(__PARTIAL_IMPORT_PREFIX))
                .collect(Collectors.toList());
    }
    
    @Override
    public ImportReport partialImport(Path zipPath, Collection<String> partialImports, Merger merger, boolean deleteBefore) throws IOException
    {
        List<ImportReport> reports = new ArrayList<>();
        
        Session session = null;
        try
        {
            session = _repository.login();
            Node pluginsNode = _getAllPluginsNode(session);
            for (String partialImport : partialImports)
            {
                ImportReport pluginImportResult = _importPlugin(partialImport, zipPath, merger, pluginsNode, deleteBefore);
                reports.add(pluginImportResult);
            }
            
            return ImportReport.union(reports);
        }
        catch (RepositoryException e)
        {
            throw new IllegalArgumentException("Unable to import plugins", e);
        }
        finally
        {
            _closeSession(session);
        }
    }
    
    private ImportReport _importPlugin(String partialImport, Path zipPath, Merger merger, Node allPluginsNode, boolean deleteBefore) throws IOException, RepositoryException
    {
        String pluginName = StringUtils.substringAfter(partialImport, __PARTIAL_IMPORT_PREFIX);
        PluginArchiver pluginArchiver = _retrievePluginArchiver(pluginName);
        getLogger().info("Importing plugin '{}' from archive with archiver '{}'", pluginName, pluginArchiver.getClass().getName());
        if (deleteBefore)
        {
            pluginArchiver.deleteBeforePartialImport(pluginName, allPluginsNode);
        }
        
        String zipPluginEntryPath = "plugins/" + pluginName;
        return pluginArchiver.partialImport(pluginName, allPluginsNode, zipPath, zipPluginEntryPath, merger);
    }
    
    private final class PluginsArchiverManifestReaderWriter implements ManifestReaderWriter
    {
        private Logger _logger;
        
        public PluginsArchiverManifestReaderWriter(Logger logger)
        {
            _logger = logger;
        }
        
        @Override
        public Object getData()
        {
            return _getManifestData();
        }
        
        @Override
        public Stream<PartialImport> toPartialImports(Object data)
        {
            return Optional.ofNullable(data)
                    .filter(Map.class::isInstance)
                    .map(this::_castToMap)
                    .map(Map::entrySet)
                    .orElseGet(() ->
                    {
                        _logger.warn("Unexpected manifest data '{}', we would expect a JSON object with key-value mappings representing the plugin names and their version. The ZIP archive probably comes from a different version of Ametys.", data);
                        return Collections.emptySet();
                    })
                    .stream()
                    .map(this::_toPartialImport)
                    .sorted(Comparator.comparing(PartialImport::getKey)); // sort by plugin name (as key is 'plugins/${pluginName}')
        }
        
        private Map<String, String> _castToMap(Object data)
        {
            return Map.class.cast(data);
        }
        
        private PartialImport _toPartialImport(Map.Entry<String, String> entry)
        {
            String pluginName = entry.getKey(); // name of the plugin
            String optionalVersion = entry.getValue(); // can be empty
            String key = __PARTIAL_IMPORT_PREFIX + pluginName;
            return PartialImport.of(key, _toPartialImportLabel(pluginName, optionalVersion));
        }
        
        private I18nizableText _toPartialImportLabel(String pluginName, String optionalVersion)
        {
            // "Plugin 'myplugin' (2.0.0)"
            String optionalVersionLabel = optionalVersion.isEmpty() ? "" : " (" + optionalVersion + ")";
            Map<String, I18nizableTextParameter> parameters = Map.of(
                    "pluginName", new I18nizableText(pluginName),
                    "optionalVersion", new I18nizableText(optionalVersionLabel)
            );
            return new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_ARCHIVE_IMPORT_PLUGIN_ARCHIVER_OPTION_ONE_PLUGIN", parameters);
        }
    }
}
