/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.archive;

import java.io.IOException;
import java.nio.file.Path;
import java.util.Collection;
import java.util.Collections;
import java.util.Optional;
import java.util.stream.Stream;
import java.util.zip.ZipOutputStream;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.slf4j.Logger;

import org.ametys.plugins.explorer.resources.jcr.JCRResourcesCollection;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Archives resources in /ametys:root/ametys:resources
 */
public class ResourcesArchiver extends AbstractLogEnabled implements Archiver, Serviceable
{
    /** Archiver id. */
    public static final String ID = "resources";
    
    private AmetysObjectResolver _resolver;
    private ResourcesArchiverHelper _resourcesArchiverHelper;
    
    private ManifestReaderWriter _manifestReaderWriter = new ResourcesArchiverManifestReaderWriter(getLogger());
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _resourcesArchiverHelper = (ResourcesArchiverHelper) manager.lookup(ResourcesArchiverHelper.ROLE);
    }
    
    @Override
    public ManifestReaderWriter getManifestReaderWriter()
    {
        return _manifestReaderWriter;
    }
    
    @Override
    public void export(ZipOutputStream zos) throws IOException
    {
        Optional<JCRResourcesCollection> rootResources = _rootResources();
        if (rootResources.isPresent())
        {
            _resourcesArchiverHelper.exportCollection(rootResources.get(), zos, "resources/");
        }
    }
    
    private Optional<JCRResourcesCollection> _rootResources()
    {
        try
        {
            return Optional.of(_resolver.resolveByPath("/ametys:resources"));
        }
        catch (UnknownAmetysObjectException e)
        {
            return Optional.empty();
        }
    }
    
    private void _deleteBeforePartialImport(JCRResourcesCollection rootResources) throws IOException
    {
        try
        {
            Node allResourcesNode = rootResources.getNode();
            Session session = allResourcesNode.getParent().getSession();
            allResourcesNode.remove();
            session.save();
        }
        catch (RepositoryException e)
        {
            throw new IOException(e);
        }
    }
    
    @Override
    public Collection<String> managedPartialImports(Collection<String> partialImports)
    {
        return partialImports.contains(ID)
                ? Collections.singletonList(ID)
                : Collections.EMPTY_LIST;
    }
    
    @Override
    public ImportReport partialImport(Path zipPath, Collection<String> partialImports, Merger merger, boolean deleteBefore) throws IOException
    {
        Optional<JCRResourcesCollection> rootResources = _rootResources();
        if (rootResources.isPresent())
        {
            // ALWAYS remove root resources, even if deleteBefore is false because it MUST be re-created with the good UUID
            _deleteBeforePartialImport(rootResources.get());
        }
        
        JCRAmetysObject root = _resolver.resolveByPath("/");
        Node parentOfRootResources = root.getNode();
        return _resourcesArchiverHelper.importCollection("resources/", parentOfRootResources, zipPath, merger);
    }
    
    private static final class ResourcesArchiverManifestReaderWriter implements ManifestReaderWriter
    {
        private static final String __DATA = "resources";
        
        private Logger _logger;
        
        public ResourcesArchiverManifestReaderWriter(Logger logger)
        {
            _logger = logger;
        }
        
        @Override
        public Object getData()
        {
            return __DATA;
        }
        
        @Override
        public Stream<PartialImport> toPartialImports(Object data)
        {
            return Optional.ofNullable(data)
                    .filter(String.class::isInstance)
                    .map(String.class::cast)
                    .filter(__DATA::equals)
                    .map(Stream::of)
                    .orElseGet(() ->
                    {
                        _logger.warn("Unexpected manifest data '{}', we would expect '{}', the ZIP archive probably comes from a different version of Ametys.", data, __DATA);
                        return Stream.empty();
                    })
                    .map(resourcesStr -> PartialImport.of(resourcesStr, new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_ARCHIVE_IMPORT_RESOURCE_ARCHIVER_OPTION_ALL")));
        }
    }
}
