/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.csv;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.tika.detect.DefaultEncodingDetector;
import org.apache.tika.metadata.Metadata;
import org.supercsv.io.CsvListReader;
import org.supercsv.io.CsvMapReader;
import org.supercsv.io.ICsvListReader;
import org.supercsv.io.ICsvMapReader;
import org.supercsv.prefs.CsvPreference;

import org.ametys.cms.contenttype.ContentAttributeDefinition;
import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.language.UserLanguagesManager;
import org.ametys.core.util.mail.SendMailHelper;
import org.ametys.core.util.mail.SendMailHelper.MailBuilder;
import org.ametys.plugins.contentio.csv.SynchronizeModeEnumerator.ImportMode;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.model.RepeaterDefinition;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.ModelHelper;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ModelItemGroup;
import org.ametys.runtime.model.ModelViewItemGroup;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.ViewElement;
import org.ametys.runtime.model.ViewElementAccessor;
import org.ametys.runtime.model.ViewItem;
import org.ametys.runtime.model.exception.BadItemTypeException;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.runtime.servlet.RuntimeConfig;

import jakarta.mail.MessagingException;

/**
 * Import contents from an uploaded CSV file.
 */
public class ImportCSVFileHelper extends AbstractLogEnabled implements Component, Serviceable, Initializable
{
    /** Avalon Role */
    public static final String ROLE = ImportCSVFileHelper.class.getName();

    /** Key in config containing the parent id */
    public static final String CONFIG_PARENT_ID = "parentId";
    /** Result key containing updated content's ids */
    public static final String RESULT_IMPORTED_COUNT = "importedCount";
    /** Result key containing number of errors */
    public static final String RESULT_NB_ERRORS = "nbErrors";
    /** Result key containing number of warnings */
    public static final String RESULT_NB_WARNINGS = "nbWarnings";
    /** Result key containing a short an explanation for a general error */
    public static final String RESULT_ERROR = "error";
    /** Result key containing the filename */
    public static final String RESULT_FILENAME = "fileName";
    /** Column name of attribute path in CSV mapping */
    public static final String MAPPING_COLUMN_ATTRIBUTE_PATH = "attributePath";
    /** Column name of header in CSV mapping */
    public static final String MAPPING_COLUMN_HEADER = "header";
    /** Column name to identify if column is an ID in CSV mapping */
    public static final String MAPPING_COLUMN_IS_ID = "isId";
    /** Key in nested mapping to define the ID columns */
    public static final String NESTED_MAPPING_ID = "id";
    /** Key in nested mapping to define the values columns */
    public static final String NESTED_MAPPING_VALUES = "values";
    
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The CSV Importer */
    protected CSVImporter _csvImporter;
    /** The content type extension point */
    protected ContentTypeExtensionPoint _contentTypeEP;
    
    private CurrentUserProvider _currentUserProvider;
    private UserManager _userManager;
    private UserLanguagesManager _userLanguagesManager;
    
    private ContentTypesHelper _contentTypesHelper;
    private I18nUtils _i18nUtils;
    private String _sysadminMail;

    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _currentUserProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
        _userManager = (UserManager) serviceManager.lookup(UserManager.ROLE);
        _userLanguagesManager = (UserLanguagesManager) serviceManager.lookup(UserLanguagesManager.ROLE);
        
        _contentTypeEP = (ContentTypeExtensionPoint) serviceManager.lookup(ContentTypeExtensionPoint.ROLE);
        _contentTypesHelper = (ContentTypesHelper) serviceManager.lookup(ContentTypesHelper.ROLE);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        
        _i18nUtils = (I18nUtils) serviceManager.lookup(I18nUtils.ROLE);
        
        _csvImporter = (CSVImporter) serviceManager.lookup(CSVImporter.ROLE);
    }
    
    public void initialize() throws Exception
    {
        _sysadminMail = Config.getInstance().getValue("smtp.mail.sysadminto");
    }

    /**
     * Retrieves the values for import CSV parameters
     * @param contentTypeId the configured content type identifier
     * @return the values for import CSV parameters
     * @throws Exception if an error occurs.
     */
    @Callable(rights = Callable.NO_CHECK_REQUIRED) // only retrieve content type definition
    public Map<String, Object> getImportCSVParametersValues(String contentTypeId) throws Exception
    {
        Map<String, Object> values = new HashMap<>();
        
        // Content types
        List<String> contentTypeIds = StringUtils.isEmpty(contentTypeId) ? List.of() : List.of(contentTypeId);
        Map<String, Object> filteredContentTypes = _contentTypesHelper.getContentTypesList(contentTypeIds, true, true, true, false, false);
        values.put("availableContentTypes", filteredContentTypes.get("contentTypes"));
        
        // Recipient - get current user email
        String currentUserEmail = null;
        UserIdentity currentUser = _currentUserProvider.getUser();
        
        String login = currentUser.getLogin();
        if (StringUtils.isNotBlank(login))
        {
            String userPopulationId = currentUser.getPopulationId();
            User user = _userManager.getUser(userPopulationId, login);
            currentUserEmail = user.getEmail();
        }
        values.put("defaultRecipient", currentUserEmail);
        
        return values;
    }
    
    /**
     * Gets the configuration for creating/editing a collection of synchronizable contents.
     * @param contentTypeId Content type id
     * @param formValues map of form values
     * @param mappingValues list of header and content attribute mapping
     * @return A map containing information about what is needed to create/edit a collection of synchronizable contents
     * @throws IOException IOException while reading CSV
     */
    @SuppressWarnings("unchecked")
    @Callable(rights = Callable.NO_CHECK_REQUIRED) // Only checks against content type definition
    public Map<String, Object> validateConfiguration(String contentTypeId, Map formValues, List<Map<String, Object>> mappingValues) throws IOException
    {
        Map<String, Object> result = new HashMap<>();

        List<Map<String, Object>> filteredMappingValues = _filterMapping(mappingValues);
        
        List<String> contentAttributes = filteredMappingValues.stream()
                .map(column -> column.get(MAPPING_COLUMN_ATTRIBUTE_PATH))
                .map(String.class::cast)
                .collect(Collectors.toList());

        Map<String, Object> mapping = new HashMap<>();
        filteredMappingValues.forEach(column -> generateNestedMapping(mapping, column));
        
        ContentType contentType = _contentTypeEP.getExtension(contentTypeId);
        result.put("contentTypeName", contentType.getLabel());
        View view = null;
        for (String itemPath : contentAttributes)
        {
            if (!ModelHelper.hasModelItem(itemPath, List.of(contentType)))
            {
                result.put("error", "bad-mapping");
                result.put("details", List.of(itemPath, contentTypeId));
                return result;
            }
        }
        
        try
        {
            view = View.of(contentType, contentAttributes.toArray(new String[contentAttributes.size()]));
        }
        catch (IllegalArgumentException | BadItemTypeException e)
        {
            getLogger().error("Error while creating view", e);
            result.put("error", "cant-create-view");
            result.put("details", contentTypeId);
            return result;
        }
        
        if (!mapping.containsKey(NESTED_MAPPING_ID))
        {
            result.put("error", "missing-id");
            return result;
        }
        
        for (ViewItem viewItem : view.getViewItems())
        {
            if (!_checkViewItemContainer(viewItem, (Map<String, Object>) mapping.get(NESTED_MAPPING_VALUES), result))
            {
                return result;
            }
        }
        result.put("success", true);
        return result;
    }
    /**
     * Gets the configuration for creating/editing a collection of synchronizable contents.
     * @param config get all CSV related parameters: path, separating/escaping char, charset and contentType
     * @param formValues map of form values
     * @param mappingValues list of header and content attribute mapping
     * @return A map containing information about what is needed to create/edit a collection of synchronizable contents
     * @throws IOException IOException while reading CSV
     */
    @Callable(rights = "Plugins_ContentIO_ImportFile", context = "/cms")
    public Map<String, Object> importContents(Map<String, Object> config, Map<String, Object> formValues, List<Map<String, Object>> mappingValues) throws IOException
    {
        Map<String, Object> result = new HashMap<>();
        result.put(RESULT_FILENAME, config.get("fileName"));
        
        String importId = (String) config.get("importId");
        File importFile = _getTempFile(importId);
        String contentTypeId = (String) config.get("contentType");
        ContentType contentType = _contentTypeEP.getExtension(contentTypeId);
        
        // CSV configuration
        CsvPreference csvPreference = new CsvPreference.Builder(
                config.getOrDefault("escaping-char", '"').toString().charAt(0),
                config.getOrDefault("separating-char", ';').toString().charAt(0),
                "\r\n"
            )
            .build();

        String encoding = (String) config.get("charset");
        Charset charset = Charset.forName(encoding);
        
        String language = (String) formValues.get("language");
        int createAction = Integer.valueOf((String) formValues.getOrDefault("createAction", "1"));
        int editAction = Integer.valueOf((String) formValues.getOrDefault("editAction", "2"));
        String workflowName = (String) formValues.get("workflow");

        ImportMode importMode = Optional.ofNullable((String) formValues.get("importMode")).filter(StringUtils::isNotEmpty).map(ImportMode::valueOf).orElse(ImportMode.CREATE_AND_UPDATE);
        Optional<String> recipient = Optional.ofNullable(config.get("recipient"))
                .filter(String.class::isInstance)
                .map(String.class::cast)
                .filter(StringUtils::isNotBlank);
        
        Optional< ? extends Content> parentContent = Optional.empty();
        String parentId = (String) config.get(CONFIG_PARENT_ID);
        if (parentId != null)
        {
            Content parent = _resolver.resolveById(parentId);
            if (parent != null)
            {
                parentContent = Optional.of(parent);
            }
        }
        
        @SuppressWarnings("unchecked")
        Map<String, Object> additionalTransientVars = (Map<String, Object>) config.getOrDefault("additionalTransientVars", new HashMap<>());
        
        List<Map<String, Object>> filteredMappingValues = _filterMapping(mappingValues);
        
        try (
            InputStream fileInputStream = new FileInputStream(importFile);
            InputStream inputStream = IOUtils.buffer(fileInputStream);
        )
        {
            result.putAll(_importContents(inputStream, csvPreference, charset, contentType, workflowName, language, createAction, editAction, filteredMappingValues, importMode, parentContent, additionalTransientVars));
            
            // If an exception occured and the recipient is empty, transfer the error mail to the configured sysadmin
            if ("error".equals(result.get(RESULT_ERROR)) && recipient.isEmpty())
            {
                recipient = Optional.ofNullable(_sysadminMail)
                        .filter(StringUtils::isNotBlank);
            }
            
            if (recipient.isPresent())
            {
                _sendMail(result, recipient.get());
            }
        }
        finally
        {
            FileUtils.deleteQuietly(importFile);
        }

        return result;
    }
    
    /**
     * Import contents from path, content type and language.
     * CSV Excel north Europe preferences are used, charset is detected on the file, workflow is detected on content type, default creation (1) and
     * edition (2) actions are used, values are automatically mapped from the header. Headers should contains a star (*) to detect identifier columns.
     * @param inputStream The (buffered) input stream to the CSV resource
     * @param contentType The content type
     * @param language The language
     * @param importMode The import mode
     * @return the result of the CSV import
     * @throws IOException if an error occurs
     */
    public Map<String, Object> importContents(InputStream inputStream, ContentType contentType, String language, ImportMode importMode) throws IOException
    {
        CsvPreference csvPreference = CsvPreference.EXCEL_NORTH_EUROPE_PREFERENCE;
        
        // Detect the charset of the current file
        Charset charset = detectCharset(inputStream);
        
        // Copy the first 8192 bytes of the initial input stream to another input stream (to get headers)
        inputStream.mark(8192);
        try (InputStream headerStream = new ByteArrayInputStream(inputStream.readNBytes(8192)))
        {
            inputStream.reset();
            
            // Get headers
            String[] headers = extractHeaders(headerStream, csvPreference, charset);
            
            // Get the mapping of the values
            List<Map<String, Object>> mappingValues = getMapping(contentType, headers);
            
            // Get the default workflow associated to the content type
            String workflowName = contentType.getDefaultWorkflowName().orElseThrow(() -> new IllegalArgumentException("The workflow can't be defined."));
            
            // Import contents (last use of the input stream)
            return _importContents(inputStream, csvPreference, charset, contentType, workflowName, language, 1, 2, mappingValues, importMode);
        }
    }
    
    private Map<String, Object> _importContents(InputStream inputStream, CsvPreference csvPreference, Charset charset, ContentType contentType, String workflowName, String language, int createAction, int editAction, List<Map<String, Object>> mappingValues, ImportMode importMode)
    {
        return _importContents(inputStream, csvPreference, charset, contentType, workflowName, language, createAction, editAction, mappingValues, importMode, Optional.empty(), Map.of());
    }
    
    private Map<String, Object> _importContents(InputStream inputStream, CsvPreference csvPreference, Charset charset, ContentType contentType, String workflowName, String language, int createAction, int editAction, List<Map<String, Object>> mappingValues, ImportMode importMode, Optional< ? extends Content> parentContent, Map<String, Object> additionalTransientVars)
    {
        Map<String, Object> result = new HashMap<>();
        
        List<String> contentAttributes = mappingValues.stream()
            .map(column -> column.get(MAPPING_COLUMN_ATTRIBUTE_PATH))
            .map(String.class::cast)
            .collect(Collectors.toList());

        Map<String, Object> mapping = new HashMap<>();
        mappingValues.forEach(column -> generateNestedMapping(mapping, column));
        
        View view = View.of(contentType, contentAttributes.toArray(new String[contentAttributes.size()]));

        try (
            InputStreamReader inputStreamReader = new InputStreamReader(inputStream, charset);
            BufferedReader reader = new BufferedReader(inputStreamReader);
            ICsvListReader csvMapReadernew = new CsvListReader(reader, csvPreference)
        )
        {
            Map<String, Object> importResult = _csvImporter.importContentsFromCSV(mapping, view, contentType, csvMapReadernew, createAction, editAction, workflowName, language, importMode, parentContent, additionalTransientVars);

            @SuppressWarnings("unchecked")
            List<String> contentIds = (List<String>) importResult.get(CSVImporter.RESULT_CONTENT_IDS);
            
            // fill RESULT_ERROR only if the content list is empty and if there are errors, as an empty file or no import in CREATE_ONLY and UPDATE_ONLY mode should not be considered as errors
            if (contentIds.size() == 0 && (int) importResult.get(CSVImporter.RESULT_NB_ERRORS) > 0)
            {
                result.put(RESULT_ERROR, "no-import");
            }
            else
            {
                result.put(RESULT_IMPORTED_COUNT, contentIds.size());
                result.put(RESULT_NB_ERRORS, importResult.get(CSVImporter.RESULT_NB_ERRORS));
                result.put(RESULT_NB_WARNINGS, importResult.get(CSVImporter.RESULT_NB_WARNINGS));
            }
        }
        catch (Exception e)
        {
            getLogger().error("Error while importing contents", e);
            result.put(RESULT_ERROR, "error");
        }
        
        return result;
    }
    
    private boolean _checkViewItemContainer(ViewItem viewItem, Map<String, Object> mapping, Map<String, Object> result)
    {
        if (viewItem instanceof ViewElement)
        {
            ViewElement viewElement = (ViewElement) viewItem;
            ElementDefinition elementDefinition = viewElement.getDefinition();
            if (elementDefinition instanceof ContentAttributeDefinition)
            {
                ViewElementAccessor viewElementAccessor = (ViewElementAccessor) viewElement;
                ContentAttributeDefinition contentAttributeDefinition = (ContentAttributeDefinition) elementDefinition;
                return _checkViewItemContentContainer(viewElementAccessor, mapping, result, contentAttributeDefinition);
            }
            else
            {
                return true;
            }
        }
        else if (viewItem instanceof ModelViewItemGroup)
        {
            return _checkViewItemGroupContainer(viewItem, mapping, result);
        }
        else
        {
            result.put("error", "unknown-type");
            result.put("details", viewItem.getName());
            return false;
        }
    }

    @SuppressWarnings("unchecked")
    private boolean _checkViewItemContentContainer(ViewElementAccessor viewElementAccessor, Map<String, Object> mapping, Map<String, Object> result, ContentAttributeDefinition contentAttributeDefinition)
    {

        boolean success = true;
        String contentTypeId = contentAttributeDefinition.getContentTypeId();
        ContentType contentType = _contentTypeEP.getExtension(contentTypeId);
        if (mapping.get(viewElementAccessor.getName()) instanceof String)
        {
            // Path matches a content instead of a contents attribute
            result.put("error", "string-as-container");
            result.put("details", mapping.get(viewElementAccessor.getName()));
            return false;
        }
        Map<String, Object> nestedMap = (Map<String, Object>) mapping.get(viewElementAccessor.getName());
        if (!nestedMap.containsKey(NESTED_MAPPING_ID) || ((List<String>) nestedMap.get(NESTED_MAPPING_ID)).isEmpty())
        {
            // No identifier for the content
            result.put("error", "missing-id-content");
            result.put("details", List.of(contentType.getLabel(), contentAttributeDefinition.getName()));
            return false;
        }
        if (viewElementAccessor.getDefinition().isMultiple())
        {
            Optional<RepeaterDefinition> repeater = _getRepeaterAncestorIfExists(viewElementAccessor.getDefinition());
            if (repeater.isPresent())
            {
                // Content is multiple inside a repeater, we do not handle this as we can't discriminate content id from repeater id
                result.put("error", "multiple-content-inside-repeater");
                result.put("details", List.of(contentType.getLabel(), repeater.get().getLabel()));
                return false;
            }
        }
        Map<String, Object> nestedMapValues = (Map<String, Object>) (nestedMap.get(NESTED_MAPPING_VALUES));
        for (ViewItem nestedViewItem : viewElementAccessor.getViewItems())
        {
            success = success && _checkViewItemContainer(nestedViewItem, nestedMapValues, result);
        }
        return success;
    }
    
    /**
     * Get the repeater ancestor of the given model item if exists, searching parent's parent recursively if needed.
     * @param modelItem the model item
     * @return the repeater ancestor if exists
     */
    private Optional<RepeaterDefinition> _getRepeaterAncestorIfExists(ModelItem modelItem)
    {
        ModelItemGroup parent = modelItem.getParent();
        // There is no parent anymore: we stop there
        if (parent == null)
        {
            return Optional.empty();
        }
        // Parent is a repeater: we stop there and return the repeater
        if (parent instanceof RepeaterDefinition repeaterParent)
        {
            return Optional.of(repeaterParent);
        }
        // The parent is not a repeater, but his parent may be one. Check if the parent have a repeater as parent.
        else
        {
            return _getRepeaterAncestorIfExists(parent);
        }
    }
    
    @SuppressWarnings("unchecked")
    private boolean _checkViewItemGroupContainer(ViewItem viewItem, Map<String, Object> mapping, Map<String, Object> result)
    {
        boolean success = true;
        ModelViewItemGroup modelViewItemGroup = (ModelViewItemGroup) viewItem;
        List<ViewItem> elementDefinition = modelViewItemGroup.getViewItems();
        Map<String, Object> nestedMap = (Map<String, Object>) mapping.get(viewItem.getName());
        
        Map<String, Object> nestedMapValues = (Map<String, Object>) (nestedMap.get(NESTED_MAPPING_VALUES));
                    
        for (ViewItem nestedViewItem : elementDefinition)
        {
            success = success && _checkViewItemContainer(nestedViewItem, nestedMapValues, result);
        }
        return success;
    }

    private static Map<String, Object> generateNestedMapping(Map<String, Object> mapping, Map<String, Object> column)
    {
        return generateNestedMapping(mapping, (String) column.get(MAPPING_COLUMN_ATTRIBUTE_PATH), (String) column.get(ImportCSVFileHelper.MAPPING_COLUMN_HEADER), (boolean) column.get(ImportCSVFileHelper.MAPPING_COLUMN_IS_ID));
    }

    @SuppressWarnings("unchecked")
    private static Map<String, Object> generateNestedMapping(Map<String, Object> mapping, String attributePath, String column, boolean isId)
    {
        Map<String, Object> values = (Map<String, Object>) mapping.computeIfAbsent(NESTED_MAPPING_VALUES, __ -> new HashMap<>());

        int separatorIndex = attributePath.indexOf('/');
        if (separatorIndex == -1)
        {
            if (isId)
            {
                List<String> ids = (List<String>) mapping.computeIfAbsent(NESTED_MAPPING_ID, __ -> new ArrayList<>());
                ids.add(attributePath);
            }
            values.put(attributePath, column);
        }
        else
        {
            Map<String, Object> subValuesMapping = (Map<String, Object>) values.computeIfAbsent(attributePath.substring(0, separatorIndex), __ -> new HashMap<>());
            generateNestedMapping(subValuesMapping, attributePath.substring(separatorIndex + 1), column, isId);
        }
        
        mapping.put(NESTED_MAPPING_VALUES, values);
        return mapping;
    }

    private void _sendMail(Map<String, Object> importResult, String recipient)
    {
        I18nizableText subject = _getMailSubject(importResult);
        I18nizableText body = _getMailBody(importResult);
        
        try
        {
            String language = _userLanguagesManager.getDefaultLanguage();
            MailBuilder mailBuilder = SendMailHelper.newMail()
                                                    .withSubject(_i18nUtils.translate(subject, language))
                                                    .withRecipient(recipient)
                                                    .withTextBody(_i18nUtils.translate(body, language));
            
            mailBuilder.sendMail();
        }
        catch (MessagingException | IOException e)
        {
            if (getLogger().isWarnEnabled())
            {
                getLogger().warn("Unable to send the e-mail '" + subject  + "' to '" + recipient + "'", e);
            }
        }
        catch (Exception e)
        {
            getLogger().error("An unknown error has occured while sending the mail.", e);
        }
    }
    
    private I18nizableText _getMailSubject(Map<String, Object> importResult)
    {
        Map<String, I18nizableTextParameter> i18nParams = Map.of("fileName", new I18nizableText((String) importResult.get(RESULT_FILENAME)));
        if (importResult.containsKey(ImportCSVFileHelper.RESULT_ERROR))
        {
            return new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_IMPORT_MAIL_ERROR_SUBJECT", i18nParams);
        }
        else
        {
            return new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_IMPORT_MAIL_SUCCESS_SUBJECT", i18nParams);
        }
    }
    
    private I18nizableText _getMailBody(Map<String, Object> importResult)
    {
        Map<String, I18nizableTextParameter> i18nParams = new HashMap<>();
        i18nParams.put("fileName", new I18nizableText((String) importResult.get(RESULT_FILENAME)));
        
        if (importResult.containsKey(ImportCSVFileHelper.RESULT_ERROR))
        {
            if ("no-import".equals(importResult.get(ImportCSVFileHelper.RESULT_ERROR)))
            {
                return new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_IMPORT_MAIL_ERROR_BODY_NO_IMPORT", i18nParams);
            }
            else
            {
                return new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_IMPORT_MAIL_ERROR_BODY_TEXT", i18nParams);
            }
        }
        else
        {
            int nbErrors = (int) importResult.get(RESULT_NB_ERRORS);
            int nbWarnings = (int) importResult.get(RESULT_NB_WARNINGS);
            i18nParams.put("importedCount", new I18nizableText(String.valueOf(importResult.get(RESULT_IMPORTED_COUNT))));
            i18nParams.put("nbErrors", new I18nizableText(String.valueOf(nbErrors)));
            i18nParams.put("nbWarnings", new I18nizableText(String.valueOf(nbWarnings)));
            
            if (nbErrors == 0 && nbWarnings == 0)
            {
                return new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_IMPORT_MAIL_SUCCESS_BODY", i18nParams);
            }
            else if (nbWarnings == 0)
            {
                return new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_IMPORT_MAIL_PARTIAL_SUCCESS_BODY_ERRORS", i18nParams);
            }
            else if (nbErrors == 0)
            {
                return new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_IMPORT_MAIL_PARTIAL_SUCCESS_BODY_WARNINGS", i18nParams);
            }
            else
            {
                return new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_CONTENT_IMPORT_MAIL_PARTIAL_SUCCESS_BODY_ERRORS_AND_WARNINGS", i18nParams);
            }
        }
    }
    
    /**
     * Indicate that an import is cancel during configuration
     * @param importId the import id
     */
    @Callable(rights = "Plugins_ContentIO_ImportFile", context = "/cms")
    public void cancelImport(String importId)
    {
        File importFile = _getTempFile(importId);
        FileUtils.deleteQuietly(importFile);
    }

    private File _getTempFile(String importId)
    {
        File importFile = FileUtils.getFile(RuntimeConfig.getInstance().getAmetysHome(), ImportCSVFile.CONTENTIO_STORAGE_DIRECTORY, importId + ".csv");
        return importFile;
    }
    
    /**
     * Detect the charset of a given resource.
     * @param inputStream The input stream to the resource, need to support {@link InputStream#mark(int)} and {@link InputStream#reset()}.
     * @return the charset
     * @throws IOException if an error occurs
     */
    public Charset detectCharset(InputStream inputStream) throws IOException
    {
        assert inputStream.markSupported();
        DefaultEncodingDetector defaultEncodingDetector = new DefaultEncodingDetector();
        return defaultEncodingDetector.detect(inputStream, new Metadata());
    }

    /**
     * Extract headers from the CSV resource.
     * @param inputStream The input stream to the resource
     * @param csvPreference The CSV preference (separators)
     * @param charset The charset of the resource
     * @return the list of headers in file
     * @throws IOException if an error occurs
     */
    public String[] extractHeaders(InputStream inputStream, CsvPreference csvPreference, Charset charset) throws IOException
    {
        try (
            InputStreamReader inputStreamReader = new InputStreamReader(inputStream, charset);
            BufferedReader reader = new BufferedReader(inputStreamReader);
            ICsvMapReader mapReader = new CsvMapReader(reader, csvPreference);
        )
        {
            String[] headers = mapReader.getHeader(true);
            return headers;
        }
    }

    /**
     * Get the mapping from the headers and corresponding to the content type.
     * @param contentType The content type
     * @param headers The CSV headers
     * @return the list of columns descriptions. An item of the list is one column, each column is defined by a header "header" (text defined into the CSV header), an
     * attribute path "attributePath" (real path extracted from the header, empty if the model item does not exist) and an identifier flag "isId" (<code>true</code>
     * if the header ends with a star "*").
     */
    public List<Map<String, Object>> getMapping(ContentType contentType, String[] headers)
    {
        return Arrays.asList(headers)
            .stream()
            .filter(StringUtils::isNotEmpty)
            .map(header -> _transformHeaderToMap(contentType, header))
            .collect(Collectors.toList());
    }
    
    private Map<String, Object> _transformHeaderToMap(ContentType contentType, String header)
    {
        Map<String, Object> headerMap = new HashMap<>();
        headerMap.put(ImportCSVFileHelper.MAPPING_COLUMN_HEADER, header);
        headerMap.put(
            MAPPING_COLUMN_ATTRIBUTE_PATH,
            Optional.of(header)
                .map(s -> s.replace("*", ""))
                .map(s -> s.replace(".", "/"))
                .map(String::trim)
                .filter(contentType::hasModelItem)
                .orElse(StringUtils.EMPTY)
        );
        headerMap.put(ImportCSVFileHelper.MAPPING_COLUMN_IS_ID, header.endsWith("*"));
        return headerMap;
    }
    
    private List<Map<String, Object>> _filterMapping(List<Map<String, Object>> mappingValues)
    {
        // Suppress empty attributePath
        // Transform attributePath with . to attributePath with / (Autocompletion in IHM has . as separator, but the API need /)
        return mappingValues.stream()
            .map(column ->
                {
                    Optional<String> attributePath = Optional.of(MAPPING_COLUMN_ATTRIBUTE_PATH)
                        .map(column::get)
                        .map(String.class::cast)
                        .filter(StringUtils::isNotEmpty)
                        .map(s -> s.replace(".", "/"));
                    
                    if (attributePath.isEmpty())
                    {
                        return null;
                    }
                    
                    // Replace original column
                    Map<String, Object> columnCopy = new HashMap<>(column);
                    columnCopy.put(MAPPING_COLUMN_ATTRIBUTE_PATH, attributePath.get());
                    return columnCopy;
                }
            )
            .filter(Predicate.not(Objects::isNull))
            .collect(Collectors.toList());
    }
}
