/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.synchronize;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;

import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.repository.LanguageExpression;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.core.schedule.progression.ProgressionTrackerFactory;
import org.ametys.core.schedule.progression.SimpleProgressionTracker;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.util.HttpUtils;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.language.UserLanguagesManager;
import org.ametys.core.util.mail.SendMailHelper;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.OrExpression;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.workflow.AbstractWorkflowComponent;
import org.ametys.plugins.workflow.AbstractWorkflowComponent.ConditionFailure;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;

import com.opensymphony.workflow.InvalidActionException;
import com.opensymphony.workflow.WorkflowException;

import jakarta.mail.MessagingException;

/**
 * Abstract implementation of {@link SynchronizableContentsCollection}.
 */
public abstract class AbstractSynchronizableContentsCollection extends AbstractStaticSynchronizableContentsCollection
{
    /** SCC unique ID field */
    protected static final String SCC_UNIQUE_ID = "scc$uniqueid";
    
    /** The i18n utils */
    protected I18nUtils _i18nUtils;
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    /** The observation manager */
    protected ObservationManager _observationManager;
    /** The content workflow helper */
    protected ContentWorkflowHelper _contentWorkflowHelper;
    /** The user languages manager */
    protected UserLanguagesManager _userLanguagesManager;
    
    /** Number of errors encountered */
    protected int _nbError;
    /** True if there is a global error during synchronization */
    protected boolean _hasGlobalError;
    
    /** Number of created contents */
    protected int _nbCreatedContents;
    /** Number of synchronized contents */
    protected int _nbSynchronizedContents;
    /** Number of unchanged contents */
    protected int _nbNotChangedContents;
    /** Number of deleted contents */
    protected int _nbDeletedContents;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
        _contentWorkflowHelper = (ContentWorkflowHelper) manager.lookup(ContentWorkflowHelper.ROLE);
        _userLanguagesManager = (UserLanguagesManager) manager.lookup(UserLanguagesManager.ROLE);
    }
    
    @Override
    public List<ModifiableContent> populate(Logger logger, ContainerProgressionTracker progressionTracker)
    {
        ContainerProgressionTracker internalPopulaitePT =  progressionTracker.addContainerStep("internalpopulate", new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_SCHEDULER_SYNCHRONIZE_COLLECTION_IMPORT_SYNCHRONIZE_CONTENTS_STEP_LABEL"), 2);
        
        SimpleProgressionTracker deleteUnexistingContentsPT = progressionTracker.addSimpleStep("deleteunexistingcontents", new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_SCHEDULER_SYNCHRONIZE_COLLECTION_DELETE_UNEXISTING_CONTENTS_STEP_LABEL"));
        
        SimpleProgressionTracker notifyPT = progressionTracker.addSimpleStep("notify", new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_SCHEDULER_SYNCHRONIZE_COLLECTION_NOTIFY_OBSERVERS_AND_SEND_MAILS_STEP_LABEL"));
        
        _nbCreatedContents = 0;
        _nbSynchronizedContents = 0;
        _nbNotChangedContents = 0;
        _nbDeletedContents = 0;
        _nbError = 0;
        _hasGlobalError = false;
        
        logger.info("Start synchronization of collection '{}'", getId());
        List<Long> times = new ArrayList<>();
        times.add(System.currentTimeMillis());
        
        // Do populate
        List<ModifiableContent> populatedContents = _internalPopulate(logger, internalPopulaitePT);
        
        if (!_hasGlobalError && removalSync())
        {
            // Delete old contents if source prevails
            deleteUnexistingContents(logger);
        }
        deleteUnexistingContentsPT.increment();
        
        times.add(System.currentTimeMillis());
        logger.info("[Synchronization of collection '{}'] Populated in {} ms", getId(), times.get(times.size() - 1) - times.get(times.size() - 2));
        _logSynchronizationResult(logger);
        
        notifyPT.setSize(2);
        
        if (_hasSomethingChanged())
        {
            // Do not notify obeservers if there is no change
            Map<String, Object> eventParams = new HashMap<>();
            eventParams.put(org.ametys.plugins.contentio.synchronize.observation.ObservationConstants.ARGS_COLLECTION_ID, this.getId());
            eventParams.put(org.ametys.plugins.contentio.synchronize.observation.ObservationConstants.ARGS_COLLECTION_CONTENT_TYPE, this.getContentType());
            _observationManager.notify(new Event(org.ametys.plugins.contentio.synchronize.observation.ObservationConstants.EVENT_CONTENT_COLLECTION_SYNCHRONIZED, _currentUserProvider.getUser(), eventParams));
            
            times.add(System.currentTimeMillis());
            logger.info("[Synchronization of collection '{}'] Listeners notified in {} ms", getId(), times.get(times.size() - 1) - times.get(times.size() - 2));
            
        }
        notifyPT.increment();
        
        if (_nbError > 0 && getReportMails().length() > 0)
        {
            try
            {
                logger.warn("{} contents were not created/updated because of an error.", _nbError);
                sendErrorMail(_nbError);
                
                times.add(System.currentTimeMillis());
                logger.info("[Synchronization of collection '{}'] Error mail sent in {} ms", getId(), times.get(times.size() - 1) - times.get(times.size() - 2));
            }
            catch (MessagingException | IOException e)
            {
                logger.warn("Unable to send mail", e);
            }
        }
        notifyPT.increment();
        
        logger.info("[Synchronization of collection '{}'] Total in {} ms", getId(), times.get(times.size() - 1) - times.get(0));

        return populatedContents;
    }
    
    /**
     * Internal implementation of {@link #populate(Logger, ContainerProgressionTracker)}
     * @param logger The logger
     * @return The list of created/synchronized contents
     */
    protected List<ModifiableContent> _internalPopulate(Logger logger)
    {
        return _internalPopulate(logger, ProgressionTrackerFactory.createContainerProgressionTracker("Internal populate", logger));
    }
    
    /**
     * Internal implementation of {@link #populate(Logger, ContainerProgressionTracker)}
     * @param logger The logger
     * @param progressionTracker The progression tracker
     * @return The list of created/synchronized contents
     */
    protected abstract List<ModifiableContent> _internalPopulate(Logger logger, ContainerProgressionTracker progressionTracker);
    
    @Override
    public void empty(Logger logger)
    {
        // Get all contents from the SCC
        Expression collectionExpression = _sccHelper.getCollectionExpression(getId());
        String xPathQuery = ContentQueryHelper.getContentXPathQuery(collectionExpression);
        List<ModifiableContent> contentsToRemove = _resolver.<ModifiableContent>query(xPathQuery)
            .stream()
            .toList();
        // Process to the deletion
        _removeSCCOrDeleteContents(contentsToRemove, logger);
    }
    
    /**
     * Delete contents created by a previous synchronization which does not exist anymore in remote source.
     * If the content belongs to several SCC, only the current SCC reference will be deleted.
     * @param logger The logger
     */
    protected void deleteUnexistingContents(Logger logger)
    {
        String query = _getContentPathQuery(null, null, null, true);
        AmetysObjectIterable<ModifiableContent> contents = _resolver.query(query);
        
        List<ModifiableContent> contentsToRemove = _getContentsToRemove(contents);

        if (!contentsToRemove.isEmpty())
        {
            if (logger.isInfoEnabled())
            {
                contentsToRemove.stream().forEach(content -> logger.info("The content '{}' ({}) does not exist anymore in remote source: it will be deleted if possible.", content.getTitle(), content.getId()));
            }
            
            _nbDeletedContents = _removeSCCOrDeleteContents(contentsToRemove, logger);
        }
    }
    
    /**
     * Filter the contents to remove.
     * @param contents The list of all the available contents
     * @return The {@link List} of {@link Content} to remove.
     */
    protected abstract List<ModifiableContent> _getContentsToRemove(AmetysObjectIterable<ModifiableContent> contents);
    
    /**
     * For each content, if the content has only this SCC, try to delete it, but if it has several SCC, remove the current collection from the SCC property.
     * @param contents The contents to delete or to remove the SCC
     * @param logger The logger
     * @return the number of really deleted contents
     */
    protected int _removeSCCOrDeleteContents(List<ModifiableContent> contents, Logger logger)
    {
        logger.info("Remove SCC on contents with multiple SCC... {} contents for all the deletion process.", contents.size());
        List<Content> contentsToReallyDelete = new ArrayList<>();
        for (ModifiableContent content : contents)
        {
            // There is more than one SCC, only remove the SCC reference
            if (_sccHelper.getSynchronizableCollectionIds(content).size() > 1)
            {
                _sccHelper.removeSCCProperty(content, getId());
                content.saveChanges();
            }
            // It was the only SCC, completely remove the content
            else
            {
                contentsToReallyDelete.add(content);
            }
        }

        logger.info("Remove contents with single SCC... {} contents remaining.", contentsToReallyDelete.size());
        int nbDeletedContents = _deleteContents(contentsToReallyDelete, logger);
        logger.info("Contents deleting process ended. {} contents has been deleted.", nbDeletedContents);
        
        return nbDeletedContents;
    }
    
    /**
     * Delete contents.
     * @param contentsToRemove List of contents to remove
     * @param logger The logger
     * @return the number of deleted contents
     */
    protected int _deleteContents(List<Content> contentsToRemove, Logger logger)
    {
        return _contentDAO.forceDeleteContentsWithLog(contentsToRemove, null, logger);
    }
    
    /**
     * Logs the result of the synchronization, containing
     * <ul>
     * <li>The number of created contents</li>
     * <li>The number of synchronized contents</li>
     * <li>The number of unchanged contents</li>
     * <li>The number of deleted contents</li>
     * </ul>
     * @param logger the logger
     */
    protected void _logSynchronizationResult(Logger logger)
    {
        logger.info("{} contents were created", _nbCreatedContents);
        logger.info("{} contents were updated", _nbSynchronizedContents);
        logger.info("{} contents did not changed", _nbNotChangedContents);
        logger.info("{} contents were deleted", _nbDeletedContents);
    }
    
    /**
     * Checks if some content have changed during the synchronization
     * @return <code>true</code> if some contents have changed, <code>false</code> otherwise
     */
    protected boolean _hasSomethingChanged()
    {
        return _nbCreatedContents > 0 || _nbSynchronizedContents > 0 || _nbDeletedContents > 0;
    }
    
    /**
     * Sends the report mails
     * @param nbError The number of error
     * @throws MessagingException if a messaging error occurred
     * @throws IOException if an error occurred building the message
     */
    protected void sendErrorMail(int nbError) throws MessagingException, IOException
    {
        String language = _userLanguagesManager.getDefaultLanguage();
        
        String pluginName = "plugin.contentio";
        List<String> params = new ArrayList<>();
        params.add(getId());
        String subject = _i18nUtils.translate(new I18nizableText(pluginName, "PLUGINS_CONTENTIO_POPULATE_REPORT_MAIL_SUBJECT", params), language);
        
        params.clear();
        params.add(String.valueOf(nbError));
        params.add(getId());
        String baseUrl = HttpUtils.sanitize(Config.getInstance().getValue("cms.url"));
        params.add(baseUrl + "/_admin/index.html?uitool=uitool-admin-logs");
        String body = _i18nUtils.translate(new I18nizableText(pluginName, "PLUGINS_CONTENTIO_POPULATE_REPORT_MAIL_BODY", params), language);
        
        SendMailHelper.newMail()
                      .withSubject(subject)
                      .withTextBody(body)
                      .withRecipients(Arrays.asList(getReportMails().split("\\n")))
                      .sendMail();
    }
    
    /**
     * Validates a content after import
     * @param content The content to validate
     * @param validationActionId Validation action ID to use for this content
     * @param logger The logger
     */
    protected void validateContent(WorkflowAwareContent content, int validationActionId, Logger logger)
    {
        Map<String, Object> inputs = new HashMap<>();
        
        try
        {
            _contentWorkflowHelper.doAction(content, validationActionId, inputs);
            logger.info("The content {} has been validated after import", content);
        }
        catch (WorkflowException | InvalidActionException e)
        {
            String failuresAsString = _getActionFailuresAsString(inputs);
            logger.error("The content {} cannot be validated after import{}", content, failuresAsString, e);
        }
    }
    
    private String _getActionFailuresAsString(Map<String, Object> actionInputs)
    {
        String failuresAsString = "";
        if (actionInputs.containsKey(AbstractWorkflowComponent.FAIL_CONDITIONS_KEY))
        {
            @SuppressWarnings("unchecked")
            List<ConditionFailure> failures = (List<ConditionFailure>) actionInputs.get(AbstractWorkflowComponent.FAIL_CONDITIONS_KEY);
            if (!failures.isEmpty())
            {
                failuresAsString = ", due to the following error(s):\n" + String.join("\n", failures.stream().map(ConditionFailure::text).toList());
            }
        }
        
        return failuresAsString;
    }
    
    /**
     * Creates content action with result from request
     * @param contentType Type of the content to create
     * @param workflowName Workflow to use for this content
     * @param initialActionId Action ID for initialization
     * @param lang The language
     * @param contentTitle The content title
     * @param logger The logger
     * @return The content id, or null of a workflow error occured
     */
    protected ModifiableContent createContentAction(String contentType, String workflowName, int initialActionId, String lang, String contentTitle, Logger logger)
    {
        try
        {
            logger.info("Creating content '{}' with the content type '{}' for language {}", contentTitle, getContentType(), lang);
            String desiredContentName = _contentPrefix + "-" + contentTitle + "-" + lang;
            
            Map<String, Object> inputs = _getAdditionalInputsForContentCreation();
            Map<String, Object> result = _contentWorkflowHelper.createContent(
                    workflowName,
                    initialActionId,
                    desiredContentName,
                    contentTitle,
                    new String[] {contentType},
                    null,
                    lang,
                    inputs);
            
            return (ModifiableContent) result.get(Content.class.getName());
        }
        catch (WorkflowException e)
        {
            _nbError++;
            logger.error("Failed to initialize workflow for content {} and language {}", contentTitle, lang, e);
            return null;
        }
    }
    
    /**
     * Retrieves additional inputs for content creation
     * @return the additional inputs for content creation
     */
    protected Map<String, Object> _getAdditionalInputsForContentCreation()
    {
        // no additional inputs by default
        return new HashMap<>();
    }

    /**
     * Construct the query to retrieve the content.
     * @param lang Lang
     * @param idValue Synchronization value
     * @param contentType Content type
     * @param forceStrictCheck <code>true</code> to force strict mode to check the collection, otherwise it read the "checkCollection" option
     * @return The {@link List} of {@link Expression}
     */
    protected List<Expression> _getExpressionsList(String lang, String idValue, String contentType, boolean forceStrictCheck)
    {
        List<Expression> expList = new ArrayList<>();
        
        if (forceStrictCheck)
        {
            expList.add(_sccHelper.getCollectionExpression(getId()));
        }
        else if (checkCollection())
        {
            expList.add(
                getCompatibleSCC(true).stream()
                    .map(_sccHelper::getCollectionExpression)
                    .collect(Collectors.toCollection(OrExpression::new))
            );
        }
        
        if (StringUtils.isNotBlank(contentType))
        {
            expList.add(new ContentTypeExpression(Operator.EQ, contentType));
        }
        
        if (StringUtils.isNotBlank(idValue))
        {
            expList.add(new StringExpression(getIdField(), Operator.EQ, idValue));
        }
        
        if (StringUtils.isNotBlank(lang))
        {
            expList.add(new LanguageExpression(Operator.EQ, lang));
        }
        
        return expList;
    }
    
    /**
     * Construct the query to retrieve the content.
     * @param lang Lang
     * @param idValue Synchronization value
     * @param contentType Content type
     * @param forceStrictCheck <code>true</code> to force strict mode to check the collection, otherwise it read the "checkCollection" option
     * @return The XPATH query
     */
    protected String _getContentPathQuery(String lang, String idValue, String contentType, boolean forceStrictCheck)
    {
        List<Expression> expList = _getExpressionsList(lang, idValue, contentType, forceStrictCheck);
        AndExpression andExp = new AndExpression(expList);
        return ContentQueryHelper.getContentXPathQuery(andExp);
    }
    
    /**
     * Remove empty parameters to the map
     * @param searchParameters the parameters
     * @return the map of none empty parameters
     */
    protected Map<String, Object> _removeEmptyParameters(Map<String, Object> searchParameters)
    {
        Map<String, Object> result = new HashMap<>();
        for (String parameterName : searchParameters.keySet())
        {
            Object parameterValue = searchParameters.get(parameterName);
            if (_isParamNotEmpty(parameterValue))
            {
                result.put(parameterName, parameterValue);
            }
        }
        
        return result;
    }
    
    /**
     * Check if the parameter value is empty
     * @param parameterValue the parameter value
     * @return true if the parameter value is empty
     */
    protected boolean _isParamNotEmpty(Object parameterValue)
    {
        return parameterValue != null && !(parameterValue instanceof String && StringUtils.isBlank((String) parameterValue));
    }
    
    public Map<String, Integer> getSynchronizationResult()
    {
        Map<String, Integer> result = new HashMap<>();
        
        result.put(RESULT_NB_CREATED_CONTENTS, _nbCreatedContents);
        result.put(RESULT_NB_SYNCHRONIZED_CONTENTS, _nbSynchronizedContents);
        result.put(RESULT_NB_NOT_CHANGED_CONTENTS, _nbNotChangedContents);
        result.put(RESULT_NB_DELETED_CONTENTS, _nbDeletedContents);
        
        return result;
    }
}
