/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.synchronize;

import java.util.List;
import java.util.Map;
import java.util.Set;

import org.slf4j.Logger;

import org.ametys.cms.data.ContentSynchronizationResult;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.core.right.RightAssignmentContext;
import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.plugins.contentio.synchronize.search.SCCSearchModelConfiguration;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * This interface represents a synchronizable collection of contents
 *
 */
public interface SynchronizableContentsCollection
{
    /** The name for the metadata indicating the ids of the collections */
    public static final String COLLECTION_ID_DATA_NAME = "scc";
    /** The name of the metadata indicating the date of the last synchronization */
    public static final String LAST_SYNCHRONIZATION_DATA_NAME = "lastSynchronization";
    /** The name of the metadata indicating the user that launched the last synchronization */
    public static final String LAST_SYNCHRONIZATION_USER_DATA_NAME = "lastSynchronizationUser";
    
    /** The result map key for number of created contents */
    public static final String RESULT_NB_CREATED_CONTENTS = "nbCreatedContents";
    /** The result map key for number of synchronized contents */
    public static final String RESULT_NB_SYNCHRONIZED_CONTENTS = "nbSynchronizedContents";
    /** The result map key for number of synchronized contents */
    public static final String RESULT_NB_NOT_CHANGED_CONTENTS = "nbNotChangedContents";
    /** The result map key for number of deleted contents */
    public static final String RESULT_NB_DELETED_CONTENTS = "nbDeletedContents";
    
    /**
     * Get the id of synchronizable collection.
     * @return the id
     */
    String getId();
    
    /**
     * Get the label of synchronizable collection
     * @return the label
     */
    I18nizableText getLabel();
    
    /**
     * Get the type of content handled by this collection
     * @return the type of content
     */
    String getContentType();
    
    /**
     * Get the languages handled by this collection
     * @return the list of language
     */
    List<String> getLanguages();
    
    /**
     * Get the id of controller responsible of synchronization of this collection
     * @return The id of controller
     */
    String getSynchronizeCollectionModelId();
    
    /**
     * Get the untyped values of parameters for controller
     * @return the untyped values
     */
    Map<String, Object> getParameterValues();
    
    /**
     * When returns <code>true</code>, a content created by a previous synchro will be removed if it does not exist anymore during the current synchro.
     * @return <code>true</code> if a content created by a previous synchro has to be removed if it does not exist anymore during the current synchro.
     */
    boolean removalSync();
    
    /**
     * When returns <code>true</code>, the process ignore restrictions on attributes to synchronize it.
     * @return <code>true</code> to ignore restrictions on synchronization
     */
    boolean ignoreRestrictions();

    /**
     * When returns <code>true</code>, the process check the collection of contents to synchronize, can be a list of collections and not only the current one.
     * @return <code>true</code> to check the collection on synchronization
     */
    boolean checkCollection();

    /**
     * Get the compatible SCC with the current SCC.
     * @param includeCurrent to include the current SCC in the list.
     * @return The {@link List} of SCC identifiers compatible with the current SCC.
     */
    List<String> getCompatibleSCC(boolean includeCurrent);
    
    /**
     * Get the name of the workflow to use for the synchronized contents
     * @return the name of the workflow to use for the synchronized contents
     */
    String getWorkflowName();
    
    /**
     * Get the id of the initial action of the workflow
     * @return The id of the initial action of the workflow
     */
    int getInitialActionId();
    
    /**
     * Get the id of the synchronize action of the workflow
     * @return The id of the synchronize action of the workflow
     */
    int getSynchronizeActionId();
    
    /**
     * Get the id of the validate action of the workflow
     * @return The id of the validate action of the workflow
     */
    int getValidateActionId();
    
    /**
     * Get the prefix to use for the creation of contents
     * @return The prefix to use for the creation of contents
     */
    String getContentPrefix();
    
    /**
     * True to validate the contents after import
     * @return True to validate the contents after import
     */
    boolean validateAfterImport();
    
    /**
     * If an exception occurs during synchronization, an error report mail will be sent to those email addresses (separated by new lines)
     * @return The email addresses to send an error report if an exception occurs during synchronization (separated by new lines)
     */
    String getReportMails();
    
    /**
     * Gets the id of the {@link SynchronizingContentOperator} extension to use during synchronization
     * @return the id of the {@link SynchronizingContentOperator} extension to use during synchronization
     */
    String getSynchronizingContentOperator();
    
    /**
     * Get the path of boolean metadata for restricted content.
     * If true, the content will be visible only for connected users.
     * @return the path to the metadata. Can be null.
     */
    String getRestrictedField();
    
    /**
     * Get the path of metadata holding the unique identifier
     * @return the path to the metadata. Can be null.
     */
    String getIdField();
    
    /**
     * Get the path of tri-state fields (with local and external values)
     * @param additionalParameters Additional parameters
     * @return the synchronized fields
     */
    Set<String> getLocalAndExternalFields(Map<String, Object> additionalParameters);
    
    /**
     * Get the search UI model for search tool (columns, criterias, url for buttons, etc.)
     * @return A {@link SCCSearchModelConfiguration}
     */
    SCCSearchModelConfiguration getSearchModelConfiguration();
    
    /**
     * True to synchronize only existing contents
     * @return True to synchronize only existing contents
     */
    boolean synchronizeExistingContentsOnly();
    
    /**
     * Empty the collection of its synchronized contents
     * @param logger The logger
     */
    public void empty(Logger logger);
    
    /**
     * Populates contents
     * @param logger The logger
     * @param progressionTracker A progression tracker
     * @return Return the populated contents (imported or synchronized)
     */
    public List<ModifiableContent> populate(Logger logger, ContainerProgressionTracker progressionTracker);
    
    /**
     * Import a content from remote values.
     * @param idValue Id (for import/synchronization) of the content to import
     * @param additionalParameters Additional parameters
     * @param logger The logger
     * @return A list of created contents
     * @throws Exception if an error occurs.
     */
    public List<ModifiableContent> importContent(String idValue, Map<String, Object> additionalParameters, Logger logger) throws Exception;
    
    /**
     * Synchronize a content with remove values.
     * @param content The content to synchronize
     * @param logger The logger
     * @throws Exception if an error occurs.
     */
    public void synchronizeContent(ModifiableContent content, Logger logger) throws Exception;
    
    /**
     * Add specific fields to the content.
     * @param content Content to update
     * @param additionalParameters Additional parameters
     * @param logger The logger
     * @return The synchronization result of the import additional operations
     */
    public default ContentSynchronizationResult additionalImportOperations(ModifiableContent content, Map<String, Object> additionalParameters, Logger logger)
    {
        return additionalCommonOperations(content, additionalParameters, logger);
    }

    /**
     * Add specific fields to the content.
     * @param content Content to update
     * @param additionalParameters Additional parameters
     * @param logger The logger
     * @return The synchronization result of the synchronize additional operations
     */
    public default ContentSynchronizationResult additionalSynchronizeOperations(ModifiableContent content, Map<String, Object> additionalParameters, Logger logger)
    {
        return additionalCommonOperations(content, additionalParameters, logger);
    }
    
    /**
     * Add specific fields to the content during import or synchronization.
     * @param content Content to update
     * @param additionalParameters Additional parameters
     * @param logger The logger
     * @return The synchronization result of the additional operations
     */
    public default ContentSynchronizationResult additionalCommonOperations(ModifiableContent content, Map<String, Object> additionalParameters, Logger logger)
    {
        // Do nothing by default
        return new ContentSynchronizationResult();
    }
    
    /**
     * Gets the content in the repository
     * @param lang the language
     * @param idValue the content name
     * @param forceStrictCheck <code>true</code> to force strict mode to search for the content only in the current collection, otherwise it read the "checkCollection" option
     * @return the content in the repository, or null if does not exist
     */
    public ModifiableContent getContent(String lang, String idValue, boolean forceStrictCheck);
    
    /**
     * Search the data to import from parameters.
     * @param searchParameters Parameters for the search
     * @param offset Begin of the search
     * @param limit Number of results
     * @param sort Sort of results (ignored for LDAP results)
     * @param logger The logger
     * @return A map of remote values by content
     */
    public Map<String, Map<String, Object>> search(Map<String, Object> searchParameters, int offset, int limit, List<Object> sort, Logger logger);

    /**
     * Method to update the synchronisation informations (collection and value of the ID field).
     * @param content Content to update
     * @param syncCode New synchronization code
     * @param logger The logger
     * @throws Exception if an error occurs.
     */
    public void updateSyncInformations(ModifiableContent content, String syncCode, Logger logger) throws Exception;
    
    /**
     * Return the total number of results for the search.
     * @param searchParameters Parameters for the search
     * @param logger The logger
     * @return The total count
     */
    public int getTotalCount(Map<String, Object> searchParameters, Logger logger);
    
    /**
     * Retrieves the result of the synchronization as a map of key / content count containing
     * <ul>
     * <li>The number of created contents</li>
     * <li>The number of synchronized contents</li>
     * <li>The number of unchanged contents</li>
     * <li>The number of deleted contents</li>
     * </ul>
     * @return the result of the synchronization
     */
    public Map<String, Integer> getSynchronizationResult();
    
    /**
     * Return true if a {@link RightAssignmentContext} should be automatically generated for the contents of this SCC
     * @return true to automatically generate a {@link RightAssignmentContext}. Return false if the rights of contents of this SCC are handle by its own {@link RightAssignmentContext}
     */
    public default boolean handleRightAssignmentContext()
    {
        return true;
    }
}
