/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.synchronize.impl;

import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.Map;

import org.apache.avalon.framework.activity.Disposable;
import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollection;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollectionModel;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollectionModelParameterParser;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.Enumerator;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.disableconditions.DisableConditions;
import org.ametys.runtime.model.type.ModelItemTypeExtensionPoint;
import org.ametys.runtime.parameter.Validator;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.runtime.plugin.component.PluginAware;
import org.ametys.runtime.plugin.component.ThreadSafeComponentManager;

/**
 * Implementation of {@link SynchronizableContentsCollectionModel} able to populate contents from a LDAP source
 *
 */
public class DefaultSynchronizableContentsCollectionModel extends AbstractLogEnabled implements SynchronizableContentsCollectionModel, Serviceable, Contextualizable, PluginAware, Configurable, Disposable
{
    private ModelItemTypeExtensionPoint _sccParameterTypeExtensionPoint;
    private ServiceManager _manager;
    private Context _context;
    
    // ComponentManager for disable conditions
    private ThreadSafeComponentManager<DisableConditions> _disableConditionsManager;
    
    // ComponentManager for validators
    private ThreadSafeComponentManager<Validator> _validatorManager;
    
    // ComponentManager for enumerators
    private ThreadSafeComponentManager<Enumerator> _enumeratorManager;
    
    private String _id;
    private Class<SynchronizableContentsCollection> _syncCollectionClass;
    private I18nizableText _label;
    private I18nizableText _description;
    private Map<String, ModelItem> _parameters = new LinkedHashMap<>();
    private String _pluginName;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _manager = manager;
        _sccParameterTypeExtensionPoint = (ModelItemTypeExtensionPoint) manager.lookup(ModelItemTypeExtensionPoint.ROLE_CONFIG);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
        _id = id;
    }
    
    @SuppressWarnings("unchecked")
    public void configure(Configuration configuration) throws ConfigurationException
    {
        _label = I18nizableText.parseI18nizableText(configuration.getChild("label"), "plugin." + _pluginName);
        _description = I18nizableText.parseI18nizableText(configuration.getChild("description"), "plugin." + _pluginName);
        
        String className = null;
        try
        {
            className = configuration.getChild("class").getAttribute("name");
            _syncCollectionClass = (Class<SynchronizableContentsCollection>) Class.forName(className);
        }
        catch (ClassNotFoundException | ConfigurationException e)
        {
            throw new ConfigurationException("SynchronizableContentsCollection model with id '" + _id + "' has an invalid configuration for class name " + (className != null ? className + " <class not found>" : "<missing tag <class>") + "'", e);
        }
        
        if (!SynchronizableContentsCollection.class.isAssignableFrom(_syncCollectionClass))
        {
            throw new ConfigurationException("SynchronizableContentsCollection model with id '" + _id + "' has an invalid configuration: '" + className + "' is not an instance of SynchronizableContentsCollection");
        }
        
        configureParameters(configuration);
    }
    
    /**
     * Configure the SCC parameters
     * @param configuration the model configuration
     * @throws ConfigurationException if a configuration exception occurred
     */
    protected void configureParameters(Configuration configuration) throws ConfigurationException
    {
        _disableConditionsManager = new ThreadSafeComponentManager<>();
        _disableConditionsManager.setLogger(getLogger());
        _disableConditionsManager.contextualize(_context);
        _disableConditionsManager.service(_manager);
        
        _validatorManager = new ThreadSafeComponentManager<>();
        _validatorManager.setLogger(getLogger());
        _validatorManager.contextualize(_context);
        _validatorManager.service(_manager);
        
        _enumeratorManager = new ThreadSafeComponentManager<>();
        _enumeratorManager.setLogger(getLogger());
        _enumeratorManager.contextualize(_context);
        _enumeratorManager.service(_manager);
        
        SynchronizableContentsCollectionModelParameterParser sccParametersParser = new SynchronizableContentsCollectionModelParameterParser(_sccParameterTypeExtensionPoint, _disableConditionsManager, _enumeratorManager, _validatorManager);
        
        Configuration parametersConfiguration = configuration.getChild("parameters");
        Configuration[] paramsConfiguration = parametersConfiguration.getChildren("param");
        for (Configuration paramConfiguration : paramsConfiguration)
        {
            ModelItem parameter = sccParametersParser.parse(_manager, _pluginName, "plugin." + _pluginName, paramConfiguration, this, null);
            String parameterPath = parameter.getPath();
            
            if (_parameters.containsKey(parameterPath))
            {
                throw new ConfigurationException("The parameter '" + parameterPath + "' is already declared. IDs must be unique.", paramConfiguration);
            }
            
            _parameters.put(parameterPath, parameter);
        }
        
        try
        {
            sccParametersParser.lookupComponents();
        }
        catch (Exception e)
        {
            throw new ConfigurationException("Unable to lookup parameter local components", configuration, e);
        }
    }
    
    @Override
    public void dispose()
    {
        _disableConditionsManager.dispose();
        _disableConditionsManager = null;
        
        _validatorManager.dispose();
        _validatorManager = null;
        
        _enumeratorManager.dispose();
        _enumeratorManager = null;
    }
    
    @Override
    public String getId()
    {
        return _id;
    }

    @Override
    public I18nizableText getLabel()
    {
        return _label;
    }

    @Override
    public I18nizableText getDescription()
    {
        return _description;
    }

    @Override
    public Collection<ModelItem> getModelItems()
    {
        return _parameters.values();
    }
    
    @Override
    public String getPluginName()
    {
        return _pluginName;
    }

    @Override
    public Class<SynchronizableContentsCollection> getSynchronizableCollectionClass()
    {
        return _syncCollectionClass;
    }
}
