/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.synchronize.impl;

import java.time.LocalDate;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.temporal.ChronoField;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.slf4j.Logger;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.core.util.ldap.AbstractLDAPConnector;
import org.ametys.core.util.ldap.IncompleteLDAPResultException;
import org.ametys.core.util.ldap.ScopeEnumerator;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollection;
import org.ametys.runtime.model.type.ModelItemTypeConstants;

/**
 * Helper component for {@link SynchronizableContentsCollection}s which need to access a LDAP
 */
public class LDAPCollectionHelper extends AbstractLDAPConnector implements Component
{
    /** Avalon Role */
    public static final String ROLE = LDAPCollectionHelper.class.getName();
    
    private static final DateTimeFormatter __BASIC_DATE_TIME = new DateTimeFormatterBuilder()
            .append(DateTimeFormatter.BASIC_ISO_DATE)
            .appendValue(ChronoField.HOUR_OF_DAY, 2)
            .appendValue(ChronoField.MINUTE_OF_HOUR, 2)
            .appendValue(ChronoField.SECOND_OF_MINUTE, 2)
            .appendFraction(ChronoField.MILLI_OF_SECOND, 1, 3, true)
            .appendOffsetId()
            .toFormatter();

    /** The content type extension point */
    protected ContentTypeExtensionPoint _contentTypeEP;
        
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _contentTypeEP = (ContentTypeExtensionPoint) serviceManager.lookup(ContentTypeExtensionPoint.ROLE); 
        super.service(serviceManager);
    }
        
    /**
     * Search over the LDAP the data from the filter.
     * @param collectionId The id of the collection being synchronized
     * @param relativeDN the name of the context or object to search
     * @param filter the filter expression to use for the search
     * @param searchScope The search scope
     * @param offset Begin of the search
     * @param limit Number of results
     * @param mapping The mapping for retrieving the remote values (keys are metadata paths)
     * @param idKey The key where to search the id value of the content
     * @param logger The logger
     * @param dataSourceId the datasource id
     * @return A map containing the content ids (keys) to import with their remote values (key is attribute, value is the remote value).
     */
    public synchronized LDAPCollectionHelperSearchResult search(String collectionId, String relativeDN, String filter, String searchScope, int offset, int limit, Map<String, List<String>> mapping, String idKey, Logger logger, String dataSourceId)
    {
        Map<String, Map<String, Object>> searchResults = new LinkedHashMap<>();
        int nbErrors = 0;
        boolean hasGlobalError = false;
        
        try
        {
            _delayedInitialize(dataSourceId);
            List<SearchResult> ldapSearchResults;
            try
            {
                ldapSearchResults = _search(relativeDN, filter, null, _getSearchControls(mapping, searchScope), offset, limit, false);
            }
            catch (IncompleteLDAPResultException e)
            {
                ldapSearchResults = e.getPartialResults();
                logger.warn("LDAP refused to return more than " + ldapSearchResults.size() + " results");
                nbErrors++;
            }
            
            for (SearchResult searchResult : ldapSearchResults)
            {
                String idValue = (String) _getIdValue(idKey, searchResult, logger);
                if (idValue == null)
                {
                    nbErrors++;
                    logger.warn("The id value '{}' for '{}' was null ", idKey, searchResult.getName());
                }
                else if (!searchResults.keySet().contains(idValue))
                {
                    try
                    {
                        Map<String, Object> values = new HashMap<>();
                        NamingEnumeration<? extends Attribute> attributes = searchResult.getAttributes().getAll();
                        while (attributes.hasMore())
                        {
                            Attribute attribute = attributes.next();
                            values.put(attribute.getID(), _getLDAPValues(attribute));
                        }
                        searchResults.put(idValue, values);
                    }
                    catch (Exception e)
                    {
                        nbErrors++;
                        logger.warn("Failed to import the content '{}'", idValue, e);
                    }
                }
                else
                {
                    logger.warn("Cannot import '{}' because its id value '{}={}' is already an id value for another content", searchResult.getName(), idKey, idValue);
                }
            }
        }
        catch (Exception e)
        {
            hasGlobalError = true;
            nbErrors++;
            logger.error("Failed to populate contents from synchronizable collection of id '{}'", collectionId, e);
        }
        
        return new LDAPCollectionHelperSearchResult(searchResults, hasGlobalError, nbErrors);
    }

    /**
     * Results of {@link LDAPCollectionHelper#search(String, String, String, String, int, int, Map, String, Logger, String)}
     * @param searchResults the search result
     * @param hasGlobalError the number of errors which occurred during search
     * @param nbErrors true if the a global error occurred during search
     */
    public record LDAPCollectionHelperSearchResult(Map<String, Map<String, Object>> searchResults, boolean hasGlobalError, int nbErrors) { /* empty */ }
    
    private List<Object> _getLDAPValues(Attribute attribute) throws NamingException
    {
        List<Object> ldapValues = new ArrayList<>();
        
        NamingEnumeration< ? > values = attribute.getAll();
        while (values.hasMore())
        {
            ldapValues.add(values.next());
        }
        
        return ldapValues;
    }
    
    /**
     * Get the LDAP search controls.
     * @param mapping The mapping
     * @param searchScope The search scope
     * @return the search controls.
     * @throws ProcessingException if the scope is not valid
     */
    protected SearchControls _getSearchControls(Map<String, List<String>> mapping, String searchScope) throws ProcessingException
    {
        SearchControls controls = new SearchControls();
        
        List<String> attributes = new ArrayList<>();
        for (List<String> attribute : mapping.values())
        {
            attributes.addAll(attribute);
        }
        String[] attrArray = attributes.toArray(new String[attributes.size()]);
        
        controls.setReturningAttributes(attrArray);
        
        controls.setSearchScope(_getScope(searchScope));
        
        return controls;
    }
    
    /**
     * Get the scope as an integer (handlable by the SearchControls) from the scope string.
     * @param scopeStr the scope string.
     * @return the scope as an integer.
     * @throws ProcessingException if the given scope is not valid
     */
    protected int _getScope(String scopeStr) throws ProcessingException
    {
        try
        {
            return ScopeEnumerator.parseScope(scopeStr);
        }
        catch (IllegalArgumentException e)
        {
            throw new ProcessingException("Unable to parse scope", e);
        }
    }
    
    /**
     * Gets id value from a ldap entry
     * @param idKey The key where to search the id value
     * @param entry The ldap entry
     * @param logger The logger
     * @return The attribute value
     * @throws NamingException if a ldap query error occurred
     */
    protected Object _getIdValue(String idKey, SearchResult entry, Logger logger) throws NamingException
    {
        Attribute ldapAttr = entry.getAttributes().get(idKey);
        
        if (ldapAttr == null)
        {
            logger.warn("LDAP attribute not found: '{}'", idKey);
        }
        return ldapAttr != null ? ldapAttr.get() : null;
    }
    
    /**
     * Transform date and datetime attributes on each result line from timestamp to LocalDate (date) or ZonedDateTime (datetime)
     * @param results The results from LDAP source
     * @param contentTypeId Content type ID from which attributes come from
     * @param allAttributes All mapped attributes
     */
    public void transformTypedAttributes(Map<String, Map<String, List<Object>>> results, String contentTypeId, Set<String> allAttributes)
    {
        // Define date and datetime attributes
        Set<String> dateAttributes = new HashSet<>();
        Set<String> datetimeAttributes = new HashSet<>();
        
        ContentType contentType = _contentTypeEP.getExtension(contentTypeId);
        for (String attributePath : allAttributes)
        {
            // Some synchronized attributes are not necessarily defined in the model
            if (contentType.hasModelItem(attributePath))
            {
                String attributeType = contentType.getModelItem(attributePath).getType().getId();
                switch (attributeType)
                {
                    case ModelItemTypeConstants.DATE_TYPE_ID:
                        dateAttributes.add(attributePath);
                        break;
                    case ModelItemTypeConstants.DATETIME_TYPE_ID:
                        datetimeAttributes.add(attributePath);
                        break;
                    default:
                        // Nothing to do
                        break;
                }
            }
        }
        
        // Transform values to typed values
        if (!dateAttributes.isEmpty() || !datetimeAttributes.isEmpty())
        {
            for (Map<String, List<Object>> resultLine : results.values())
            {
                _transformValuesAsTypedValues(resultLine, dateAttributes, s -> LocalDate.parse(s, __BASIC_DATE_TIME));
                _transformValuesAsTypedValues(resultLine, datetimeAttributes, s -> ZonedDateTime.parse(s, __BASIC_DATE_TIME));
            }
        }
    }
    
    private <R> void _transformValuesAsTypedValues(Map<String, List<Object>> resultLine, Set<String> attributeNames, Function<String, R> typedFunction)
    {
        for (String attributeName : attributeNames)
        {
            _transformValueAsTypedValue(resultLine, attributeName, typedFunction);
        }
    }
    
    private <R> void _transformValueAsTypedValue(Map<String, List<Object>> resultLine, String attributeName, Function<String, R> typedFunction)
    {
        List<Object> newValues = Optional.of(attributeName)
                // Get the date attribute for the current result line
                .map(resultLine::get)
                // Stream the list
                .map(List::stream)
                .orElseGet(Stream::empty)
                // Transform each element of the list to a String
                .map(Object::toString)
                // Transform it to LocalDate
                .map(typedFunction)
                // Collect
                .collect(Collectors.toList());

        // If there are values, update the line
        if (!newValues.isEmpty())
        {
            resultLine.put(attributeName, newValues);
        }
    }
}
